/*
 *
 * Copyright 2018-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup;

import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

/**
 * Diese Klasse adaptiert ein altes {@link BackupModul} auf die neue Schnittstelle {@link BackupImplementation}.
 * <p>
 * Da die neue Implementierung on-the-fly die Medien-ID einträgt, das alte Interface aber dateibasiert arbeitet, muss eine temporäre Kopie der Datei angelegt werden.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("deprecation")
public class BackupAdapter implements BackupImplementation {
	private final BackupModul _delegate;

	private final List<File> _tmpFiles = new ArrayList<>();

	private final Path _tmpDir;

	/**
	 * Erstellt eine neue BackupAdapter
	 *
	 * @param delegate Altes Backup-Modul
	 */
	protected BackupAdapter(final BackupModul delegate) throws IOException {
		super();
		_delegate = delegate;
		_tmpDir = Files.createTempDirectory("backup");
	}

	@Override
	public void openMediumBeforeBackup(final int mediumID, final String backupRunID) throws BackupException {
		_delegate.openMediumBeforeBackup(mediumID, backupRunID);
	}

	@Override
	public boolean backupContainer(final int mediumID, final Container containerData) throws BackupException {
		if (!_delegate.hasMediumCapacity(mediumID, containerData.getFileSize(), BackupImplementation.estimateIndexSize(_tmpFiles.size() + 1))) {
			return false;
		}
		try {
			File tmp = _tmpDir.resolve(containerData.getFileName()).toFile();
			containerData.backup(new FileOutputStream(tmp), mediumID);
			_delegate.backupContainer(mediumID, containerData.getContainerId(), tmp);
			_tmpFiles.add(tmp);
		} catch (IOException e) {
			throw new BackupException(e);
		}
		return true;
	}

	@Override
	public void closeMediumAfterBackup(final int mediumID, final File indexFile) throws BackupException {
		_delegate.closeMediumAfterBackup(mediumID, indexFile);
		_tmpFiles.forEach(File::delete);
		_tmpFiles.clear();
	}

	@Override
	public InputStream restoreContainer(final int mediumID, final String contFileName) throws BackupException {
		return _delegate.restoreContainer(mediumID, contFileName);
	}

	@Override
	public List<String> getAllContFileNames(final AbstractTask task, final int mediumID) throws BackupException {
		return _delegate.getAllContFileNames(task, mediumID, -1);
	}
}
