/*
 *
 * Copyright 2018-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup;

import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;
import de.bsvrz.ars.migration.BackupTask;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

/**
 * Eine Subklasse dieses Interfaces ist die Implementierung eines Backup-Moduls (Plugins). Eine Instanz dieser Klasse wird von {@link BackupModule} erzeugt.
 *
 * @author Kappich Systemberatung
 */
public interface BackupImplementation {

	/**
	 * Diese Methode wird aufgerufen, bevor der erste Container auf dem Medium gesichert wird. Mit dem Parameter backupRunID können mehrere Backup-Läufe, die
	 * dasselbe Medium verwenden, unterschieden werden. Laut Spezifikation in {@code [TanfArS]} kann es im Regelfall nicht auftreten, dass ein Container
	 * mehrmals auf demselben Medium abgelegt ist, weil bereits gesicherte Container kein zweites mal gesichert werden. Es ist dennoch notwendig ein Verhalten für
	 * diesen Fall vorzugeben. Beim Wiederherstellen eines Containers wird daher der Container des Mediums verwendet, der unter der alphabetisch größten
	 * backupRunID abgelegt wurde.
	 *
	 * @param mediumID    ID des Mediums
	 * @param backupRunID Zur Identifizierung des Sicherungslaufs falls mehrere Sicherungsläufe auf einem Medium sind. Bei der Wiederherstellung eines Containers
	 *                    werden die Sicherungsläufe von der alphabetisch letzten backupRunID beginnend bis zur alphabetisch ersten backupRunID nach dem
	 *                    angeforderten Container durchsucht.
	 * @throws BackupException Die Implementierung kann bei Schreibfehlern, etc. hiermit Exceptions erzeugen, um das Archivsystem über den Fehlschlag zu informieren.
	 */
	void openMediumBeforeBackup(int mediumID, String backupRunID) throws BackupException;

	/**
	 * Mit dieser Methode wird festgelegt, dass ein Container auf dem angegebenen und vorher geöffneten Medium zu speichern ist. Ob dies sofort geschieht oder
	 * später ist Sache der Implementierung. Das {@code BackupModul} kann den Parameter {@code containerData} verwenden, um z.B. die Container-Dateien zu
	 * sortieren oder um Hilfsdatenstrukturen für einen schnellen Zugriff anzulegen.
	 * <p>
	 * Diese Methode sollte true zurück geben, wenn der Container erfolgreich auf dem Medium ergänzt wurde, oder false, wenn er nicht mehr auf das
	 * Medium passt. In dem Fall wird der {@link BackupTask} das Medium schließen, das nächste Medium mit {@link #openMediumBeforeBackup(int, String)}
	 * öffnen/anlegen und dann diese Methode mit dem gleichen Container erneut aufrufen.
	 *
	 * @param mediumID      Medien-ID, auf die der Container gesichert werden soll. Sofern die Medien-Id in {@link #openMediumBeforeBackup(int, String)} gemerkt wurde
	 *                      oder die Container alle in {@link #closeMediumAfterBackup(int, File)} gesichert werden, kann dieser Parameter möglicherweise
	 *                      ignoriert werden.
	 * @param containerData Klasse, mit der man Infos über den Container abrufen kann, sowie eine {@link Container#backup(OutputStream, int) Methode für die eigentliche Kopier-Aktion}.
	 *                      Das Objekt kann zwischengespeichert werden, bis das Medium in {@link #closeMediumAfterBackup(int, File)} abgeschlossen wird.
	 *                      Danach sind die enthaltenen Informationen möglicherweise ungültig.
	 * @return true wenn der Container noch auf das Medium gepasst hat, sonst false
	 * @throws BackupException Die Implementierung kann bei Schreibfehlern, etc. hiermit Exceptions erzeugen um das Archivsystem über den Fehlschlag zu informieren.
	 */
	boolean backupContainer(int mediumID, Container containerData) throws BackupException;

	/**
	 * Sobald auf einem Medium keine Kapazität für weitere Container mehr vorhanden ist, oder sonst am Ende des Backups, wird der aktuelle Backup-Lauf auf diesem Medium mit dieser Methode
	 * abgeschlossen. Der Parameter übergibt eine Indexdatei, die alle {@code ContainerID}s enthält, die im aktuellen Backup-Lauf auf diesem Medium gesichert
	 * wurden. Diese Datei wird auf dem Medium gespeichert. Die weiteren Aufgaben, die einzelne Implementierungen der Schnittstelle hier zu erfüllen
	 * haben, sind unterschiedlich. Module, die sofort auf ein Medium schreiben können (Wechselplatten) erfordern keine weiteren Schritte. Für Medien wie DVDs wird
	 * in dieser Methode aus den gesammelten Containern ein Image erstellt und auf die DVD gebrannt; bei Keydox wird ein Logout ausgeführt.
	 *
	 * @param mediumID  Medien-ID
	 * @param indexFile Indexdatei, die die ContainerIDs enthält
	 * @throws BackupException Die Implementierung kann bei Schreibfehlern, etc. hiermit Exceptions erzeugen um das Archivsystem über den Fehlschlag zu informieren.
	 */
	void closeMediumAfterBackup(int mediumID, File indexFile) throws BackupException;

	/**
	 * Mit dieser Methode wird ein {@code InputStream} der Containerdaten des Containers mit Namen {@code contFile} auf dem Medium {@code mediumID}
	 * erzeugt. Der BackupManager kopiert die Daten je nach Vorgang ganz oder teilweise in das Archivsystem oder in ein temporäres Verzeichnis. Diese Methode wird
	 * zum Wiederherstellen eines Containers, zum Abgleich/Wiederherstellung der Verwaltungsinformation und zur Verifikation verwendet.
	 *
	 * @param mediumID     Medien-ID
	 * @param contFileName Dateiname des wiederherzustellenden Containers
	 * @return InputStream
	 * @throws BackupException Die Implementierung kann bei Lesefehlern, etc. hiermit Exceptions erzeugen um das Archivsystem über den Fehlschlag zu informieren.
	 */
	InputStream restoreContainer(int mediumID, String contFileName) throws BackupException;

	/**
	 * Liefert die zuletzt gesicherten Container, die sich auf dem Mediums befinden.
	 * <p>
	 * Wird für den Abgleich von Verwaltungsinformationen benoetigt.
	 *
	 * @param task     Wird zum Anhalten des Backups benötigt, falls das Archivsystem in einen kritischen Bereich eintritt.
	 * @param mediumID MedienId des Mediums.
	 * @return Liste mit den Filenames der Container.
	 * @throws BackupException Die Implementierung kann bei Lesefehlern, etc. hiermit Exceptions erzeugen um das Archivsystem über den Fehlschlag zu informieren.
	 */
	@Deprecated
	List<String> getAllContFileNames(AbstractTask task, int mediumID) throws BackupException;

	/**
	 * Schätzt die Indexgröße für eine angegebene Zahl Einträge. Wird gebraucht um festzustellen, ob noch genug Platz für einen Container + Index auf einem Medium
	 * ist
	 *
	 * @param countSavedContainers Anzahl der Container
	 * @return Byte-Anzahl die der Index maximal brauchen wird.
	 */
	static int estimateIndexSize(final int countSavedContainers) {
		return countSavedContainers * (2 + 13 + 4 + 1); // "dc" + zahl + ".dat" + \n == 20
	}
}
