/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.backup;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Properties;

public class BackupInitializer {

	/**
	 * Setzt das zu verwendende Backup-Modul.
	 *
	 * @param propertiesFile Datei in dem das zu verwendende Modul gespeichert ist.
	 * @return Backup-Implementierung oder null, wenn es kein backup-Modul gibt
	 * @throws BackupException Fehler beim Vorgang
	 */
	public static BackupImplementation getBackupModul(final String propertiesFile) throws BackupException {
		// Wenn keine Properties gesetzt sind, ist das egal.
		if (propertiesFile == null || propertiesFile.isEmpty()) {
			return null;
		}

		final Properties properties = new Properties();
		try {
			try (FileInputStream fis = new FileInputStream(new File(propertiesFile))) {
				properties.load(fis);
			}
		} catch (IOException ioe) {
			throw new BackupException("Fehler beim Einlesen der BackupProperties-Datei: ", ioe);
		}

		final String moduleName = properties.getProperty("backupModul");
		if (moduleName == null || moduleName.isEmpty()) {
			throw new BackupException("Die BackupProperties-Datei '" + propertiesFile + "' enthält keinen Eintrag 'backupModul'");
		}

		final Class<?> backupClass;
		try {
			backupClass = Class.forName(moduleName);
		} catch (ClassNotFoundException e) {
			throw new BackupException("Das BackupModul '" + moduleName + "' konnte nicht gefunden werden. ", e);
		}

		Object instance;
		try {
			instance = backupClass.getDeclaredConstructor().newInstance();
		} catch (InstantiationException | IllegalAccessException | NoSuchMethodException |
		         InvocationTargetException e) {
			throw new BackupException("Das BackupModul '" + moduleName + "' konnte nicht instantiiert werden. ", e);
		}

		try {
			return initializeBackupModule(properties, instance);
		} catch (Exception e) {
			throw new BackupException("Das BackupModul '" + moduleName + "' konnte nicht initialisiert werden. ", e);
		}
	}


	/**
	 * Initialisiert das Backup-Modul. Wurde eine Klasse des veralteten Typs {@link BackupModul} (ohne e) angegeben,
	 * wird diese in einen {@link BackupAdapter} verpackt.
	 *
	 * @param properties Properties aus der Datei
	 * @param instance   Instanz der angeforderten Klasse
	 * @return Backup-Modul
	 * @throws Exception Exception
	 */
	@SuppressWarnings("deprecation")
	private static BackupImplementation initializeBackupModule(final Properties properties, final Object instance) throws Exception {
		if (instance instanceof BackupModule backupModule) {
			return backupModule.getInstance(properties);
		} else if (instance instanceof BackupModul backupModul) {
			backupModul.initialize(properties);
			return new BackupAdapter(backupModul);
		} else {
			throw new BackupException("Das BackupModul ist nicht vom korrekten Typ de.bsvrz.ars.ars.backup.BackupModule oder de.bsvrz.ars.ars.backup.BackupModul: " + instance.getClass());
		}
	}
}
