/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup;

import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

/**
 * Diese Klasse dient als Interface für alle Backup-Module.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 * @deprecated Sollte durch {@link BackupModule} ersetzt werden, da eine bessere Performance und ein einfacheres Interface erreicht werden kann.
 */
@Deprecated
public interface BackupModul {

	String PROP_BACKUP_CONTAINER_PER_ZIP = "containerPerZip";

	String PROP_BACKUP_MEDIUMSIZE = "sicherungsMedienGroesseKB";

	/**
	 * Diese Methode dient dem Initialisieren des BackupModuls. Da die Plugins immer mit leerem Konstruktor initialisiert werden, werden mit dieser Methode die
	 * Plugin-spezifischen Einstellungen vorgenommen. Fehlt ein obligatorischer Property-Key, wird eine Exception geworfen.
	 *
	 * @param props
	 */
	void initialize(Properties props);

	/**
	 * Liefert true, falls die noch zur Verfügung stehende Kapazität des angegebenen Mediums ausreicht, um einen Container der Größe {@code fileSize} und den
	 * Medienindex mit der momentanen Größe {@code indexSize} zu speichern. Falls die Gesamtkapazität nicht durch den Medientyp eindeutig festgelegt ist,
	 * liefert das Backup-Modul einen sinnvollen Wert. Beispiel: 1 GB pro Aktenschrank in Keydox, jedoch keinesfalls die Größe der gesamten Festplatte. Ist das
	 * gewünschte Medium nicht im Zugriff, wird eine entsprechende Ausnahme ausgelöst.
	 *
	 * @param mediumID
	 * @param containerSize
	 * @param mediumIndexSize
	 * @return boolean
	 */
	boolean hasMediumCapacity(int mediumID, long containerSize, int mediumIndexSize);

	/**
	 * Diese Methode wird aufgerufen, bevor der erste Container auf dem Medium gesichert wird. Mit dem Parameter backupRunID können mehrere Backup-Läufe, die
	 * dasselbe Medium verwenden, unterschieden werden. Laut Spezifikation in {@code [TanfArS]} kann es im Regelfall nicht auftreten, dass ein Container
	 * mehrmals auf demselben Medium abgelegt ist, weil bereits gesicherte Container kein zweitesmal gesichert werden. Es ist dennoch notwendig ein Verhalten für
	 * diesen Fall vorzugeben. Beim Wiederherstellen eines Containers wird daher der Container des Mediums verwendet, der unter der alphabetisch größten
	 * backupRunID abgelegt wurde.
	 *
	 * @param mediumID    ID des Mediums
	 * @param backupRunID Zur Identifizierung des Sicherungslaufs falls mehrere Sicherungsläufe auf einem Medium sind. Bei der Wiederherstellung eines Containers
	 *                    werden die Sicherungsläufe von der alphabetisch letzten backupRunID beginnend bis zur alphabetisch ersten backupRunID nach dem
	 *                    angeforderten Container durchsucht.
	 */
	void openMediumBeforeBackup(int mediumID, String backupRunID);

	/**
	 * Mit dieser Methode wird festgelegt, dass ein Container auf dem angegebenen und vorher geöffneten Medium zu speichern ist. Ob dies sofort geschieht oder
	 * später ist Sache der Implementierung. Das {@code BackupModul} kann den Parameter {@code containerID} verwenden, um z.B. die Container-Dateien zu
	 * sortieren oder um Hilfsdatenstrukturen für einen schnellen Zugriff anzulegen.
	 *
	 * @param mediumID
	 * @param contID
	 * @param contFile
	 */
	void backupContainer(int mediumID, long contID, File contFile);

	/**
	 * Sobald auf einem Medium keine Kapazität für weitere Container mehr vorhanden ist, wird der aktuelle Backup-Lauf auf diesem Medium mit dieser Methode
	 * abgeschlossen. Der Parameter übergibt eine Indexdatei, die alle {@code ContainerID}s enthält, die im aktuellen Backup-Lauf auf diesem Medium gesichert
	 * wurden. Diese Datei wird auf dem Medium gespeichert. Die weiteren Aufgaben, die einzelne Implementierungen der BackupModule-Schnittstelle hier zu erfüllen
	 * haben, sind unterschiedlich. Module, die sofort auf ein Medium schreiben können (Wechselplatten) erfordern keine weiteren Schritte. Für Medien wie DVDs wird
	 * in dieser Methode aus den gesammelten Containern ein Image erstellt und auf die DVD gebrannt; bei Keydox wird ein Logout ausgeführt.
	 *
	 * @param mediumID
	 * @param indexFile
	 */
	void closeMediumAfterBackup(int mediumID, File indexFile);

	/**
	 * Mit dieser Methode wird ein {@code InputStream} der Containerdaten des Containers mit Namen {@code contFile} auf dem Medium {@code mediumID}
	 * erzeugt. Der BackupManager kopiert die Daten je nach Vorgang ganz oder teilweise in das Archivsystem oder in ein temporäres Verzeichnis. Diese Methode wird
	 * zum Wiederherstellen eines Containers, zum Abgleich/Wiederherstellung der Verwaltungsinformation und zur Verifikation verwendet.
	 *
	 * @param mediumID
	 * @param contFileName
	 * @return InputStream
	 */
	InputStream restoreContainer(int mediumID, String contFileName);

	/**
	 * Liefert die zuletzt gesicherten Container, die sich auf dem Mediums befinden.
	 * <p>
	 * Wird für den Abgleich von Verwaltungsinformationen (@link HeaderRestoreTask) benoetigt.
	 *
	 * @param task     Wird zum Anhalten des Backups benötigt, falls das Archivsystem in einen kritischen Bereich eintritt.
	 * @param mediumID MedienId des Mediums.
	 * @param maxLimit maximale Anzahl an zurückgegebenen Containernamen. Wenn -1, dann ist das Ergebnis unlimitiert.
	 * @return Liste mit den Filenames der Container.
	 */
	List<String> getAllContFileNames(AbstractTask task, int mediumID, int maxLimit);
}
