/*
 *
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup.plugins;

import de.bsvrz.ars.ars.backup.BackupException;
import de.bsvrz.ars.ars.backup.BackupImplementation;
import de.bsvrz.ars.ars.backup.Container;
import de.bsvrz.ars.ars.backup.MediumNotAccessibleException;
import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;
import de.bsvrz.ars.ars.persistence.ContainerFile;
import de.bsvrz.ars.migration.BackupTask;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.*;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Gemeinsame Basisklasse für DVD und Filesystem-Backup.
 *
 * @author Kappich Systemberatung
 */
public abstract class AbstractFileSystemBackup implements BackupImplementation {

	/**
	 * Key der Properties-Einstellung für das Sicherungsverzeichnis
	 */
	public static final String PROP_BACKUP_DIR = "sicherungsVerzeichnis";

	/**
	 * Key der Properties-Einstellung für die Anzahl Container pro Zip-Datei
	 */
	public static final String PROP_BACKUP_CONTAINER_PER_ZIP = "containerPerZip";

	/**
	 * Key der Properties-Einstellun ür die maximale Mediengröße
	 */
	public static final String PROP_BACKUP_MEDIUMSIZE = "sicherungsMedienGroesseKB";

	/**
	 * Aktueller Wert für die Anzahl Container pro Zip-Datei
	 */
	protected int _maxContainersPerZip = 1000;

	/**
	 * Aktueller Wert für die maximale Mediengröße in Kilobytes
	 */
	protected long _maxMediumSizeKb = 4300L * 1024L;

	/**
	 * Debug-Logger
	 */
	protected final Debug logger = Debug.getLogger();

	/**
	 * Busher benutzter Speicherplatz von Zip-Dateien auf dem aktuellen Medium
	 */
	protected long currentSpaceOccupiedByZipfiles;

	/**
	 * Bisher benutzter Speicherplatz der Containerdateien für die aktuelle zip-Datei (noch unkomprimiert)
	 */
	protected long currentSpaceOccupiedByContainers;

	/**
	 * Aktueller Unterordner, in den gesichert werden soll
	 */
	protected String currentBackupPath;

	/**
	 * Backup-Basispfad. Muss von implementierenden Klassen gesetzt werden.
	 */
	protected String backupBasePath;

	/**
	 * Zähler der gesicherten Container zur Realisierung der maximalen Containeranzahl pro Zip-Datei
	 */
	private int currentContIndex;

	/**
	 * Zwischenspeicherung der zu sichernden Container auf dem aktuellen Medium
	 */
	private final List<Container> currentContainersToBeSaved = new ArrayList<>();

	/**
	 * Gibt zurück, ob für einen Container der Größe ContainerSize noch auf dem aktuellen Medium Platz ist
	 *
	 * @param containerSize   Container-Größe in bytes
	 * @param mediumIndexSize Geschätzte Größe des Medium-Indexes (Indexdatei)
	 * @return true: Es ist Platz vorhanden, false: Es ist kein Platz vorhanden
	 */
	public abstract boolean hasMediumCapacity(long containerSize, int mediumIndexSize);

	@Override
	public boolean backupContainer(int mediumID, Container contFile) throws BackupException {
		if (!hasMediumCapacity(contFile.getFileSize(), BackupImplementation.estimateIndexSize(currentContainersToBeSaved.size() + 1))) {
			return false;
		}
		try {
			// bis max-Grenze erreicht ist: File merken
			if (++currentContIndex % _maxContainersPerZip != 0) {
				currentContainersToBeSaved.add(contFile);
				currentSpaceOccupiedByContainers += contFile.getFileSize();
			} // wenn max-Grenze erreicht ist: Zippen, Speichern und neues Verzeichnis bestimmen
			else {
				currentContainersToBeSaved.add(contFile);
				currentSpaceOccupiedByContainers += contFile.getFileSize();
				backupContainerList(mediumID);
			}
			return true;
		} catch (Exception e) {
			throw new BackupException(e);
		}
	}


	@Override
	public void openMediumBeforeBackup(int mediumID, String backupRunID) {
		String tmpMediumPath = backupBasePath + File.separator + "Medium_" + mediumID;
		if (!new File(tmpMediumPath).exists()) {
			new File(tmpMediumPath).mkdir();
		}
		currentBackupPath = tmpMediumPath + File.separator + backupRunID;
		if (!new File(currentBackupPath).exists()) {
			new File(currentBackupPath).mkdir();
		}
	}

	@Override
	public void closeMediumAfterBackup(int mediumID, File indexFile) throws BackupException {
		try {
			if (!currentContainersToBeSaved.isEmpty()) {
				// die letzten Container noch zippen und schreiben:
				backupContainerList(mediumID);
			}

			// indexFile abspeichern
			File newIndexFile = new File(currentBackupPath, indexFile.getName());
			try (FileInputStream is = new FileInputStream(indexFile)) {
				try (FileOutputStream os = new FileOutputStream(newIndexFile)) {
					copyStream(is, os);
				}
			}
		} catch (Exception e) {
			throw new BackupException(e);
		} finally {
			currentContainersToBeSaved.clear();
			currentSpaceOccupiedByContainers = 0;
			currentSpaceOccupiedByZipfiles = 0;
		}
	}

	@Override
	public InputStream restoreContainer(int mediumID, String contFileName) throws BackupException {

		String backupPath = getBackupPath(mediumID);
		logger.fine("Suche nach Container " + contFileName + " im Verzeichnis: " + backupPath);

		long contID = ContainerFile.getContID(contFileName);
		File zipFile = null;
		try {
			List<File> backupRuns = Arrays.asList(Util.listDirectories(backupPath));
			Collections.sort(backupRuns);

			// indexFile auspacken und nachschauen, ob Container enthalten ist
			boolean containerFound = false;
			int i = backupRuns.size() - 1;
			String backupRun = "";
			while (!containerFound && i >= 0) {
				backupRun = backupRuns.get(i--).getName();

				File indexFile = new File(backupPath + File.separator + backupRun, BackupTask.INDEXFILE_NAME);
				if (!indexFile.exists()) {
					logger.info("kein Index-File gefunden: " + indexFile.getAbsolutePath());
					continue;
				}

				try (RandomAccessFile raf = new RandomAccessFile(indexFile, "r")) {
					for (String line; (line = raf.readLine()) != null; ) {
						if (line.contains(contFileName)) {
							containerFound = true;
							break;
						}
					}
				}
			}
			if (!containerFound) {
				logger.error("Container " + contFileName + " konnte nicht auf MediumId=" + mediumID + " gefunden werden.");
				throw new BackupException("Container " + contFileName + " konnte nicht auf MediumId=" + mediumID + " gefunden werden.");
			}

			// nach dem richtigen Zip-File suchen:
			File[] zipfiles = new File(backupPath, backupRun).listFiles(
					pathname -> pathname.getName().endsWith(".zip")
			);

			if (zipfiles != null) {
				for (i = 0; i < zipfiles.length; i++) {
					String name = zipfiles[i].getName();
					String firstContainerNameAbb = name.substring(0, name.indexOf('-'));
					String lastContainerNameAbb = name.substring(name.indexOf('-') + 1, name.length() - 4);
					long firstContID = Long.parseLong(firstContainerNameAbb.substring(2));
					long lastContID = Long.parseLong(lastContainerNameAbb.substring(2));

					if (contID >= firstContID && contID <= lastContID) {
						zipFile = new File(backupPath + File.separator + backupRun, name);
						break;
					}
				}
			}
			if (zipFile == null) {
				logger.warning("Zip-Datei mit ContainerId=" + contID + " konnte nicht auf MediumID=" + mediumID + " gefunden werden.");
				throw new BackupException("Zip-Datei mit ContainerId=" + contID + " konnte nicht gefunden werden.");
			}
		} catch (Exception e) {
			logger.warning("Zip-Datei mit ContainerId=" + contID + " konnte nicht auf MediumID=" + mediumID + " gefunden werden.", e.getMessage());
			throw new BackupException("Zip-Datei mit ContainerId=" + contID + " konnte nicht auf MediumID=" + mediumID + " gefunden werden.", e);
		}
		try {
			return detachFileFromZip(zipFile, contFileName);
		} catch (Exception e) {
			throw new BackupException(e);
		}
	}

	/**
	 * Gibt den Ordnernamen zurück, in den Daten des angegebenen Medium gespeichert werden sollen
	 *
	 * @param mediumID medien-ID
	 * @return Ordnernamen
	 */
	@NotNull
	public abstract String getBackupPath(int mediumID);

	@Override
	public List<String> getAllContFileNames(AbstractTask task, int mediumID) throws BackupException {

		String backupPath = backupBasePath + File.separator + "Medium_" + mediumID;

		File tmpBackupPathFile = new File(backupPath);
		if (!tmpBackupPathFile.exists()) {
			logger.error("BackupPath " + backupPath + " existiert nicht.");
			throw new MediumNotAccessibleException(mediumID);
		}

		List<String> result = new ArrayList<>();
		List<File> backupRuns = Arrays.asList(Util.listDirectories(backupPath));
		Collections.sort(backupRuns);

		for (int i = backupRuns.size() - 1; i >= 0; i--) {
			task.suspendTaskIfNecessary();    //Task anhalten, falls Archivsystem überlastet
			String backupRun = backupRuns.get(i).getName();

			File indexFile = new File(backupPath + File.separator + backupRun, BackupTask.INDEXFILE_NAME);
			if (!indexFile.exists()) {
				continue;
			}
			try (RandomAccessFile raf = new RandomAccessFile(indexFile.getAbsolutePath(), "r")) {
				for (String line; (line = raf.readLine()) != null; ) {
					result.add(line.substring(line.lastIndexOf(File.separator) + 1));
				}
			} catch (Exception e) {
				Debug.getLogger().error("FileSystemBackup: Indexdatei " + indexFile.getAbsolutePath() + " konnte nicht gelesen werden.", e.getMessage());
			}
		}
		return result;
	}

	/**
	 * Entpackt eine ContainerDatei aus einem Zip-File.
	 *
	 * @param zipFile      ZIP-Datei
	 * @param contFileName Dateiname der Containerdatei, die aus dem Zip entpackt werden soll
	 * @return Ein InputStream, der den Containerinhalt enthält
	 * @throws BackupException falls ein Lesefehler auftritt oder die Datei nicht enthalten ist.
	 */
	private static InputStream detachFileFromZip(File zipFile, String contFileName) throws BackupException {
		try {
			ZipInputStream zin = new ZipInputStream(new BufferedInputStream(new FileInputStream(zipFile)));
			ZipEntry zen;
			while ((zen = zin.getNextEntry()) != null) {
				if (zen.getName().equals(contFileName)) {
					return zin;
				}
			}
		} catch (IOException e) {
			throw new BackupException(e);
		}
		throw new BackupException(
				"Fehler beim Wiederherstellen: Datei '" + contFileName + "' konnte nicht aus Zip-Datei '" + zipFile.getAbsolutePath() + "' extrahiert werden."
		);
	}

	/**
	 * Zippt die in der Containerlist aufgelisteten Dateien und legt sie in den {@code currentBackupPath}.
	 *
	 * @param mediumId Medien-ID
	 * @throws Exception Fehler beim Schreiben
	 */
	private void backupContainerList(final int mediumId) throws Exception {

		File backupDir = new File(currentBackupPath);

		currentContainersToBeSaved.sort(Comparator.comparingLong(Container::getContainerId));

		String nameFirst = currentContainersToBeSaved.get(0).getFileName();
		String nameLast = currentContainersToBeSaved.get(currentContainersToBeSaved.size() - 1).getFileName();
		String containerZipName = nameFirst.substring(0, nameFirst.indexOf('.'));
		containerZipName += "-" + nameLast.substring(0, nameLast.indexOf('.'));

		File zipFile = new File(backupDir, containerZipName + ".zip");
		try (ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(zipFile)))) {
			for (Container file : currentContainersToBeSaved) {
				ZipEntry entry = new ZipEntry(file.getFileName());
				entry.setSize(file.getFileSize());
				entry.setTime(file.lastModified());
				out.putNextEntry(entry);
				file.backup(out, mediumId);
				out.closeEntry();
			}
			out.finish();
		}
		currentSpaceOccupiedByZipfiles += zipFile.length();
		currentSpaceOccupiedByContainers = 0;
		currentContainersToBeSaved.clear();
	}

	private static void copyStream(InputStream in, OutputStream out) throws IOException {
		byte[] buffer = new byte[1024 * 8];
		int len;
		while ((len = in.read(buffer)) >= 0) out.write(buffer, 0, len);
	}

	public String toString() {
		return "FileSystem (" + backupBasePath + ")";
	}
}
