/*
 *
 * Copyright 2018-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.migration.
 *
 * de.bsvrz.ars.migration is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.migration is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.migration.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.backup.plugins;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.io.File;
import java.util.Properties;

/**
 * Diese Klasse ist für die Sicherung von Containern auf dem Filesystem verantwortlich.
 *
 * @author beck et al. projects GmbH
 * @author Christian Wied
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class FileSystemBackupImpl extends AbstractFileSystemBackup {

	/**
	 * Erstellt ein neues FileSystemBackupImpl-Objekt
	 *
	 * @param props Backup-Properties (Parameter)
	 */
	public FileSystemBackupImpl(final Properties props) {
		super();

		String zipSize = props.getProperty(PROP_BACKUP_CONTAINER_PER_ZIP);
		try {
			if (zipSize != null) {
				_maxContainersPerZip = Integer.parseInt(zipSize);
			}
		} catch (NumberFormatException ignored) {
		} // do nothing, default 100 is ok.

		String mediumSize = props.getProperty(PROP_BACKUP_MEDIUMSIZE);
		try {
			if (mediumSize != null) {
				_maxMediumSizeKb = Long.parseLong(mediumSize);
			}
		} catch (NumberFormatException ignored) {
		} // do nothing, take default.

		String backupDirName = props.getProperty(PROP_BACKUP_DIR);

		if (backupDirName == null) {
			throw new IllegalArgumentException("Benötigter Parameter '" + PROP_BACKUP_DIR + "' nicht angegeben.");
		}
		File backupDir = new File(backupDirName);
		if (!backupDir.exists()) {
			backupDir.mkdir();
		}
		backupBasePath = backupDir.getAbsolutePath();
	}

	@Override
	public boolean hasMediumCapacity(long containerSize, int mediumIndexSize) {
		return _maxMediumSizeKb * 1024L > currentSpaceOccupiedByZipfiles + currentSpaceOccupiedByContainers + containerSize + mediumIndexSize;
	}

	@Override
	@NotNull
	public String getBackupPath(final int mediumID) {
		return backupBasePath + File.separator + "Medium_" + mediumID;
	}

	public String toString() {
		return "FileSystem (" + backupBasePath + ")";
	}
}
