package de.bsvrz.sys.dcf77.lib;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Utility package ; Datum- und Zeitkonvertierungsfunktionen.
 * @author Chriesten
 *
 */
public class UtlDateTime
{
  
  /**
   * Wandelt ms - zeitwert  in Datum
   * 
   * @param zeitMs Zeit in ms seit 1970 ...
   * @return Datum
   */
  public static Date zeit2date(long zeitMs)
  {
    return (new Date(zeitMs));
  }
  
  /**
   * Wandelt zeitwert ms in String 
   * 
   * @param zeitMs Zeit in ms seit 1970 ...
   * @return Datum als String ("dd.MM.yyyy HH:mm:ss")
   */
  public static String zeit2String(Long zeitMs)
  {
    if ( zeitMs == null )
      return("");
    
    SimpleDateFormat formatter = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");
    String formattedDate = formatter.format(new Date(zeitMs));
    return (formattedDate);
  }

  

  /**
   * liefert das übergebne Datum als String zurück
   *
   * @param date Datum
   * @param dateFormat Format, das zurückgeliefert werden soll
   * @return datumString (dd.MM.yyyy HH:mm:ss)
   */
  public static String date2String(Date date, String dateFormat)
  {
    SimpleDateFormat formatter = new SimpleDateFormat(dateFormat);
    String formattedDate = formatter.format(date);
    return(formattedDate);
  }

  /**
   * liefert das übergebne Datum als String zurück
   *
   * @param date Datum
   * @return datumString (dd.MM.yyyy HH:mm:ss)
   */
  public static String date2String(Date date)
  {
    return date2String(date, "dd.MM.yyyy HH:mm:ss");
  }

  /**
   * liefert die Zeit in ms zum übergebenen Datumstring. Format dd.MM.yyyy HH:mm:ss
   *
   * @param datumsString Datum
   * @return zeitMs (msec seit 1970)
   */
  public static Long string2Zeit(String datumsString)
  {
    return string2Zeit(datumsString, false);
  }


  /**
   * liefert die Zeit in ms zum übergebenen Datumstring. Format dd.MM.yyyy HH:mm:ss
   *
   * @param datumsString Datum
   * @param useLenient true: Lenient mode, nicht auf existierende Datuswerte achten
   * @return zeitMs (msec seit 1970)
   */
  public static Long string2Zeit(String datumsString, boolean useLenient)
  {
    String dateFormat = "dd.MM.yyyy HH:mm:ss";
    return string2Zeit(datumsString, dateFormat, useLenient);
  }

  /**
   * liefert die Zeit in ms zum übergebenen Datums und Formatstring
   *
   * @param datumsString Datum
   * @param dateFormat Datumsformat
   * @param useLenient true: Lenient mode, nicht auf existierende Datuswerte achten
   * @return zeitMs (msec seit 1970)
   */
  public static Long string2Zeit(String datumsString, String dateFormat, boolean useLenient)
  {
    SimpleDateFormat formatter = new SimpleDateFormat(dateFormat);
    Date date = null;

    if ( !useLenient )
    {
      //Weißt den Parser an auf existierende Datumswerte zu achten, ohne die Zeile wird
      // aus 32.01.2003, 01.02.2003
      formatter.setLenient(false);
    }

    try
    {
      date = formatter.parse(datumsString);
    }
    catch (java.text.ParseException e)
    {
      System.err.println("Format ungueltig: "+ datumsString);
      return(null);
    }
    return date.getTime();
  }

  /**
   * Liefert den Tagesanfang 00:00 (in ms) ausgehend von der Übergebenen Zeit (ms)
   * @param zeitMs irgend eine Uhrzeit des zu betrachtenden Tages oder 0:heute, -1 gestern ...
   * @return 00:00 (in ms) des Tages von zeitMs
   */
  public static Long getTagesanfang(Long zeitMs)
  {
    if ( zeitMs < 10 )
    {
      zeitMs = new Date().getTime() + zeitMs * 24*3600*1000;
    }
    // Tagesanfang ermitteln
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis( zeitMs );
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    return(cal.getTimeInMillis());
  }

  /**
   * Liefert den Jahresanfang 1.1 00:00 (in ms) ausgehend von der Übergebenen Zeit (ms)
   * @param zeitMs irgend eine Uhrzeit des zu betrachtenden Tages oder 0:heute, -1 gestern ...
   * @return 00:00 (in ms) des Tages von zeitMs
   */
  public static Long getJahresanfang(Long zeitMs)
  {
    Long tagesanfang = getTagesanfang( zeitMs );
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis( tagesanfang );
    cal.set(Calendar.DAY_OF_MONTH, 1);
    cal.set(Calendar.MONTH, 0);
    return(cal.getTimeInMillis());
  }

  /**
   * Liefert das Tagesende 23:59:59 (in ms) ausgehend von der Übergebenen Zeit (ms)
   * @param zeitMs irgend eine Uhrzeit des zu betrachtenden Tages oder 0:heute, -1 gestern ...
   * @return 23:59:59-999 (in ms) des Tages von zeitMs
   */
  public static Long getTagesende(Long zeitMs)
  {
    if ( zeitMs < 10 )
    {
      zeitMs = new Date().getTime() + zeitMs * 24*3600*1000;
    }
    // Tagesende ermitteln
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis( zeitMs );
    cal.set(Calendar.HOUR_OF_DAY, 23);
    cal.set(Calendar.MINUTE, 59);
    cal.set(Calendar.SECOND, 59);
    cal.set(Calendar.MILLISECOND, 999);
    return(cal.getTimeInMillis());
  }


  /**
   * Liefert Feldwerte
   *
   * @param zeitMs (msec seit 1970)
   * @param field Calendar Field
   * @return Feldwerte
   */
  public static int extractFieldValues(long zeitMs, int field)
  {
    // zeitMs -> Caldendar
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis( zeitMs );
    int val = cal.get( field );
    if ( field==Calendar.MONTH ) // Jan == 0!
      val++;
    return val;
  }

  /**
   * Erzeugt aus den übergebenen Feldern einen time value
   * @param year Jahr 4-stellig
   * @param month Monat (1-12)
   * @param day_of_month Tag
   * @param hour_of_day (1-24)
   * @param minute Minuten
   * @param second Sekunden
   * @return Zeitpunkt
   */
  static public long fields2Zeit(int year, int month, int day_of_month, int hour_of_day, int minute, int second)
  {
    // month is 0 based!
    Calendar cal = Calendar.getInstance();
    cal.set( year, month-1, day_of_month, hour_of_day, minute, second );
    long zeitMs = cal.getTimeInMillis();

    return(zeitMs);
  }


  /**
   * Testprogramm für getTagesanfang und -ende
   */
  public static void testGetTagesAE()
  {
    UtlDateTime.getTagesanfang( 0L );
    String message;
    message = "heute:";
    message += UtlDateTime.zeit2String( UtlDateTime.getTagesanfang( 0L ));
    message += "   " + UtlDateTime.zeit2String( UtlDateTime.getTagesende( 0L ));
    System.out.println(message);

    message = "gestern:";
    message += UtlDateTime.zeit2String( UtlDateTime.getTagesanfang( -1L ));
    message += "   " + UtlDateTime.zeit2String( UtlDateTime.getTagesende( -1L ));
    System.out.println(message);
}

  /**
   * liegen die beiden übergebenen Zeitpunkte im selben Tag?
   * @param zp1 Zeitpunkt1
   * @param zp2 Zeitpunkt2
   * @return true: Die beiden Zeitpunkte beziehen sich auf denselben Tag, false sonst
   */
  public static boolean amSelbenTag(Long zp1, Long zp2)
  {
    Boolean bret = false;
    if ( (zp1==null) || (zp2==null) )
      return(false);

    int tag1 = extractFieldValues(zp1, Calendar.DATE);
    int tag2 = extractFieldValues(zp2, Calendar.DATE);

    if ( (tag1==tag2) && (Math.abs(zp1-zp2)<24*3600000) )
    {
      bret=true;
    }
    return bret;
  }


  /**
   * Testprogramm für pruefeAmSelbenTag
   */
  public static void testAmSelbenTag()
  {
    boolean bret = true;
    long zp1 = UtlDateTime.string2Zeit( "13.04.2008 14:30:00" );
    long zp2 = UtlDateTime.string2Zeit( "13.04.2008 17:30:00" );
    if ( (amSelbenTag(zp1, zp2)==false) )
      bret = false;

    zp1 = UtlDateTime.string2Zeit( "13.04.2008 17:30:00" );
    zp2 = UtlDateTime.string2Zeit( "13.04.2008 14:00:00" );
    if ( (amSelbenTag(zp1, zp2)==false) )
      bret = false;

    zp1 = UtlDateTime.string2Zeit( "13.04.2008 14:30:00" );
    zp2 = UtlDateTime.string2Zeit( "13.03.2008 14:30:00" );
    if ( (amSelbenTag(zp1, zp2)==true) )
      bret = false;

    if ( bret )
    {
      String meldung = "Test erfolgreich";
      System.out.println(meldung);
    }
    else
    {
      String meldung = "Test fehlgeschlagen";
      System.err.println(meldung);
    }
  }


  /**
   * Testprogramm für extractFieldValues
   */
  public static void testFields2Zeit()
  {
    long zeitMs = UtlDateTime.fields2Zeit(2007, 4, 28, 15, 28, 3);
    System.out.println(UtlDateTime.zeit2String( zeitMs ));
  }

  /**
   * Testprogramm für extractFieldValues
   */
  public static void testFieldValues()
  {
    long zeitMs = string2Zeit("29.04.2008 15:32:28");
    System.out.println( extractFieldValues(zeitMs, Calendar.DAY_OF_MONTH) );
    System.out.println( extractFieldValues(zeitMs, Calendar.MONTH) ); 
    System.out.println( extractFieldValues(zeitMs, Calendar.YEAR) );
    System.out.println( extractFieldValues(zeitMs, Calendar.HOUR_OF_DAY) ); // 24 H
    System.out.println( extractFieldValues(zeitMs, Calendar.HOUR) ); // 12 H
    System.out.println( extractFieldValues(zeitMs, Calendar.MINUTE) );
    System.out.println( extractFieldValues(zeitMs, Calendar.SECOND) );
    System.out.println( extractFieldValues(zeitMs, Calendar.MILLISECOND) );
}
  
  /**
   * @param args Parameter
   */
  public static void main( String[] args )
  {
    String ss = date2String(new Date(), "dd.MM.yyyy HH:mm:ss,SSS");
    System.out.println(ss);
  }

}
