/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.dcf77.ntp.realclient;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.regex.Pattern;

import de.bsvrz.sys.dcf77.zeitverw.DataAspZustand;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Liest zyklisch die aktuelle NTP Server Info mit ntpq -p;
 * Aenderungen werden dem RealNTPClient gemeldet
 * @author ChriestenJ
 * @author Braeuner, BaWü, RPT, LST
 * Korrekturen und Änderungen 03/2016
 */


public class NTPinfoPoller extends Thread
{
  private Debug _debug = Debug.getLogger();

  /**
   * Kommando zur Abfrage der NTP Info.
   * Das ausgeführte Kommando muss Standard ntp Client Info (wie bei ntpq -p)
   * auf den Output stream schreiben
   * Eine Meldungen auf den Error Stream wird als Fehler gewertet
   */
  private String m_cmd;

  /**
   * Klasse, die das Interface implementiert. Von hier holt der
   * Zeitverwaltungs SenderThread seine Info
   */
  private RealNTPClient m_realNTPClient;

  /**
   * die zuletzt gelesene ServerInforamtion
   */
  private NtpServerInfo m_lastNtpServerInfo = null;

  /**
   * Zaehler Lesefehler. Bei 5 Fehlern wird Programm abgebrochen.
   */
  private int m_failCount = 0;

  private Boolean m_shutDown = false;

  /**
   * Beendet den Thread dieser Instanz
   */
  public void set_shutDown()
  {
    this.m_shutDown = true;
  }

  // Steuerung Thread Wartezeiten und Beendigung
  //-------------------------------------------------------

  /**
   * Zeitdauer zwischen 2 NTP ServerInfo Abfragen in sec
   */
  private long m_zyklusZeit;

  /**
   * Fuer wait-Zyklus. Beim ersten mal nicht warten
   */
  private Boolean m_erstesMal = true;

  /**
   * Anzahl bisheriger Zyklen.
   */
  private long m_zyklusCount = 0;

  /**
   * Maximale Zyklusanzahl.
   */
  private long m_zyklusCountMax = 5;

  /**
   * offset nur Protokollieren, wenn betragsmäßig grösser
   * negativ: immer protokollieren
   */
  private long m_oschwelle = -1L;
  //-------------------------------------------------------



  /**
   * Konstruktor
   * @param cmd Auszufuerendes Kommando. Ausgabe wie ntpq -p
   * @param realNTPClient realer NTP Client
   * @param zyklusZeit Wartezeit zwischen Pruefungen in sec
   * @param zyklusCountMax höchstens so viele Schleifen durchlaufen (-1) unendlich
   * @param oschwelle Abweichung offset nur protokollieren,
   * wenn Schwelle betragsmäßig über oder unterschritten wird
   */
  public NTPinfoPoller(String cmd, RealNTPClient realNTPClient,
                       long zyklusZeit, long zyklusCountMax, long oschwelle)
  {
    m_cmd = cmd;
    m_realNTPClient = realNTPClient;
    m_zyklusZeit = zyklusZeit;
    m_zyklusCountMax = zyklusCountMax;
    m_oschwelle = oschwelle;
System.out.println("OSCWELLE="+m_oschwelle);
  }

  /**
   * Hauptoutine der Klasse. Startet das Kommando (ntpq -p) zur Abfrage der ServerInfo.
   * Aenderungen werden der Zeitverwaltung zu Verfuegung gestellt.
   */
  private void checkNewServerInfo()
  {
    int ret;
    DataAspZustand aktuellerZustand;

    // Server Info lesen
    NtpServerInfo ntpServerInfo = new NtpServerInfo();
    ret = readData(m_cmd, ntpServerInfo);
    if (ret < 0)
      m_failCount = m_failCount+1;
    else
      m_failCount = 0;

    if ( m_failCount > 1000)
    {
      // Programm beenden -- too many errors
      this.set_shutDown();
    }

    //DEBUG
//    ntpServerInfo.aus( "neue Serverinfo" );
//    if (m_lastNtpServerInfo != null)
//      m_lastNtpServerInfo.aus( "alte Serverinfo" );
//    aktuellerZustand = cpNtpServerInfo2DataAspZustand(ntpServerInfo);
//    aktuellerZustand.aus("Ausgabe AspZustand");


    // Protokollrelevante Aenderung?
    if ( !ntpServerInfo.equals(m_lastNtpServerInfo) || (m_failCount==20))
    {
      aktuellerZustand = cpNtpServerInfo2DataAspZustand(ntpServerInfo);
      m_realNTPClient.setAktuellerZustand( aktuellerZustand, true );
    }

    m_lastNtpServerInfo = ntpServerInfo;
  }

  /**
   * Hilffunktion: Konvertiert ntpServerInfo in eine Instanz des Typs DataAspZustand
   * @param ntpServerInfo zu konvertierende Instanz
   * @return konvertierte Instanz
   */
  private DataAspZustand cpNtpServerInfo2DataAspZustand(NtpServerInfo ntpServerInfo)
  {
    // vorhandeneZeitserver Collection in String[] Array kopieren
    //-------------------------------------------------------------
    String[] vorhandeneZeitserver;
    int size = ntpServerInfo.vorhandeneZeitserver.size();
    if ( size == 0 )
    {
      vorhandeneZeitserver = new String[1];
      vorhandeneZeitserver[0] = "n/a";
    }
    else
    {
      vorhandeneZeitserver = new String[size];
      int i = 0;
      for (Iterator<String> iterator = ntpServerInfo.vorhandeneZeitserver.iterator(); iterator.hasNext();)
      {
        vorhandeneZeitserver[i] = iterator.next();
        i++;
      }
    }
    //-------------------------------------------------------------

    // letzte Korrekturzeit ermitteln
    //-------------------------------------------------------------
    Date today = new java.util.Date();
    SimpleDateFormat formatter =
      new SimpleDateFormat("dd.MM.yyyy, HH:mm:ss");
    String mydate = formatter.format( today );
    String letzteKorrekturZeit = mydate+","+ntpServerInfo.offset;

    DataAspZustand dataAspZustand = new DataAspZustand(ntpServerInfo.aktuellerZeitserver, vorhandeneZeitserver,
        letzteKorrekturZeit);
    //-------------------------------------------------------------

    return(dataAspZustand);
  }


  /**
   * Startet das uebergebene Kommando (ntpq -p),
   * interpretiert dessen Ausgabe und stellt es in der Instanz
   * ntpServerInfo zur Verfügung.
   * @param cmd Auszuführendes Kommando, Ausgabe muss ntpq -p entsprechen
   * @param ntpServerInfo Ausgabeinstanz
   * @return >0: ok, <0: Fehler
   */
  private int readData (String cmd, NtpServerInfo ntpServerInfo)
  {
    if ( cmd.equals( "NULLCOMM" ))
    {
      System.out.println("NULL Command. No Action.");
      return 1;
    }

    String s = null;

    BufferedReader stdInput = null;
    BufferedReader stdError = null;
    Process p = null;
    try
    {
      // run the command
      p = Runtime.getRuntime().exec(m_cmd);

      _debug.finest("Kommando #" + m_cmd + "# wurde abgesetzt.");

      stdInput = new BufferedReader(new
             InputStreamReader(p.getInputStream()));

      stdError = new BufferedReader(new
             InputStreamReader(p.getErrorStream()));

      if (p != null && stdInput != null && stdError != null)
      _debug.finest("Alle Variablen zur Verarbeitung des Kommandos initialisiert.");

      // Warte auf Input
      Boolean bret = warteBereit(stdError, stdInput);
      if ( !bret )
      {
        String errMessg = "Kommando \""+m_cmd+"\" antwortet nicht.";
        _debug.warning( errMessg );
        return(-1);
      }

      // Fehler vom Kommando?
      // --------------------------------------------------------------------------
      String cmdError = new String();
      Boolean flgCmdError = false;

      while ( (stdError.ready()) && (s = stdError.readLine()) != null)
      {
          cmdError = cmdError+"\n"+s;
          flgCmdError = true;
      }

      if ( flgCmdError )
      {
        String errMessg = "Kommando \""+m_cmd+"\" meldet Fehler.";
        errMessg += "\n" + cmdError;
        _debug.warning( errMessg );
        return(-1);
      }
      // --------------------------------------------------------------------------


      // Interpretiere die Kommandoausgabe
      // --------------------------------------------------------------------------
      String cmdOutput = new String();

      // read the output from the command

      while ( (stdInput.ready()) && ((s = stdInput.readLine()) != null)
             )
      {
        cmdOutput = cmdOutput + "\n" + s;
      }

      _debug.finest("Antwort auf das Kommando: #" + cmdOutput);

      //System.out.println(cmdOutput);
      parseInfo(cmdOutput, ntpServerInfo);
      // Ersatzserver relevant?
      if (  (ntpServerInfo.aktuellerZeitserver== "n/a") && ntpServerInfo.ersatzOffset != null)
      {
        //ntpServerInfo.offset = ntpServerInfo.ersatzOffset;
      }

      return(1);
      // --------------------------------------------------------------------------
    }
    catch (Exception e)
    {
        String errMssg = "Die Verarbeitung von \""+m_cmd+"\" ist fehlgeschlagen.";
        errMssg += "\n" + e.getMessage();
        _debug.warning( errMssg );
        e.printStackTrace();
        return(-1);
    }
    // jc neu eingefuegt 17.01.2011
    // jc modifiziert 30.05.2011
    finally
    {
      //Close the Process and (before) all Process - Streams!
      destroyProcess(p);

      // Notwendig??
      myClose(stdInput);
      myClose(stdError);
    }
  }

  /**
   * zerstort p und gibt dessen resourcen frei
   *
   * @param p Prozess, der zerstoert werden soll
   */
  private void destroyProcess(Process p)
  {
    if ( p==null )
      return;

    myClose( p.getInputStream() );
    myClose( p.getErrorStream() );
    myClose( p.getOutputStream() );
    p.destroy();
  }

  /**
   * schliesst das uebergebene closable. Wenn es sich um einen
   * Standardstream handelt wird nicht geschlossen.
   * @param c zu schliessendes Objekt
   */
  public void myClose(Closeable c)
  {
    if ( c==null )
      return;

    boolean isStdStream = c==System.out || c==System.in || c==System.err;
    if ( !isStdStream )
    {
      try
      {
        c.close();
      }
      catch ( IOException e )
      {
        String errMssg = "c.close() fehlgeschlagen";
        _debug.warning( errMssg );
        System.err.println(errMssg);
      }
    }
  }

  /**
   * Wartet bis br1 oder br2 bereit sind, Hilffunktion zu {@link #warteBereit(BufferedReader, BufferedReader)}
   * Lesen vor bereit führt in manachen Fällen zum Aufhängen des Lesebefehls.
   * Falls keiner der Reader innerhalb von 6,15 sec bereit ist, wird das Programm beendet
   *
   * @param br1 zu pruefender BufferedReader
   * @param br2 zu pruefender BufferedReader
   *
   * @return true: mindestens ein Reader ist bereit, false sonst
   * @throws InterruptedException
   * @throws Exception
   */
  private Boolean warteBereit( BufferedReader br1, BufferedReader br2 ) throws IOException, InterruptedException
  {
    int zaehler = 0;
    int warteZeit = 50;
    int maxWaits = (6000 - 3*warteZeit) / 500 + 1 + 3;
    while ( zaehler < maxWaits && !br2.ready() && !br1.ready() )
    {
      if ( zaehler == 3 )
      {
        warteZeit = 500;
      }

      Thread.sleep( warteZeit );
      zaehler++;
    }

    if ( zaehler < maxWaits )
      return (true);
    else
      return (false);
  }


  /**
   * Hilfvariablen zur Ausgabeinterpretation
   */
//  private final int REMOTE = 1;
//  private final int REFID = 2;
//  private final int ST = 3;
//  private final int WHEN = 5;
//  private final int REACH = 7;
//  private final int OFFSET = 9;
//  private final int JITTER = 10;

  // Korrektur Br: Array cols[] ist null-basiert
  private final int REMOTE = 0;
  private final int REFID = 1;
  private final int ST = 2;
  private final int WHEN = 4;
  private final int REACH = 6;
  private final int OFFSET = 8;
  private final int JITTER = 9;

  /**
   * private Klasse zur Speicherung der gelesenen Serverinformation
   * @author chriesten
   *
   */
  class NtpServerInfo
  {
    String aktuellerZeitserver = "n/a";
    ArrayList<String> vorhandeneZeitserver = new ArrayList<String>();
    String refid = new String();
    String st  = new String();
    String t  = new String();
    String when  = new String();
    String poll  = new String();
    String reach  = new String();
    String delay  = new String();
    String offset  = "n/a";
    String jitter = new String();
    // bei diesem Server sind die Spalten when und offset gesetzt.
    String ersatzZeitserver = null;
    String ersatzOffset  = null;

    /**
     * ueberschreibt die equals Methode
     *
     * 2 Instanzen werden als gleich betrachtet,
     * falls der aktuellerZeitserver und offset Komponenten uebereinstimmen
     * @param compObj Vergleichsobjekt
     * @return true (gleich) oder false (ungleich)
     */
    @Override
    public boolean equals(Object compObj)
    {
      if ( this == compObj ) return true;

      if ( !(compObj instanceof NtpServerInfo) ) return false;
      //Alternative to the above line :
      if ( compObj == null || compObj.getClass() != this.getClass() ) return false;

      //cast to native object is now safe
      NtpServerInfo that = (NtpServerInfo)compObj;

      return
        (this.aktuellerZeitserver.equals( that.aktuellerZeitserver)) &&
        (this.offset.equals(that.offset))
        ;
    }

    /**
     *
     * sollen Änderungen protokolliert werden?
     * falls der aktuellerZeitserver != dem Vergleichszeitserver
     * oder offset != vergleichsoffset und einer der offsets überschreitet Schwelle ->
     * Änderung protokollieren
     * @param compObj Vergleichsobjekt
     * @return true (protokollieren) oder false (nich protokollieren)
     */
    public boolean protAenderung(Object compObj)
    {
      if ( this == compObj ) return false;

      if ( !(compObj instanceof NtpServerInfo) ) return true;
      //Alternative to the above line :
      if ( compObj == null || compObj.getClass() != this.getClass() ) return true;

      //cast to native object is now safe
      NtpServerInfo that = (NtpServerInfo)compObj;


      // ---------------------------------
      // Schwellprüfung
      // ---------------------------------
      boolean schwellPruefung = false;
      if ( m_oschwelle < 0L )
        schwellPruefung = true;

      Float myoffset;
      try
      {
        myoffset = new Float(this.offset);
        if (Math.abs( myoffset ) > m_oschwelle )
          schwellPruefung = true;
        myoffset = new Float(that.offset);
        if (Math.abs( myoffset ) > m_oschwelle )
          schwellPruefung = true;
      }
      catch ( Exception e )
      {
        schwellPruefung = true;
      }
      // ---------------------------------

      return
        (!this.aktuellerZeitserver.equals( that.aktuellerZeitserver)) ||
        ((!this.offset.equals(that.offset) && schwellPruefung))
        ;
    }

    /**
     * Testfunktion. Schreibt die aktuellen Objektdaten auf die Console.
     *
     * @param titel Ueberschrift der Testausgabe
     */
    public void aus(String titel)
    {
      System.out.println("");
      System.out.println("------- " + titel + " ---------------------------------");
      System.out.println("AktuellerZeitServer="+aktuellerZeitserver);

      System.out.println("VorhandeneZeitServer:");
      if ( vorhandeneZeitserver != null )
      {
        for (Iterator<String> iterator = vorhandeneZeitserver.iterator(); iterator.hasNext();)
        {
          String vorhandenerZeitserver = iterator.next();
          System.out.println("     "+vorhandenerZeitserver);
        }
      }

      System.out.println("refid="+refid);
      System.out.println("poll="+poll);
      System.out.println("offset="+offset);
      System.out.println("when="+when);
      System.out.println("");
    }
  }

  /**
   * Interpretiert cmdInfo (Ausgabe von ntpq -p)
   * @param cmdInfo String mit der Ausgabe von ntpq -p
   * @param ntpServerInfo Ergebnisinstanz
   */
  private void parseInfo (String cmdInfo, NtpServerInfo ntpServerInfo)
  {
	  _debug.finest("Beginn der Verarbeitung der Antwort");

    // Ausgabe von ntpq zeilenweise auswerten
    // und in Struktur bereitstellen
    // ------------------------------------------------
    Pattern p = Pattern.compile( "[\n]" );
    String[] lines = p.split(cmdInfo);
//    for (int i = 0; i < lines.length; i++)
//    {
//      if ( i >= 3 ) // Line 0: leer, 1: header, 2: trenner
//      {
//        parseInfoLine(lines[i], ntpServerInfo);
//      }
//    }
    // Änderung Br: Allgemeingültige Interpretation der Ausgabe
    boolean trenner = false;
    for (String zeile : lines) {
    	_debug.finest("Zeile: " + zeile);
    	// Standardmäßig drei Kopfzeilen
    	// Zeile 0: Leerzeile, wird in readData eingefügt: cmdOutput = cmdOutput + "\n" + s;
    	// Zeile 1: Header  "     remote           refid      st t when poll reach   delay   offset  jitter"
    	// Zeile 2: Trenner "=============================================================================="
    	if (trenner) {  // Trenner wurde identifiziert
    		parseInfoLine(zeile, ntpServerInfo);
    	}
    	if (zeile.startsWith("===="))
    		trenner = true;
    }
    // Ende Änderung Br
    Collections.sort(ntpServerInfo.vorhandeneZeitserver);
    // ------------------------------------------------
  }

  /**
   * Interpretiert eine Zeile des Kommandos (ntpq -p)
   * @param line eine Ergebniszeile (kein Header)
   * @param ntpServerInfo Ergebnisstruktur
   */
  private void parseInfoLine (String line, NtpServerInfo ntpServerInfo)
  {
	  _debug.finest("Beginn der Verarbeitung der einzelnen Zeilen");
	  
    String[] cols = null;
    Character z1 = line.charAt( 0 );
    line = line.substring( 1 );
    // linkes Pattern: 0. Spalte aller Eintraege ignorieren 
    // Pattern p = Pattern.compile( "^[*+ ]|[\\s]+" );
    Pattern p = Pattern.compile( "^|[\\s]+" );
   cols = p.split(line);
   
   _debug.finest("Abgetrenntes erstes Zeichen: " + z1 + ", restliche Zeile: " + line);

    switch ( z1)
    {
      case('*'):
    	  _debug.finest("Aktueller Server (*) identifiziert");
      
      _debug.finest("Ausgabespalten: remote: " + cols[0] + ", refid: " + cols[1] + 
    		  ", st: " + cols[2] + ", when: " + cols[4] + ", reach: " + cols[6] + 
    		  ", offset: " + cols[8] + ", jitter: " + cols[9]);
      
        ntpServerInfo.aktuellerZeitserver = cols[REMOTE];
        ntpServerInfo.vorhandeneZeitserver.add(cols[REMOTE]+"(*)");
        ntpServerInfo.refid = cols[REFID];
        ntpServerInfo.st = cols[ST];
        ntpServerInfo.reach = cols[REACH];
        ntpServerInfo.offset = cols[OFFSET];
        ntpServerInfo.when = cols[WHEN];
        ntpServerInfo.jitter = cols[JITTER];
        
        _debug.finest("Datenstruktur: aktueller Zeitserver: " + ntpServerInfo.aktuellerZeitserver +
        ", refid: " + ntpServerInfo.refid + ", st: " + ntpServerInfo.st + ", reach: " + ntpServerInfo.reach + 
        ", offset: " + ntpServerInfo.offset + ", when: " + ntpServerInfo.when + ", jitter: " + ntpServerInfo.jitter);
        
        break;
        
      case('+'):
        ntpServerInfo.vorhandeneZeitserver.add(cols[REMOTE]+"(+)");
        break;
        
      case(' '):
      case('-'):
        ntpServerInfo.vorhandeneZeitserver.add(cols[REMOTE]+"(-)");
        break;
    }

    if (line.charAt( 0 ) != '*')
    { // kein aktueller Server, als Ersatzserver verwendbar?
      if ( !(cols[WHEN].equals( "-")) && !(cols[OFFSET].equals( "0.000")) ) 
      {
        ntpServerInfo.ersatzZeitserver = cols[REMOTE];
        ntpServerInfo.ersatzOffset = cols[OFFSET]+"("+cols[REMOTE]+")";
      }
    }
    
  }
  
  /** 
   * Wrapper Funktion fuer wait()
   * beim ersten Aufruf nach Obejtkinstanziierung wird nicht gewartet
   * Wenn der Thread beendet werden soll, liefert die Routine false, ansonsten true
   * 
   * @return false (true): Thread (nicht) beenden 
   */
  private synchronized Boolean mywait() 
  { 
    if ( m_zyklusCount == m_zyklusCountMax )
    {
      this.set_shutDown();
      return(!m_shutDown);
    }

    m_zyklusCount = m_zyklusCount+1;
    
    // Beim ersten Mal nicht warten
    if (m_erstesMal)
    {
      m_erstesMal = false;
      return(!m_shutDown);
    };
    
    try
    {
      wait(m_zyklusZeit*1000);
    }
    catch ( InterruptedException e )
    {
      _debug.warning("NTPinfoPoller.mywait: InterruptedException Caught");
      //m_interrupted = true;
    }
    catch ( Exception e )
    {
      String errMess = "NTPinfoPoller.mywait: Exception Caught";
      errMess += "\n"+e.getMessage();
      _debug.warning( errMess );
      e.printStackTrace();
    }
    finally
    {
      //System.out.println("ServerThread: Wartezyklus beendet");
    }
    
    return( !m_shutDown);
  } 

  /**
   * Liest zyklisch die Zeitserver Informationen in einer Endlosschleife. 
   * Aenderungen werden an dem {@link RealNTPClient} geliefert. 
   */
  @Override 
  public synchronized void run() 
  { 
    while(mywait())
    {
      this.checkNewServerInfo();
    }

    System.out.println( "Der Thread de.bsvrz.sys.dcf77.ntp.realclient.NTPinfoPoller wurde beendet!" );
    System.out.println( "Die ganze Applikation wird gestoppt!" );
    System.exit(5);
  } 


  /**
   * Testroutine fuer das lokale Modul.
   * Wird fuer verschiedene Tests im Laufe des Entwicklungs- und Wartungsprozesses verwendet
   * 
   * @param args Testargumente
   */
  public static void main( String[] args )
  {
    String line = "*alpha           .DCFa.           1 u    0   64  377    0.399   -1.49    0.617";
    line = line.substring( 1 );
    Pattern p = Pattern.compile( "^|[\\s]+" );
    String cols[] = p.split(line);
    for (String col : cols)
    {
      System.out.println(col);
    }
    if ( 1==1 )
      return;
    
    String cmd = new String ("rsh axpu22 -l root -n /usr/xntp/ntp-4.1.2/ntpq/ntpq -p");
    //String cmd = new String ("C:/Programme/NTP/bin/ntpq -p");
    //String cmd = new String ("ntpq -p");
    RealNTPClient realNTPClient = new RealNTPClient();
    NTPinfoPoller infoPoller = new NTPinfoPoller(cmd, realNTPClient, 2L, -1L, -1L);
    infoPoller.start();
    try
    {
      sleep(10000);
    }
    catch ( InterruptedException e )
    {
      e.printStackTrace();
    }
    infoPoller.interrupt();
    infoPoller.set_shutDown();
    System.out.println("Main beendet");
  }

}
