/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.dcf77.lib;

import java.util.HashMap;
import java.util.Iterator;
import java.util.regex.Pattern;

import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Hilfsklasse, enthaelt verschiedene Routinen um Argumente zu scannen.
 * @author chriesten
 *
 */
public class ArgumentParser
{
  private static Debug _debug = Debug.getLogger();

  /**
   * Argument Parser (String). 
   * argString wird in Schluessel(Argument)/Werte(Argumentwert) 
   * Paare geparst
   * @param argString String mit Argumenten z.B. <br>
   * String argString="---cmd=ntpq -p ---zyklusZeit=2L ---maxZyklen=20L";
   * @param argSep Argumenttrenner z.B. "---"
   * @return HashMap mit den Argument Werten Paaren z.B <br>
   * "cmd"/"ntpq -q", "zyklusZeit"/"2L", "maxZyklen"/"20L"
   */
  public static HashMap<String, String> parseArgString (String argString, String argSep)
  {
    //String argString="---cmd=/usr/xntp/ntp-4.1.2/ntpq/ntpq -p ---zyklusZeit=2L ---maxZyklen=20L"; 
    HashMap<String, String> schluesselWertPaare; 

    String[] args = splitZk (argString, argSep, 0);

    schluesselWertPaare = parseArgStrings(args, "=");
    return(schluesselWertPaare);
  }

  /**
   * Teilt die übergebenen zuTrennen Zeichenkette. Trenner ist der übergebene reguläre Ausdruck
   * trenner. max entspricht der Maximalzahl der Ergebniszeilen
   * vgl. 2. Prameter in Pattern.split()
   *
   * @param zuTrennen Trennstring siehe obige Beschreibung
   * @param trenner Trenner
   * @param max siehe obige Beschreibung
   * @return Array mit den Teilstrings
   */
  public static String[] splitZk (String zuTrennen, String trenner, int max)
  {
    // extrahiere ParameterWerte Strings
    String[] args = null;
    Pattern p = Pattern.compile( trenner );
    args = p.split(zuTrennen, max);

    // doppelte blanks und blanks vorne und hinten entfernen
    for (int i = 0; i < args.length; i++)
    {
      args[i] = args[i].replaceAll( "[ ]+", " ").trim();
    }

    return(args);
  }

  /**
   * Argument Parser (String Array).
   * jedes Zeile in args wird in Wertepaare gesplittet
   * @param args StringArray mit Argumenten der Form schluessel=wert
   * @param argSep Argumenttrenner z.B. "="
   * @return HashMap mit den Argument Werten Paaren der Form schluessel/wert
   */
  public static HashMap<String, String> parseArgStrings ( String[] args, String argSep )
  {
    HashMap<String, String> schluesselWertPaare = new HashMap<String, String>();

    // Erzeuge Argument- Wertepaare
    Pattern p = Pattern.compile( argSep );
    for (String argWert : args)
    {
      if ( (argWert==null) || (argWert=="") )
      {
        System.err.println("leerer Parameter übergeben!!");
        continue;
      }

      String argPaare[] = p.split(argWert, 2);
      if ( argPaare.length == 2 )
        schluesselWertPaare.put( argPaare[0].trim(), argPaare[1].trim());
    }

    return(schluesselWertPaare);
}

  /**
   * liefert zum uebergebene Schluessel (Argument) den zugehoerigen Wert
   * @param schluesselWertPaare mit parseArgString generierte Hash map
   * @param schluessel Argument
   * @param nichtGefundenFehler true: Fehlermeldung, falls schluessel nicht gefunden
   * @param loeschen true: Wertepaare nach dem Auslesen entfernen
   * @return Wert zum schluessel oder null, falls schluessel nicht vorhanden
   */
  public static String getWert(HashMap<String, String> schluesselWertPaare, String schluessel, Boolean nichtGefundenFehler, Boolean loeschen)
  {
    String wert = schluesselWertPaare.get(schluessel);
    if ( nichtGefundenFehler && wert == null )
    {
      String errMess = "notwendiger Parameter " +schluessel +" nicht spezifiziert";
      _debug.error( errMess );
      System.exit( -1 );
    }

    if ( loeschen && (wert != null) )
      schluesselWertPaare.remove( schluessel );

    return(wert);
  }

  /**
   * Argument Konverter (String -&gt; String[]).
   * Der Parameterstring argString wird geparst (siehe ArgumentParser()).
   * Anschließend wird jedes Argument/Werte Paare in eine eine Zeile des Ergebnisfelds argString
   * kopiert.
   * 
   * 
   * @param argString String mit Argumenten z.B. <br>
   * String argString="---cmd=ntpq -p ---zyklusZeit=2 ---maxZyklen=20";
   * @param argSep Argumenttrenner z.B. "---"
   * @return Argument String Feld mit den Argument Werten Paaren z.B <br>
   * -cmd=ntpq -q
   * -zyklusZeit=2 
   * -maxZyklen=20L
   */
  public static String[] konvArgString (String argString, String argSep)
  {
    HashMap<String, String> schluesselWertPaare = ArgumentParser.parseArgString(argString, "---");
    String[] args = new String[schluesselWertPaare.size()];
    int i = 0;
    for (Iterator<String> iterator = schluesselWertPaare.keySet().iterator(); iterator.hasNext();)
    {
      String mykey = iterator.next();
      args[i] = "-"+mykey + "=" + schluesselWertPaare.get(mykey);
      i++;
    }
    return(args);
  }

  /**
   * Testroutine fuer das lokale Modul.
   * Wird fuer verschiedene Tests im Laufe des Entwicklungs- und Wartungsprozesses verwendet
   * 
   * @param args Testargumente
   */
  public static void main( String[] args )
  {
    //String argString="cmd=/usr/xntp/ntp-4.1.2/ntpq/ntpq -p ---zyklusZeit=2L ---maxZyklen=20L"; 
    String argString="datenverteiler=192.168.1.34:8083---datenverteiler=localhost:8083---datenverteiler=192.0.1.97:8085---debugLevelStdErrText=FINE   ---benutzer=Tester---authentifizierung=C:/Datenverteiler/skripte-dosshell/passwd---rechner=rechner.Rechner1---ntpKlasse=de.bsvrz.sys.dcf77Test.NTPClientTester---zyklusZeit=-1";

    HashMap<String, String> schluesselWertPaare = ArgumentParser.parseArgString(argString, "---");
    for (Iterator<String> iterator = schluesselWertPaare.keySet().iterator(); iterator.hasNext();)
    {
      String mykey = iterator.next();
      System.out.println("key="+mykey+"X" +" ,val="+ArgumentParser.getWert(schluesselWertPaare, mykey, true, false)+"X" );
    }
    if (true) return;
    
    String[] myargs =  konvArgString(argString, "---");
    for (int i = 0; i < myargs.length; i++)
    {
      System.out.println(myargs[i]+"xx");
    }
    if ( true ) return;
    
    String s[] = ArgumentParser.splitZk( "a     bcc,ef",",", 3 );
    for (String string : s)
    {
      System.out.println(string);
    }
    if ( true ) return;

  }

}
