package de.bsvrz.sys.dcf77.lib;


import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Fuer Logging Information. Debugausgaben können abgeschaltet und/oder in Datei
 * umgeleitet werden.
 *
 * @author Chriesten
 *
 */

public class MyLogger
{
  /**
   * Instanz der Klasse; es gibt nur eine (Singleton)
   */
  public static MyLogger myLogger = null;

  /**
   * aktueller PrintStream
   */
  private PrintStream m_out = null;

  /**
   * liefert den aktuellen Ausgabestrom
   * @return aktueller Ausgabestrom
   */
  public static PrintStream get_out()
  {

    return MyLogger.myLogger == null ? null : MyLogger.myLogger.m_out;
  }


  /**
   * Datum vor jede Ausgabezeile?
   */
  private Boolean m_mitDatum = false;


  /**
   * Ausgabe zusaetzlich auf System.out?
   */
  private Boolean m_auchSysOut = false;

  /**
   * nicht loggen?
   */
  private Boolean m_loggingAus = false;

  /**
   * vorherige Ausgabedatei
   */
  private static String _fileName_vorher = null;

  /**
   * aktuelle Ausgabedatei
   */
  private static String _fileName_akt = null;

  /**
   * voriger Initialisierungsmodus
   */
  private static long _modus_vorher;

  /**
   * aktueller Initialisierungsmodus
   */
  private static long _modus_akt;

  /**
   * Befindet sich die Ausgabe am Anfang einer neuen Zeile?
   */
  private Boolean m_neueZeile = true;

  /**
   * verdeckter Konstruktor. Wird von init() gerufen.
   */
  private MyLogger()
  {
  }

  /**
   * Initialisiert das Ausgabeziel.
   * Der Parameter dateiModus wird in dateiName und Modus aufgesplittet
   * und {@link MyLogger}{@link #init(String, long)} gerufen.
   * @param dateiModus dateiName:modus
   * @return true:ok, fehler sonst
   */
  public static synchronized Boolean init( String dateiModus)
  {
    int ind = dateiModus.lastIndexOf( ":" );
    String dateiName;
    long modus = 0;
    if ( ind != -1 )
    {
      dateiName = dateiModus.substring( 0, ind );
      try
      {
        modus = new Integer(dateiModus.substring( ind+1));
      }
      catch ( Exception e)
      {
        System.err.println("ungültiger Aufruf: MyLogger.init("+dateiModus+") ,"+e.getMessage() );
      }
    }
    else
    {
      dateiName = dateiModus;
      modus = 0;
    }
    Boolean bret = init( dateiName, modus);
    return(bret);
  }

  /**
   * Initialisiert das Ausgabeziel.
   * 
   * @param dateiName Logging Informationen in diese Datei schreiben. <br>
   * Sonderziele: out, err, zurueck, null:  Ausgabe auf System.out, System.err, das letzte Ausgabeziel, nur Fehlerausgabe<br>
   * &lt;UHRZEIT&gt; im Dateinamen wird ersetzt durch Datum/Uhrzeit
   * @param modus 1: anhaengen<br>
   *              2: Datum vor jede Ausgabezeile<br>
   *              4: Zusaetzlich Ausgabe auf System.out<br>
   * @return true: Initialisierung erfolgreich, false: Fehler
   */
  public static synchronized Boolean init( String dateiName, long modus)
  {
    //---------------------------------------------
    // vorhergehenden Output verwenden?
    //---------------------------------------------
    if ( dateiName.equalsIgnoreCase( "null" ))
      dateiName = null;
    
    if ( (dateiName!=null) && (dateiName.equalsIgnoreCase("ZURUECK")) )
    {
      if (myLogger != null)
      {
        dateiName = _fileName_vorher;
        modus = _modus_vorher | 1;
      }
      else
        dateiName = null;
    }

    Boolean anhaengen = (modus & 1) == 1; 
    Boolean mitDatum =  (modus & 2) == 2;
    Boolean auchSysOut = (modus & 4) == 4;
    
    String fileName = ersetzeZeitToken( dateiName );

    _fileName_vorher = _fileName_akt;
    _modus_vorher = _modus_akt;
    _fileName_akt = fileName;
    _modus_akt = modus;
    
    //---------------------------------------------
    // alten Logger freigeben
    //---------------------------------------------
    if ( myLogger != null )
    {
      myLogger.close();
      myLogger = null;
    }

    //---------------------------------------------
    // neuen Logger instanziieren
    //---------------------------------------------
    myLogger = new MyLogger();
    if ( myLogger == null )
      return(false);
    
    // default == kein Logging
    if (fileName == null )
    {
      myLogger.m_loggingAus = true;
      return(true);
    }

    Boolean bret = myLogger.open( fileName, anhaengen ); 
    if ( !bret )
    {
      myLogger = null;
      return(false);
    }
      
    myLogger.m_mitDatum = mitDatum;
    myLogger.m_auchSysOut = auchSysOut;

    return (bret);
  }

  public static synchronized void reinit()
  {
    if ( myLogger != null )
    {
      _fileName_vorher = _fileName_akt;
      myLogger.close();
      myLogger = null;
    }
  }

  /**
   * Schreibe Logging Info
   * 
   * @param kategorie Kategorie
   * @param ausgabe Ausgabe mit Zeilenumbruch.
   */
  public static void logln( String kategorie, String ausgabe )
  {
    MyLogger.privlog1( kategorie, ausgabe, true );
  }

  /**
   * Schreibe Logging Info
   *
   * @param kategorie Kategorie
   * @param ausgabe Ausgabe, ohne Zeilenumbruch.
   */
  public static void log( String kategorie, String ausgabe )
  {
    MyLogger.privlog1( kategorie, ausgabe, false );
  }

  /**
   * Schreibe Logging Info; private Wrapper Funktion.
   * 
   * @param ausgabe Ausgabe
   * @param nl true: mit NewLine
   */
  private static void privlog1( String kategorie, String ausgabe, boolean nl )
  {
    // Logging abgeschaltet?
    if ( (MyLogger.myLogger != null) && myLogger.m_loggingAus )
    {
      // Fehler trotzdem loggen!
      if ( (MyLogger.myLogger != null) && (kategorie == "ERROR" ) )
        MyLogger.myLogger.privlog( System.err, kategorie, ausgabe, nl );
      return;
    }
    
    if ( MyLogger.myLogger == null )
    {
      MyLogger.init("out", 1L);
      if ( MyLogger.myLogger == null )
      {
        return;
      }
    }

    MyLogger.myLogger.privlog( myLogger.m_out, kategorie, ausgabe, nl );
    if ( MyLogger.myLogger.m_auchSysOut && (MyLogger.myLogger.m_out != System.out) )
      MyLogger.myLogger.privlog( System.out, kategorie, ausgabe, nl );
  }

  /**
   * Schreibe Logging Info; private Aktionsfunktion.
   * 
   * @param kategorie Ausgabekategorie
   * @param ausgabe Ausgabe
   * @param nl true: mit NewLine
   */
  private void privlog( PrintStream ps, String kategorie, String ausgabe, boolean nl )
  {
    // Datum einfuegen?
    String meinDatum = "";
    if ( m_mitDatum && m_neueZeile )
    {
      SimpleDateFormat sdf = new SimpleDateFormat("yyMMdd_HHmmss_S ");
      meinDatum = sdf.format(new Date());
    }
    
    // Verarbeite kategorie
    String myKat = (kategorie == "" || !m_neueZeile) ? "" : kategorie+": ";

    if ( ps == null )
      return;

    try
    {
      ps.print( meinDatum + myKat + ausgabe );
      if ( nl )
      {
        ps.println( "" );
        m_neueZeile = true;
      }
      else
        m_neueZeile = false;

      ps.flush();
    }
    catch ( Exception e )
    {
      e.printStackTrace( System.err );
    }
  }

  /**
   * Oeffne Datei
   * 
   * @param fileName zu oeffnende Datei
   * @param append anhaengen?
   * @return true: open erfolgreich, false;: Fehler
   */
  private Boolean open( String fileName, boolean append )
  {
    if ( (fileName.equalsIgnoreCase( "zurueck" )) )
    {
      return (true);
    }

    if ( (fileName.equalsIgnoreCase( "out" )) || (fileName.equalsIgnoreCase( "System.out" )) )
    {
      m_out = System.out;
      return (true);
    }

    if ( (fileName.equalsIgnoreCase( "err" )) || (fileName.equalsIgnoreCase( "System.err" )) )
    {
      m_out = System.err;
      return (true);
    }

    try
    {
      m_out = new PrintStream( new FileOutputStream( fileName, append ) );
    }
    catch ( IOException e )
    {
      e.printStackTrace();
      return (false);
    }
    return (true);
  }

  /**
   * Schliesse Datei.
   */
  private void close()
  {
    if ( m_out != null )
    {
      m_out.flush();
    }

    if ( m_out != null && m_out != System.out && m_out != System.err )
    {
      try
      {
        m_out.close();
      }
      catch ( Exception e )
      {
        return;
      }
    }
    return;
  }
  
  /**
   * Ersetzt das token <UHRZEIT> in quelle durch Datum/Uhrzeit.
   * @param quelle
   * @return quelle mit der Ersetzung
   */
  private static String ersetzeZeitToken( String quelle )
  {
    if ( (quelle==null) || (quelle.indexOf( "<UHRZEIT>" ) == -1) )
      return(quelle);
    SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmss");
    String uhrzeit = sdf.format(new Date());
    return quelle.replaceAll( "<UHRZEIT>", uhrzeit);
  }
  

  /**
   * Testroutine fuer das lokale Modul. Wird fuer verschiedene Tests im Laufe des Entwicklungs- und Wartungsprozesses verwendet
   * 
   * @param args
   *          Testargumente
   */
  public static void main( String[] args )
  {
    Boolean bret;

    bret = MyLogger.init( "out", 2);
    MyLogger.log( "X", "123 " );
    MyLogger.logln( "X", "456" );
    bret = MyLogger.init( null, 2);
    MyLogger.logln( "ERROR", "789" );
    MyLogger.log( "X", "abc" );
    MyLogger.log( "X", "def" );
    bret = MyLogger.init( "out", 2);
    MyLogger.logln( "X", "ABC" );
    MyLogger.log( "X", "DEF" );
    MyLogger.logln( "", "1"+2+"33" );

  }
}