/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.dcf77.ntp.realclient;


import java.util.HashMap;

import de.bsvrz.sys.dcf77.lib.ArgumentParser;
import de.bsvrz.sys.dcf77.zeitverw.DataAspZustand;
import de.bsvrz.sys.dcf77.zeitverw.IntpAllgemein;
import de.bsvrz.sys.dcf77.zeitverw.IntpTrigger;

/**
 * Klasse realisiert einen realen NTP-Client. Der Client erfragt zyklisch den Zustand des NTP Servers. 
 * Der aktuelle Zustand wird der Zeitverwaltung ueber das Interface {@link IntpAllgemein} zur Verfuegung 
 * gestellt.<br>
 * Der ntp Status wird mit Hilfe des Programms ntpq -p überprueft.
 *
 * @author ChriestenJ
 *
 */

public class RealNTPClient implements IntpAllgemein
{
  /**
   * aktueller Aspekt "Recher.dcfStatus.Zustand"
   */
  private DataAspZustand m_aktuellerZustand = new DataAspZustand();

  /**
   * Schnittstellenobjekt um den Senderthread aufzuwecken
   */
  private IntpTrigger m_ntpTrigger = null;

  /**
   * Thread Klasse, die im Konstruktor gestartet wird
   */
  private NTPinfoPoller m_NTPinfoPoller = null;

  /**
   * Konstruktor mit leeren Parametern.
   * Defaultmaessig werden die Parameter folgendermassen belegt:
   * cmd=ntpq -p<br>
   * zyklusZeit=60<br>
   * maxZyklen=-1 (unendlich viele Zyklen)<br>
   * Parameterbeschreibung, siehe {@link #RealNTPClient(String)}
   */
  public RealNTPClient()
  {
    initKonstruktor( "" );
  }

  /**
   * Konstruktor
   *
   * @param argString optionaler Argumentstring <br>
   * ---cmd=... ---zyklusZeit=... ---maxZyklen=...<br>
   * Der Befehl cmd wird zyklisch alle zyklusZeit sec ausgefuehrt. Es werden maximal maxZyklen
   * durchgefuehrt.
   */
  public RealNTPClient( String argString )
  {
    initKonstruktor( argString );
  }

  /**
   * Hilfsfunktion fuer Konstrukturtoren.
   *
   * @param argString
   *          siehe Kontruktur mit Parameter argString
   */
  private void initKonstruktor( String argString )
  {
    // argString auswerten
    HashMap<String, String> schluesselWertPaare = ArgumentParser.parseArgString( argString, "---" );
    String cmd = ArgumentParser.getWert( schluesselWertPaare, "cmd", false, false );
    String zyklusZeitc = ArgumentParser.getWert( schluesselWertPaare, "zyklusZeit", false, false );
    String maxZyklenc = ArgumentParser.getWert( schluesselWertPaare, "maxZyklen", false, false );
    String oschwellec = ArgumentParser.getWert( schluesselWertPaare, "oschwelle", false, false );

    if ( cmd == null )
      cmd = new String( "/usr/sbin/ntpq -p" );
    if ( zyklusZeitc == null )
      zyklusZeitc = "60";
    if ( maxZyklenc == null )
      maxZyklenc = "-1";
    if ( oschwellec == null )
      oschwellec = "-1";

    long zyklusZeit = Long.parseLong( zyklusZeitc.trim() );
    long maxZyklen = Long.parseLong( maxZyklenc.trim() );
    long oschwelle = Long.parseLong( oschwellec.trim() );

    NTPinfoPoller infoPoller = new NTPinfoPoller( cmd, this, zyklusZeit, maxZyklen, oschwelle);
    infoPoller.start();
    m_NTPinfoPoller = infoPoller;
    System.out.println( "Instanz RealNTPClient initialisiert \n" + "cmd=" + cmd + " ,zyklusZeit=" + zyklusZeit
        + " ,maxZyklen=" + maxZyklen );
  }

  /*
   * (non-Javadoc)
   *
   * @see de.bsvrz.sys.dcf77.zeitverw.IntpAllgemein#getAktuellerZeitServer()
   */
  public String getAktuellerZeitServer()
  {
    return m_aktuellerZustand.aktuellerZeitserver;
  }

  /*
   * (non-Javadoc)
   *
   * @see de.bsvrz.sys.dcf77.zeitverw.IntpAllgemein#getLetzteZeitKorrektur()
   */
  public String getZeitKorrektur()
  {
    return m_aktuellerZustand.letzteKorrekturZeit;
  }

  /*
   * (non-Javadoc)
   *
   * @see de.bsvrz.sys.dcf77.zeitverw.IntpAllgemein#getVorhandeneZeitServer()
   */
  public String[] getVorhandeneZeitServer()
  {
    return m_aktuellerZustand.vorhandeneZeitserver;
  }

  /*
   * (non-Javadoc)
   *
   * @see de.bsvrz.sys.dcf77.zeitverw.IntpAllgemein#setInstance(de.bsvrz.sys.dcf77.zeitverw.IntpTrigger)
   */
  public void setInstance( IntpTrigger ntpTrigger )
  {
    m_ntpTrigger = ntpTrigger;
  }

  /**
   * Beendet den NTPClient
   */
  public void beenden()
  {
    m_NTPinfoPoller.interrupt();
    m_NTPinfoPoller.set_shutDown();
  }

  /**
   * Setzt den aktuellen Zustand im Zeitserverclient. Dieser Zustand wird vom SenderThread
   * der Zeitverwaltung zyklisch gelesen.
   *
   * @param aktuellerZustand
   *          Instanz mit den aktuellen Zeitserverdaten
   * @param wakeSenderThread
   *          Flag zur Unterbrechung des aktuellen Senderthread Warte-Zyklus, default = false
   */
  public void setAktuellerZustand( DataAspZustand aktuellerZustand, Boolean wakeSenderThread )
  {
    m_aktuellerZustand = aktuellerZustand;
    aktuellerZustand.aus( "RealNtpClient" );

    if ( wakeSenderThread && (m_ntpTrigger != null) )
      m_ntpTrigger.setTrigger();
  }

  /**
   * siehe {@link #setAktuellerZustand(DataAspZustand, Boolean)}
   *
   * @param aktuellerZustand
   *          Übergabe der aktuellen Zeitserverdaten als Objekt
   */
  public void setAktuellerZustand( DataAspZustand aktuellerZustand )
  {
    setAktuellerZustand( aktuellerZustand, false );
  }

  /**
   * Testroutine fuer das lokale Modul.
   * Wird fuer verschiedene Tests im Laufe des Entwicklungs- und Wartungsprozesses verwendet
   * 
   * @param args Testargumente
   */
  public static void main( String[] args )
  {
    // RealNTPClient realNTPClient2 = new RealNTPClient("---cmd=rsh axpu22 -l root -n /usr/xntp/ntp-4.1.2/ntpq/ntpq -p ---zyklusZeit=2 ---maxZyklen=20");
    RealNTPClient realNTPClient2 = new RealNTPClient();
    System.out.println( realNTPClient2 );
    if ( true )
      return;
    RealNTPClient realNTPClient = new RealNTPClient();
    System.out.println( realNTPClient );
    return;
  }

}
