/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.dcf77.zeitverw;

import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse die mit dem NTP (NetworkTimeProkol) Client kommuniziert.
 * Ueber das Interface IntpAllgemein wird zyklisch der NTP Zustand auf dem 
 * aktuellen Recher erfragt. Aenderungen werden im DAV ueber den Aspekt Recher.dcfStatus.Zustand 
 * protokolliert
 * 
 * @author ChriestenJ 
 */

public class SenderThread extends Thread implements IntpTrigger
{
  //---------------------------------------------------------------------------
  // Membervariablen 
  //---------------------------------------------------------------------------
  private Debug _debug = Debug.getLogger();

  /** 
   * Schnittstellenobjekt zum NTP Server
   */
  private IntpAllgemein m_ntpAllgemein;

  /** 
   * DCF DAV Verbindung. Objekt besitzt Datenverteilerverbindung
   * und stellt Methoden zur Versendung und Empfang des Aspekts recher.dcf.zustand an.
   */
  private DCFdav m_dcfDAV;

  /**
   * Zeitdauer zwischen 2 Zeitserverabfragen in sec
   */
  private long m_zyklusZeit;

  /** 
   * aktueller NTP Zustands
   */
  private DataAspZustand m_dataAspZustandNeu = new DataAspZustand();

  // ------------------------------------------------------------
  // fuer JUNIT Test (ohne DaV)
  // ------------------------------------------------------------

  // fuer Tests ohne Dav auf false
  private boolean m_schreibeInDav = true;
  
  /**
   * setzt m_schreibeInDav; 
   * @param schreibeInDav false: NTP Zustände nicht in den DaV schreiben ; nur für Tests.
   */
  public void set_schreibeInDav( boolean schreibeInDav )
  {
    m_schreibeInDav = schreibeInDav;
  }

  /**
   * gibt aktuellen NTP Zustands zurueck (== den zuletzt über das
   * Interface IntpAllgemein gelesenen Zustand).
   * Methode wird fuer JUNIT Test verwendet
   * leseNTPStatus
   * @return DataAspZustand
   */
  public DataAspZustand get_dataAspZustandNeu()
  {
    return m_dataAspZustandNeu;
  }

  /**
   * letzter NTP Zustands
   */
  private DataAspZustand m_dataAspZustandAlt = new DataAspZustand();

  /**
   * soll der Thread beendet werden?
   */
  Boolean m_shutDown = false;

  /**
   * Beendet den Thread dieser Instanz
   */
  public void set_shutDown()
  {
    this.m_shutDown = true;

    // unterbreche Warteschleife
    this.setTrigger();
  }


  /**
   * lokale Hilfsvariable zur Steuerung des waits
   */

  /*
   * lokale Hilfsvariable: erste Wartezyklus?
   */
  private Boolean m_erstesMal = true;

  /*
   * lokale Hilfsvariable: wieviele Zyklen wurden bereits ausgeführt?
   */
  private long m_zyklusCount = 0;

  /*
   * lokale Hilfsvariable: wieviele Zyklen sollen maximal ausgeführt werden?
   */
  private long m_zyklusCountMax = -1;

  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  // Konstruktoren
  //---------------------------------------------------------------------------

  /**
   * Konstruktor1: Initialisiert das Objekt.
   * @param ntpAllgemein Schnittstellenobjekt zum NTP Server
   * @param dcfDAV       Ueber dieses dcfDAV Objekt wird der DCF Zustand verschickt.
   * @param zyklusZeit   Wartezeit zwischen 2 Zeitserverabfragen in sec. (-1: nur auf Interrupts reagieren).
   */
  public SenderThread(IntpAllgemein ntpAllgemein,
                      DCFdav dcfDAV,
                      long zyklusZeit)
  {
    initialize_SenderThread(ntpAllgemein, dcfDAV, zyklusZeit, -1);
  }

  /**
   * Konstruktor2: Initialisiert das Objekt.
   * @param ntpAllgemein Schnittstellenobjekt zum NTP Server
   * @param dcfDAV       Ueber dieses dcfDAV Objekt wird der DCF Zustand verschickt.
   * @param zyklusZeit   Wartezeit zwischen 2 Zeitserverabfragen in sec. (-1: nur auf Interrupts reagieren).
   * @param zyklusCountMax Maximalzahl der Zyklen, die der SenderThread ausführt (-1: unendlich)
   */
  public SenderThread(IntpAllgemein ntpAllgemein,
      DCFdav dcfDAV,
      long zyklusZeit,
      long zyklusCountMax)
  {
    initialize_SenderThread(ntpAllgemein, dcfDAV, zyklusZeit, zyklusCountMax);
  }

  /**
   * lokale Hilfsroutine fuer die Konstruktoren
   */
  private void initialize_SenderThread(IntpAllgemein ntpAllgemein,
      DCFdav dcfDAV,
      long zyklusZeit,
      long zyklusCountMax)
  {
    m_ntpAllgemein = ntpAllgemein;
    m_dcfDAV = dcfDAV;
    m_zyklusZeit = zyklusZeit*1000;
    
    m_erstesMal = true;
    m_zyklusCountMax = zyklusCountMax;
  }

  /**
   * Konstruktor3 nur fuer lokale Tests
   */
  private SenderThread()
  {
    
  }
  //---------------------------------------------------------------------------

  
  
  /**
   * In einem Endlosloop wird zyklisch der Status des NTP (Zeitservers) eingelesen
   * Bei Datenaenderungen wird der geaenderte Status im DAV protokolliert. 
   */
  @Override 
  public synchronized void run() 
  {
    mywait (1000);
if (isInterrupted())
{
  _debug.fine("SenderThread isInterrupted");
}
    while ( (!isInterrupted() && (!m_shutDown)) && (m_zyklusCountMax != m_zyklusCount) )
    {
      mywait(m_zyklusZeit);
      leseUndSendeNTPStatus();
      
      m_zyklusCount = m_zyklusCount+1;
    }

    System.out.println( "Der Thread de.bsvrz.sys.dfctest.zeitverw.SenderThread wurde beendet!" );
  } 

  
  /**
   * Liest den aktuellen NTP Zustand
   * @return true Zustand seit dem letzten lesen geaendert, false sonst
   */
  public boolean leseNTPStatus()
  {
    //---------------------------------------------------------------------------
    // Einlesen aktueller Zustand und Vergleich mit letztem Zustand
    //---------------------------------------------------------------------------
    m_dataAspZustandAlt.copy(m_dataAspZustandNeu);
    m_dataAspZustandNeu.copy(m_ntpAllgemein.getAktuellerZeitServer(),
                             m_ntpAllgemein.getVorhandeneZeitServer(),
                             m_ntpAllgemein.getZeitKorrektur());

    Boolean aenderung = (!m_dataAspZustandAlt.equals(m_dataAspZustandNeu));
    if ( !aenderung )
    { // nichts zu tun
System.out.println("SenderThread: keine Aktion");
    }
    return(aenderung);
  }

  private void sendeNTPStatus()
  {
    // sende geaenderten Zustand an DAV
    System.out.println("SenderThread: sende neuen Zustand");
    m_dataAspZustandNeu.aus("SenderThread_Sendung");
    //MyLogger.logln( "TEST", m_dataAspZustandNeu.letzteKorrekturZeit );
    m_dcfDAV.sendDavZustand( m_dataAspZustandNeu );
  
    // Schreibe System Message
    return;
  }

  private void leseUndSendeNTPStatus()
  {
    Boolean aenderung = leseNTPStatus();
    if ( m_schreibeInDav )
    {
      if ( aenderung )
        sendeNTPStatus();
    }
  }

  
  /**
   * Liest den aktuellen NTP Zustand
   * Bei Aenderungen wird er ueber den DAV protokolliert.
   */
  private void leseUndSendeNTPStatusOld()
  {
    //---------------------------------------------------------------------------
    // Einlesen aktueller Zustand und Vergleich mit letztem Zustand
    //---------------------------------------------------------------------------
    m_dataAspZustandAlt.copy(m_dataAspZustandNeu);
    m_dataAspZustandNeu.copy(m_ntpAllgemein.getAktuellerZeitServer(),
                             m_ntpAllgemein.getVorhandeneZeitServer(),
                             m_ntpAllgemein.getZeitKorrektur());

    
    Boolean keineAenderung = m_dataAspZustandAlt.equals(m_dataAspZustandNeu);

    if ( keineAenderung )
    { // nichts zu tun
System.out.println("SenderThread: keine Aktion");
      return;
    }
    
    // sende geaenderten Zustand an DAV
System.out.println("SenderThread: sende neuen Zustand");
m_dataAspZustandNeu.aus("SenderThread_Sendung");
//MyLogger.logln( "TEST", m_dataAspZustandNeu.letzteKorrekturZeit );
    m_dcfDAV.sendDavZustand( m_dataAspZustandNeu );

    // Schreibe System Message

    return;
  }
  
  /**
   * Implementation der Schnittstelle IntpTrigger 
   * Methode zum Unterbrechen der wait-Zyklen im Sender Thread.
   */
  public synchronized void setTrigger()
  {
    this.notify ();
System.out.println(" SenderThread.setTrigger gerufen ");
  }
  
  /** 
   * Wrapper Funktion fuer wait()
   * bei timeout = -1000 wird ewig gewartet
   * beim ersten Aufruf nach Obejtkinstanziierung wird nicht gewartet
   * 
   * @param timeout Wartezeit in ms 
   */
  private synchronized void mywait(long timeout) 
  { 
    // Beim ersten Mal nicht warten
    if (m_erstesMal)
    {
      m_erstesMal = false;
      return;
    };
    
    try
    {
      if ( timeout < 0 )
        wait();
      else
        wait(timeout);
    }
    catch ( InterruptedException e )
    {
      System.out.println("SenderThread.mywait: InterruptedException Caught");
      //m_shutDown = true;
    }
    catch ( Exception e )
    {
      System.out.println("SenderThread.mywait: Exception Caught");
      System.out.println(e.getMessage());
      e.printStackTrace();
    }
    finally
    {
      System.out.println("ServerThread: Wartezyklus beendet");
    }
  } 

  //------------------------------------------------------------------
  // Interface Methoden
  //------------------------------------------------------------------

  /* (non-Javadoc)
   * @see stauma.dav.clientside.ClientSenderInterface#dataRequest(stauma.dav.configuration.interfaces.SystemObject, stauma.dav.clientside.DataDescription, byte)
   */
  public void dataRequest( Object object, Object dataDescription, byte state )
  {
  }
  
  /* (non-Javadoc)
   * @see stauma.dav.clientside.ClientSenderInterface#isRequestSupported(stauma.dav.configuration.interfaces.SystemObject, stauma.dav.clientside.DataDescription)
   */

  public boolean isRequestSupported( Object object, Object dataDescription )
  {
    return false;
  }
  //------------------------------------------------------------------

  /**
   * Testroutine fuer das lokale Modul.
   * Wird fuer verschiedene Tests im Laufe des Entwicklungs- und Wartungsprozesses verwendet
   * 
   * @param args Testargumente
   */
  public static void main( String[] args )
  {
    SenderThread senderThread = new SenderThread();
    if ( senderThread.m_dataAspZustandAlt == null )
      System.out.println("Ist Null");
    else
      senderThread.m_dataAspZustandAlt.aus();

    if ( senderThread.m_dataAspZustandAlt.equals(null) )
      System.out.println("gleich");
    else
      System.out.println("verschieden");
  }
}
