/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.dcf77.zeitverw;

import java.lang.reflect.Constructor;
import java.util.HashMap;

import de.bsvrz.sys.dcf77.lib.ArgumentParser;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Hauptmodul der SWE DCF77 (Zeitverwaltung).
 * Das Modul Zeitverwaltung erfragt zyklisch den Zustand des NTP Dienstes (Zeitserverdienstes)
 * und protokolliert Zustandänderungen an den DAV (Aspekt "Rechner.DcfStatus.Zustand").
 * Die Zeitverwaltung kommuniziert mit dem NTP Server Client ueber die Schnittstellle {@link IntpAllgemein}.
 *
 * @author J.Chriesten
 *
 */

public class Zeitverwaltung
{
  /**
   * Ausgabe Logger
   */
  private Debug _debug = Debug.getLogger();

  //---------------------------------------------------------------------------
  // Membervariablen
  //---------------------------------------------------------------------------

  /**
   * Zeitdauer zwischen 2 Zeitserverabfragen in sec
   */
  private long m_zyklusZeit;

  /**
   * Name der NTP Clientklasse. Sie muss IntpAllgemein implementieren.
   */
  //private static String m_ntpKlasse = "de.bsvrz.sys.dfctest.ntp.DefaultNtp";
  private String m_ntpKlasse;

  /**
   * Optionaler Parameterstring. Wird der IntpKlasse
   * uebergeben, falls diese einen <String> Konstruktor vorsieht
   */
  private String m_ntpKlassenParameter=null;

 /**
  * von DefaultNTP abgeleiteten Klasse.
  */
 private IntpAllgemein m_ntpAllgemein;

 /**
  * Liefert die ntp Client Instanz, die über das
  * Interface IntpAllgemein angesprochen wird.
  * @return ntp Client Instanz
  */
 public IntpAllgemein get_ntpAllgemein()
 {
   return m_ntpAllgemein;
 }

 /**
   * DCF DAV Kommunikator. Objekt besitzt Datenverteilerverbindung
   * und stellt Methoden zur Versendung und Empfang des Aspekts recher.dcf.zustand an.
   */
  private DCFdav m_dcfDAV;

  /**
   * Liefert die Instanz des IdcfTestReader.
   * siehe {@link IdcfTestReader}
   * @return Instanz des IdcfTestReader
   */
  public IdcfTestReader get_dcfTestReader()
  {
    return (IdcfTestReader) m_dcfDAV;
  }

  private IdcfTester m_dcfTester;

  /**
   * Liefert die Instanz des IdcfTester.
   * siehe {@link IdcfTester}.
   * @return Instanz des IdcfTester
   */
  public IdcfTester get_dcfTester()
  {
    return m_dcfTester;
  }

  /**
   * Thread zur Ueberwachung des NTP Zustandes
   */
  private SenderThread m_SenderThread;

  /**
   * Erzeugt aus dem uebergebenen Klassennamen ein Objekt vom Type IntpAllgemein
   * @param ntpKlasse Klassennamen. Die Klasse muss das Interface IntpAllgemein implementieren.
   * @return Generiertes Objekt vom Type IntpAllgemein
   */


  IntpAllgemein genNtpAllgemeinObject(String ntpKlasse)
  {
    IntpAllgemein ntpAllgemein = null;

    try
    {
      Object c;

      Class<?> clazz;
      clazz = (Class<?>)Class.forName(ntpKlasse);

      if ( m_ntpKlassenParameter != null )
      {
        try
        {
          // Konstruktor der Klasse mit String Parameter bestimmen
          Class<?>[] paramTypes = new Class<?>[] {String.class};
          Constructor<?> con=clazz.getConstructor(paramTypes);

          // Neue Instanz der Klasse mit diesem Konstruktor bilden
          c = con.newInstance(m_ntpKlassenParameter);
        }
        catch ( Exception e )
        {
          System.out.println("Fehler beim Instanziieren der Klasse " + ntpKlasse +
              "\n mit den Parametern: "+ m_ntpKlassenParameter);
          System.out.println(e.getMessage());
          e.printStackTrace();

          // Versuch mit parameterlosem Konstrukor
          Constructor con=clazz.getConstructor();
          c = con.newInstance((Object[])null);
        }
      }
      else
      {
        // Instanziierung mit parameterlosem Konstrukor
        Constructor con=clazz.getConstructor();
        c = con.newInstance((Object[])null);
      }

      // Umkopieren in Membervariable
      ntpAllgemein = (IntpAllgemein)c;
      m_ntpAllgemein = ntpAllgemein;

      if ( (c instanceof IdcfTester) )
      { // Das Interface ... muss nicht implementiert sein
        m_dcfTester = (IdcfTester)c;
      }
    }
    catch ( Exception e )
    {
      String errMess = "Fehler beim Instanzieren der Klasse " + ntpKlasse;
      errMess += "\n" + e.getMessage();
      e.printStackTrace();
      System.err.println(errMess);
      _debug.error( errMess );
      System.exit(2);
    }
    return ntpAllgemein;
  }

  /**
   * Auswertung der Parameter der Zeitverwaltung
   *
   * @param args Argumente siehe main Methode
   *
   */
  public void parseZeitverwArgumente( String[] args )
  {
    HashMap<String, String> schluesselWertPaare = ArgumentParser.parseArgStrings( args, "=" );

    m_ntpKlasse = ArgumentParser.getWert(schluesselWertPaare, "-ntpKlasse", true, false);
    System.out.println("main: -ntpKlasse= "+m_ntpKlasse);

    String zz = ArgumentParser.getWert(schluesselWertPaare, "-zyklusZeit", true, false);
    m_zyklusZeit = Long.parseLong( zz );
    System.out.println("main: -zyklusZeit= "+m_zyklusZeit);

    m_ntpKlassenParameter = ArgumentParser.getWert(schluesselWertPaare, "-ntpKlassenParameter", false, false);
    System.out.println("main: -ntpKlassenpars= "+m_ntpKlassenParameter);
  }


  /**
   * lokale Hilfsfunktion. Startet den Senderthread
   *
   * @param ntpAllgemein Schnittstellenobjekt
   * @param dcfDAV       Datenverteilerobjekt zum lesen/schreiben rechner.dcfStatus.zustand
   * @param zyklusZeit   Länge der Wartezyklen des Sendertrhread
   */
  private void startSenderThread(IntpAllgemein ntpAllgemein,
                                 DCFdav dcfDAV,
                                 long zyklusZeit)
  {
    // SenderThread starten, bei Auslösen einer Exception in dem Thread
    // (z.B. IllegalArgumentException) wird das ganze Programm beendet.
    // Dies wird über einen UncaughtExceptionHandler erreicht.
    Thread.UncaughtExceptionHandler eh = new Thread.UncaughtExceptionHandler ()
    {
      public void uncaughtException (Thread t, Throwable e)
      {
        e.printStackTrace();
        System.exit(5);
      }
    };

    m_SenderThread = new SenderThread(ntpAllgemein, dcfDAV, zyklusZeit, -1);
    m_SenderThread.setUncaughtExceptionHandler( eh );
    m_SenderThread.start();
  }


  /**
   * Hauptroutine zur Ausführung der Zeitverwaltung.
   *
   * Aufbau der Verbindung zum Datenverteiler und Anmeldung
   * Sender und Empfaenger für Aspekt Rechner.dcfStatus.Zustand
   *
   * Instanziierung der SS IntpAllgemein (== NTP Server)
   * Starten des SenderThreads
   *
   * @param args Programmargumente siehe main - Methode
   */
  public void init( String[] args )
  {
    // zeige Aufrufparameter
    System.out.println("--------------------------------------------");
    for (String arg : args)
    {
      System.out.println(arg);
    }
    System.out.println("--------------------------------------------");

    Zeitverwaltung zeitverwaltung = this;

    // Thread mit Verbindung zum Datenverteiler starten
    // Argumente fuer Datenverteiler extrahieren
    String DAVargs[] = new String[args.length];
    if ( args.length < 6)
    {
      String errMess = "Zeitverwaltung: zu wenig Parameter: ";
      _debug.error( errMess );
      System.exit(-1);
    }

    System.arraycopy( args, 0, DAVargs, 0,  DAVargs.length);
    DCFdav dcfDav = new DCFdav(DAVargs);

    zeitverwaltung.m_dcfDAV = dcfDav;

    // Quelle für den Aspekt rechner.dcfStatus.zustand anmelden
    dcfDav.subscribeSendDavZustand();

    // Die nicht vom Datenverteiler benoetigten Argumente lesen
    zeitverwaltung.parseZeitverwArgumente(args);

    // Die NTP Server Klasse erzeugen
    IntpAllgemein ntpAllgemein = zeitverwaltung.genNtpAllgemeinObject(zeitverwaltung.m_ntpKlasse);

    // und testen
    //DataAspZustand dataAspZustand = new DataAspZustand();
    DataAspZustand dataAspZustand = new DataAspZustand();
    dataAspZustand.aktuellerZeitserver = ntpAllgemein.getAktuellerZeitServer();
    dataAspZustand.vorhandeneZeitserver = ntpAllgemein.getVorhandeneZeitServer();
    dataAspZustand.letzteKorrekturZeit = ntpAllgemein.getZeitKorrektur();
    dataAspZustand.aus("vom NTP Server erzeugt");

    // SenderThread starten
    zeitverwaltung.startSenderThread(ntpAllgemein,
                                     zeitverwaltung.m_dcfDAV,
                                     zeitverwaltung.m_zyklusZeit);

    // Der NTP Dienst Klasse die Instanz von Zeitverwaltung übergeben, damit diese die
    // Methode setTrigger aufrufen kann.
    // Dies darf erst hier geschehen, da vorher der SenderThread noch nicht aktiv ist.
    ntpAllgemein.setInstance( (IntpTrigger) m_SenderThread );

    return;
  }

  /**
   * Reinitialisierung der Zeitverwaltung.
   *
   * Abmelden DAV Sender und - Empfaenger für Aspekt Rechner.dcfStatus.Zustand
   * Abbau der Verbindung zum Datenverteiler
   *
   * Aufruf  IntpAllgemein.beenden (stoppen realen NTPClient)
   * Stoppen des SenderThreads
   *
   */
  public void reinit()
  {
    this.m_SenderThread.set_shutDown();
    this.m_SenderThread.interrupt();

    this.m_dcfDAV.disconnect();

    return;
  }

  //------------------------------------------------------------------
  // Interface Methoden
  //------------------------------------------------------------------

  /* (non-Javadoc)
   * @see stauma.dav.clientside.ClientSenderInterface#dataRequest(stauma.dav.configuration.interfaces.SystemObject, stauma.dav.clientside.DataDescription, byte)
   */
  public void dataRequest( Object object, Object dataDescription, byte state )
  {
  }

  /* (non-Javadoc)
   * @see stauma.dav.clientside.ClientSenderInterface#isRequestSupported(stauma.dav.configuration.interfaces.SystemObject, stauma.dav.clientside.DataDescription)
   */

  public boolean isRequestSupported( Object object, Object dataDescription )
  {
    return false;
  }

  /* (non-Javadoc)
   * @see sys.funclib.application.StandardApplication#initialize(stauma.dav.clientside.ClientDavInterface)
   */
  public void initialize( Object davConnection ) throws Exception
  {
  }

  /* (non-Javadoc)
   * @see sys.funclib.application.StandardApplication#parseArguments(sys.funclib.ArgumentList)
   */
  public void parseArguments( Object argumentList ) throws Exception
  {
  }
  // -------------------------------------------------------------

  //------------------------------------------------------------------


  /**
   * startet das Modul Zeitverwaltung
   *
   * @param args Argumente. Werden im folgenden aufgeführt. Die in spitzer Klammer
   * uebrgebenen Werte dienen als Beispiel<br>
   * -datenverteiler=&lt;192.0.1.27:8083&gt;; DAV Server und Port<br>
   * -debugLevelStdErrText=&lt;FINE&gt;; DAV Errorlevel<br>
   * -benutzer=&lt;Tester&gt;; Benutzer des DAV<br>
   * -authentifizierung=&lt;C:/Datenverteiler/.../passwd&gt;; DAV Passwortdatei<br>
   * -rechner=&lt;rechner.Rechner1&gt;; optionale RechnerPid. Das DaV Objekt in das archiviert wird. Standardmäßig der lokale Rechner.<br>
   * -zyklusZeit=&lt;60&gt;; Zykluszeit SenderThread, -1: immer Warten<br>
   * -ntpKlasse=&lt;de.bsvrz.sys.dcf77.ntp.realclient.RealNTPClient&gt;; NTP Client, Klasse muss Interface<br>
   * IntpAllgemein realisieren
   * -ntpKlassenParameter=&lt;"cmd=ntpq -p ---zyklusZeit=2 ---maxZyklen=20&gt;"; optionaler Parameterstring
   * fuer -ntpKlasse. Die einzelnen Parameter sind durch --- getrennt<br>
   */
  public static void main( String[] args )
  {
    Zeitverwaltung zeitverwaltung = new Zeitverwaltung();
    zeitverwaltung.init( args );

    try
    {
      Thread.sleep( 10000 );
    }
    catch ( InterruptedException e )
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    //zeitverwaltung.reinit();
System.out.println("main beendet");
    
    return;
  }

}
