/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 * 
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * Contact Information:
 * Dambach-Werke GmbH
 * Elektronische Leitsysteme
 * Fritz-Minhardt-Str. 1
 * 76456 Kuppenheim
 * Phone: +49-7222-402-0
 * Fax: +49-7222-402-200
 * mailto: info@els.dambach.de
 */

package de.bsvrz.sys.funclib.konfig;

import java.util.List;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse für den Zugriff auf die Konfiguration.
 *
 * @version $Revision: 1.2 $ / $Date: 2009/10/26 13:44:16 $ / ($Author: Drapp $)
 *
 * @author Dambach-Werke GmbH
 * @author Thomas Drapp
 *
 */
public class KonfigZugriff
{
  /**
   * DebugLogger für Debug-Ausgaben.
   */
  private static Debug m_debug = Debug.getLogger();
  /**
   * Datenverteilerverbindung.
   */
  private ClientDavInterface m_connection;
  /**
   * Pid des ObjektTyps.
   */
  private String m_pidTyp = null;

  /**
   * Konstruktor mit Übergabe der Datenverteilerverbindung und der Pid des ObjektTyp.
   * Auf die Objekte der mit diesem Konstruktor erzeugten Instanz wird mit getObjects() zugegriffen.
   *
   * @param connection Datenverteilerverbindung.
   * @param pidTyp Pid des ObjektTyp.
   */
  public KonfigZugriff(ClientDavInterface connection, String pidTyp)
  {
    if (connection == null)
    {
      m_debug.error("Ungültige Datenverteilerverbindung übergeben.");

      throw new IllegalArgumentException("Ungültige Datenverteilerverbindung übergeben");
    }

    if (pidTyp == null)
    {
      m_debug.error("Ungültiger ObjektTyp übergeben (null).");

      throw new IllegalArgumentException("Ungültiger ObjektTyp übergeben (null)");
    }

    if (connection.getDataModel().getType(pidTyp) == null)
    {
      m_debug.error("Ungültiger ObjektTyp übergeben (kein ObjektTyp).");

      throw new IllegalArgumentException("Ungültiger ObjektTyp übergeben (kein ObjektTyp)");
    }

    m_connection = connection;
    m_pidTyp = pidTyp;
  }

  /**
   * Konstruktor mit Übergabe der Datenverteilerverbindung.
   * Auf die Objekte der mit diesem Konstruktor erzeugten Instanz wird mit getObjects(String) zugegriffen.
   *
   * @param connection Datenverteilerverbindung.
   */
  public KonfigZugriff(ClientDavInterface connection)
  {
    if (connection == null)
    {
      m_debug.error("Ungültige Datenverteilerverbindung übergeben.");

      throw new IllegalArgumentException("Ungültige Datenverteilerverbindung übergeben");
    }

    m_connection = connection;
  }

  /**
   * Holt die SystemObjekte des im Konstruktor angegebenen ObjektTyps.
   *
   * @return SystemObjekte des im Konstruktor angegebenen ObjektTyps oder <code>null</code>, wenn der andere Konstruktor verwendet wurde.
   */
  public List<SystemObject> getObjects()
  {
    if (m_pidTyp == null) // falscher Konstruktor verwendet, keine Pid des Typs übergeben
    {
      m_debug.error("Fehlerhafte Verwendung der Methode getObjects(), da im Konstruktor kein ObjektTyp angegeben wurde.");

      throw new IllegalArgumentException("Fehlerhafte Verwendung der Methode getObjects(), da im Konstruktor kein ObjektTyp angegeben wurde");
    }

    return m_connection.getDataModel().getType(m_pidTyp).getElements();
  }

  /**
   * Holt die SystemObjekte des übergebenen ObjektTyps.
   *
   * @param pidTyp Pid des ObjektTyp.
   * @return SystemObjekte des übergebenen ObjektTyps.
   */
  public List<SystemObject> getObjects(String pidTyp)
  {
    if (m_pidTyp != null) // falscher Konstruktor verwendet, da Pid des Typs schon übergeben
    {
      m_debug.error("Fehlerhafte Verwendung der Methode getObjects(String), da im Konstruktor schon ein ObjektTyp angegeben wurde.");

      throw new IllegalArgumentException("Fehlerhafte Verwendung der Methode getObjects(String), da im Konstruktor schon ein ObjektTyp angegeben wurde");
    }

    if (pidTyp == null)
    {
      m_debug.error("Ungültiger ObjektTyp übergeben (null).");

      throw new IllegalArgumentException("Ungültiger ObjektTyp übergeben (null)");
    }

    if (m_connection.getDataModel().getType(pidTyp) == null)
    {
      m_debug.error("Ungültiger ObjektTyp übergeben (kein ObjektTyp).");

      throw new IllegalArgumentException("Ungültiger ObjektTyp übergeben (kein ObjektTyp)");
    }

    return m_connection.getDataModel().getType(pidTyp).getElements();
  }
}
