/*
 * Copyright (c) 2016-2024 by inovat
 * innovative systeme - verkehr - tunnel - technik,
 * Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.inovat.dua.pufferlzzsnachfordern.VewDeLve255
 *
 * de.inovat.dua.pufferlzzsnachfordern.VewDeLve255 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.inovat.dua.pufferlzzsnachfordern.VewDeLve255 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.inovat.dua.pufferlzzsnachfordern.VewDeLve255.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Diese Datei ist Teil von de.inovat.dua.pufferlzzsnachfordern.VewDeLve255.
 *
 * de.inovat.dua.pufferlzzsnachfordern.VewDeLve255 ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder (nach Ihrer Wahl) jeder späteren
 * veröffentlichten Version, weiterverbreiten und/oder modifizieren.
 *
 * de.inovat.dua.pufferlzzsnachfordern.VewDeLve255 wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 *
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * An der Krautwiese 37
 * D-53783 Eitorf
 * +49 (0)2243 8464 193
 * info@inovat.de
 * www.inovat.de
 */

package de.inovat.dua.pufferlzzsnachfordern;

import static java.util.Calendar.HOUR_OF_DAY;
import static java.util.Calendar.MILLISECOND;
import static java.util.Calendar.MINUTE;
import static java.util.Calendar.SECOND;

import java.io.IOException;

//~ JDK IMPORTE ===============================================================

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.stream.Stream;

//~ NICHT JDK IMPORTE =========================================================
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.Dataset;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.archive.ArchiveData;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveDataQueryResult;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.dav.daf.main.archive.ArchiveQueryPriority;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestOption;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.DatasetReceiverInterface;
import de.bsvrz.dav.daf.main.archive.HistoryTypeParameter;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

//~ KLASSEN ===================================================================

/**
 * Verwaltungsklasse je DeLve mit DeNummer = 255, welches für genau ein Objekt
 * den Pufferabruf für einen Auftrag ausführt.
 * <p>
 * Das Verwaltungsobjekt fragt sowohl die Archivdaten ab, berechnet die ggf.
 * nachzufordernden Langzeitdaten und sendet die Pufferabfrage.
 * <p>
 * Weiterhin gibt die Verwaltungsklasse eine aussagekräftige Meldung zurück, die
 * den vollständigen Ablauf der Aggregation beschreibt.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kniß (HCK)
 */
public class VewDeLve255 implements ClientSenderInterface, DatasetReceiverInterface {

	/**
	 * DebugLogger für Debug-Ausgaben.
	 */
	private static final Debug debug = Debug.getLogger();
	private static final long EINE_MINUTE_IN_MILLISEKUNDEN = 60 * 1000;
	private static final long EINE_STUNDE_IN_MILLISEKUNDEN = EINE_MINUTE_IN_MILLISEKUNDEN * 60;
	private static final long EIN_TAG_IN_MILLISEKUNDEN = EINE_STUNDE_IN_MILLISEKUNDEN * 24;
	private static final long ZEHN_MINUTEN_IN_MILLISEKUNDEN = EINE_MINUTE_IN_MILLISEKUNDEN * 10;

	/**
	 * Archivsystem
	 */
	private static ArchiveRequestManager _archiv;
	private static Aspect _aspExterneErfassung;
	private static Aspect _aspTlsAbruf;
	private static AttributeGroup _atgFs;
	private static AttributeGroup _atgPufferAbrufLve;

	/**
	 * Datenverteilerverbindung
	 */
	private static ClientDavInterface _dav;

	/**
	 * Die Konfiguration des DaV.
	 */
	private static DataModel _konfiguration;

	/**
	 * Zeitraum in Tagen, für den die Datennachforderung erfolgen soll
	 */
	private static int _datenNachforderungInTagen;

	/**
	 * Minimale Wartezeit zwischen zwei Abrufversuchen der Pufferdaten
	 */
	private static int _minWarteZeitZwischenAbrufVersuchenInStunden;

	/**
	 * Maximale Anzahl der Abrufversuche, wenn Daten nicht verfügbar sind
	 */
	private static int _maxAnzahlAbrufVersuche;

	// ~ FELDER ================================================================

	/**
	 * Liste der Zeitpunkte der Datenverfügbarkeit der Verkehrsdaten
	 */
	private final Set<Long> _setVorhandeneDatensaetze = new ConcurrentSkipListSet<>();

	/**
	 * Intervallbeginn des letzten überwachten Pufferabrufs
	 */
	private long _letzterIntervallBeginn;

	/**
	 * Das Systemobjekt, für welches diese Verwaltungsklasse die Daten berechnet
	 */
	SystemObject _deLve255;

	/**
	 * Menge mit den FS, die dem LVE DE 255 zugeordnet sind (für die die
	 * Pufferabfrage durchgeführt werden muss).
	 */
	Set<SystemObject> _mengeFs;

	// ~ KONSTRUKTOREN (und vom Konstruktor verwendete Methoden) ==============

	/**
	 * Konstruktor der Verwaltungsklasse.
	 *
	 * @param dav                                         Die
	 *                                                    Datenverteilerverbindung
	 * @param deLve255                                    Das Systemobjekt, für
	 *                                                    welches diese
	 *                                                    Verwaltungsklasse die
	 *                                                    Daten berechnen soll
	 * @param mengeFs                                     Menge FS
	 * @param datenNachforderungInTagen                   Zeitraum in Tagen, für den
	 *                                                    die Langzeitdaten
	 *                                                    nachgefordert werden
	 *                                                    sollen
	 * @param minWarteZeitZwischenAbrufVersuchenInStunden Minimale Wartezeit in
	 *                                                    Stunden, die zwischen zwei
	 *                                                    Pufferabrufen einzuhalten,
	 *                                                    ist
	 * @param maxAnzahlAbrufVersuche                      Maximale Anzahl von
	 *                                                    Versuchen, fehlende
	 *                                                    Langzeitdaten abzurufen
	 */
	public VewDeLve255(ClientDavInterface dav, SystemObject deLve255, Set<SystemObject> mengeFs,
			int datenNachforderungInTagen, int minWarteZeitZwischenAbrufVersuchenInStunden,
			int maxAnzahlAbrufVersuche) {
		_dav = dav;
		_deLve255 = deLve255;
		_mengeFs = mengeFs;
		_datenNachforderungInTagen = datenNachforderungInTagen;
		_minWarteZeitZwischenAbrufVersuchenInStunden = minWarteZeitZwischenAbrufVersuchenInStunden;
		_maxAnzahlAbrufVersuche = maxAnzahlAbrufVersuche;
		_konfiguration = _dav.getDataModel();
		_archiv = _dav.getArchive();
		_atgFs = _konfiguration.getAttributeGroup("atg.verkehrsDatenLangZeitIntervall");
		_aspExterneErfassung = _konfiguration.getAspect("asp.externeErfassung");
		_atgPufferAbrufLve = _konfiguration.getAttributeGroup("atg.tlsLveAbrufPufferInhalt");
		_aspTlsAbruf = _konfiguration.getAspect("asp.tlsAbruf");

		// --------------------------------------------------------------------------
		// Sendeanmeldung für Zielattributgruppe (Pufferabfrage LVE der TLS)
		// --------------------------------------------------------------------------
		try {
			_dav.subscribeSender(this, _deLve255, new DataDescription(_atgPufferAbrufLve, _aspTlsAbruf),
					SenderRole.sender());
		} catch (OneSubscriptionPerSendData e) {
			debug.warning(String.format(
					"Doppelanmeldung für [%s] (anderer Sender vorhanden?). Pufferabruf wird für Objekt nicht weiter berücksichtigt.",
					_deLve255.getPid()), e);
		}

		// --------------------------------------------------------------------------
		// Archivanfrage für Zeitraum "datenNachforderungInTagen" und online-Anmeldung
		// für alle Fahrstreifen an diesem Verwaltungsobjekt
		// --------------------------------------------------------------------------
		DataDescription ddFS = new DataDescription(_atgFs, _aspExterneErfassung);
		long aktuelleZeit = System.currentTimeMillis();
		long vonZeitDatenNachforderung = ermittleIntervallBeginnAmTag(
				aktuelleZeit - (_datenNachforderungInTagen * EIN_TAG_IN_MILLISEKUNDEN));

		for (SystemObject soFs : mengeFs) {

			ArchiveTimeSpecification timeSpec = new ArchiveTimeSpecification(TimingType.ARCHIVE_TIME, false,
					vonZeitDatenNachforderung, aktuelleZeit);
			ArchiveDataSpecification arsSpec = new ArchiveDataSpecification(timeSpec, ArchiveDataKindCombination.all(),
					ArchiveOrder.BY_INDEX, ArchiveRequestOption.NORMAL, ddFS, soFs);
			ArchiveDataQueryResult queryResult = _archiv.request(ArchiveQueryPriority.MEDIUM, arsSpec);

			try {
				if (queryResult.isRequestSuccessful()) {
					Stream.of(queryResult.getStreams()).forEach(s -> {
						try {
							ArchiveData archiveData = s.take();
							while (archiveData != null) {
								if (DataState.DATA.equals(archiveData.getDataType())) {
									update(new Dataset[] { archiveData });
								}
								archiveData = s.take();
							}
						} catch (IllegalStateException | InterruptedException | IOException e) {
							debug.warning("Fehler beim Empfang von Archivdaten", e);
						}
					});
				}
			} catch (IllegalStateException | InterruptedException e) {
				debug.warning("Fehler beim Empfang von Archivdaten", e);
			}

			_archiv.subscribeReceiver(this, soFs, ddFS, ReceiveOptions.normal(), HistoryTypeParameter.TIME,
					aktuelleZeit - vonZeitDatenNachforderung);
		}

		// --------------------------------------------------------------------------
		// Auftragsliste für den Nachforderungszeitraum erzeugen
		// --------------------------------------------------------------------------
		long intervallBeginn;
		long ausfuehrungsZeitPunkt = ermittleZeitpunktMinute(aktuelleZeit) + ZEHN_MINUTEN_IN_MILLISEKUNDEN;

		int anzahlInialerAuftraege = 0;
		for (intervallBeginn = vonZeitDatenNachforderung; intervallBeginn < (aktuelleZeit
				- (_minWarteZeitZwischenAbrufVersuchenInStunden
						* EINE_STUNDE_IN_MILLISEKUNDEN)); intervallBeginn += EINE_STUNDE_IN_MILLISEKUNDEN) {

			if (!_setVorhandeneDatensaetze.contains(intervallBeginn)) {
				Auftrag auftrag = new Auftrag(ausfuehrungsZeitPunkt, intervallBeginn,
						intervallBeginn + EINE_STUNDE_IN_MILLISEKUNDEN, 1, this);
				AuftragsListe.getInstanz().addAuftrag(auftrag);
				anzahlInialerAuftraege++;
			}
			ausfuehrungsZeitPunkt += EINE_MINUTE_IN_MILLISEKUNDEN;
			_letzterIntervallBeginn = intervallBeginn;
		}

		debug.info(String.format("Initial %d Aufträge für [%s] erstellt.", anzahlInialerAuftraege, _deLve255.getPid()));
	}

	// ~ METHODEN ==============================================================

	/**
	 * Liste der Zeitpunkte mit vorhandenen Datensätzen aufräumen, dazu alle
	 * Einträge löschen, die älter sind als der nachzufordernde Zeitraum
	 */
	private void alteEintraegeLoeschen() {

		// Liste aufräumen, alle Einträge löschen, die älter sind als der
		// nachzufordernde Zeitraum (plus 1 Tag Reserve)
		long aeltesterZeitpunkt = System.currentTimeMillis() - (_datenNachforderungInTagen * EIN_TAG_IN_MILLISEKUNDEN)
				- EIN_TAG_IN_MILLISEKUNDEN;

		_setVorhandeneDatensaetze.removeIf(zeitpunkt -> zeitpunkt < aeltesterZeitpunkt);
	}

	/**
	 * Methode, die den eigentlichen Auftrag ausführt.
	 *
	 * @param auftrag Der auszuführende Auftrag.
	 */
	public void ausfuehren(Auftrag auftrag) {

		// --------------------------------------------------------------------------
		// Prüfen, ob Daten bereits vorliegen, wenn nein, dann Auftrag ausführen
		// und Folgeauftrag erstellen, sonst nichts tun
		// --------------------------------------------------------------------------
		// ToDo: NUR in der Testphase verwendet:
		// debug.fine(ausgabeVorhandeneDatenZeitPunkte());

		updateVorhandeneDatensaetze(auftrag);

		if (!_setVorhandeneDatensaetze.contains(auftrag.getIntervallBeginn())
				&& auftrag.getAktAnzahlAbrufVersuche() < _maxAnzahlAbrufVersuche) {
			DataDescription datenBeschreibung = new DataDescription(_atgPufferAbrufLve, _aspTlsAbruf);
			Data datenPufferAbruf = _dav.createData(_atgPufferAbrufLve);

			// Standardpufferanfrage für das aktuelle Intervall abfragen
			datenPufferAbruf.getTimeValue("Zeitpunkt").setMillis(auftrag.getIntervallBeginn());
			datenPufferAbruf.getUnscaledValue("Anzahl").set(1);

			try {
				_dav.sendData(new ResultData(_deLve255, datenBeschreibung, System.currentTimeMillis(), datenPufferAbruf,
						false));
			} catch (SendSubscriptionNotConfirmed e) {
				debug.warning(String.format(
						"Sendeversuch für Pufferabfrage wegen fehlender Sendeanmeldung fehlgeschlagen: [%s]",
						datenBeschreibung), e);
			}

			// Folgeauftrag erstellen
			long ausfuehrungsZeitPunkt = auftrag.getAusfuehrungsZeitPunkt()
					+ _minWarteZeitZwischenAbrufVersuchenInStunden * EINE_STUNDE_IN_MILLISEKUNDEN;
			Auftrag folgeAuftrag = new Auftrag(ausfuehrungsZeitPunkt, auftrag.getIntervallBeginn(),
					auftrag.getIntervallEnde(), auftrag.getAktAnzahlAbrufVersuche() + 1, auftrag.getVewDeLve255());

			AuftragsListe.getInstanz().addAuftrag(folgeAuftrag);
			debug.info(String.format("Ausgeführter Auftrag: %s%nNachfolge Auftrag   : %s", auftrag, folgeAuftrag));
		}
	}

	/**
	 * Abfrage der vorhandenen Daten aus dem Archiv für den Zeitraum des Auftrags.
	 */
	private void updateVorhandeneDatensaetze(Auftrag auftrag) {
		DataDescription ddFS = new DataDescription(_atgFs, _aspExterneErfassung);

		for (SystemObject soFs : _mengeFs) {

			ArchiveTimeSpecification timeSpec = new ArchiveTimeSpecification(TimingType.ARCHIVE_TIME, false,
					auftrag.getIntervallBeginn(), auftrag.getIntervallEnde());
			ArchiveDataSpecification arsSpec = new ArchiveDataSpecification(timeSpec, ArchiveDataKindCombination.all(),
					ArchiveOrder.BY_INDEX, ArchiveRequestOption.NORMAL, ddFS, soFs);
			ArchiveDataQueryResult queryResult = _archiv.request(ArchiveQueryPriority.MEDIUM, arsSpec);

			try {
				if (queryResult.isRequestSuccessful()) {
					Stream.of(queryResult.getStreams()).forEach(s -> {
						try {
							ArchiveData archiveData = s.take();
							while (archiveData != null) {
								if (DataState.DATA.equals(archiveData.getDataType())) {
									update(new Dataset[] { archiveData });
								}
								archiveData = s.take();
							}
						} catch (IllegalStateException | InterruptedException | IOException e) {
							debug.warning("Fehler beim Empfang von Archivdaten", e);
						}
					});
				}
			} catch (IllegalStateException | InterruptedException e) {
				debug.warning("Fehler beim Empfang von Archivdaten", e);
			}
		}
	}

	/**
	 * Gibt die sortierte Liste aller Datenzeitpunkte zurück, für die Langzeitdaten
	 * vorliegen.
	 *
	 * @return sortierte Liste aller Datenzeitpunkte zurück, für die Langzeitdaten
	 *         vorliegen.
	 */
	@Deprecated(forRemoval = true)
	private String ausgabeVorhandeneDatenZeitPunkte() {

		// Inhalt der Liste zu Debugzwecken ausgeben
		SimpleDateFormat datumsFormatierer = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss,SSS");
		StringBuilder sb = new StringBuilder();

		sb.append(this).append(String.format(":%n"));

		for (long zeitpunkt : _setVorhandeneDatensaetze) {
			sb.append(datumsFormatierer.format(new Date(zeitpunkt))).append(String.format("%n"));
		}

		return sb.toString();
	}

	/**
	 * Sendesteuerung des Datenverteilers an die Applikation. Diese Methode muss von
	 * der Applikation implementiert werden, um den Versand von Daten zu starten
	 * bzw. anzuhalten. Der Datenverteiler signalisiert damit einer Quelle oder
	 * einem Sender, dass mindestens ein Abnehmer bzw. kein Abnehmer mehr für die
	 * zuvor angemeldeten Daten vorhanden ist. Die Quelle wird damit aufgefordert
	 * den Versand von Daten zu starten bzw. zu stoppen.
	 *
	 * @param object          Das in der zugehörigen Sendeanmeldung angegebene
	 *                        Objekt, auf das sich die Sendesteuerung bezieht.
	 * @param dataDescription Beschreibende Informationen zu den angemeldeten Daten
	 *                        auf die sich die Sendesteuerung bezieht.
	 * @param state           Status der Sendesteuerung. Kann einen der Werte
	 *                        <code>START_SENDING</code>, <code>STOP_SENDING</code>,
	 *                        <code>STOP_SENDING_NO_RIGHTS</code>,
	 *                        <code>STOP_SENDING_NOT_A_VALID_SUBSCRIPTION</code>
	 *                        enthalten.
	 * @see #START_SENDING
	 * @see #STOP_SENDING
	 * @see #STOP_SENDING_NO_RIGHTS
	 * @see #STOP_SENDING_NOT_A_VALID_SUBSCRIPTION
	 */
	@Override
	public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {

		// Wird nicht unterstützt. Ergebnisdaten werden immer gesendet.
	}

	/**
	 * Ermittelt den IntervallBeginn ausgehend von einem Zeitpunkt für 0:00 Uhr an
	 * diesem Tag.
	 *
	 * @param zeitPunkt Anfangszeitpunkt in ms seit 1.1.1970, ab dem spätestens die
	 *                  Daten berechnet werden sollen. Daraus wird dann als
	 *                  Intervallbeginn 0:00 Uhr an diesem Tag ermittelt
	 * @return Intervallbeginn (GENAU!) 0:00 Uhr an dem Tag des übergebenen
	 *         Zeitpunktes <code>zeitPunkt</code>
	 */
	private long ermittleIntervallBeginnAmTag(long zeitPunkt) {
		Calendar vonZeitPunkt = new GregorianCalendar();

		// vonZeitPunkt darf spätestens sein...
		vonZeitPunkt.setTimeInMillis(zeitPunkt);

		// Sekunden und Millisekunden auf 0 gesetzt werden.
		vonZeitPunkt.set(MILLISECOND, 0);
		vonZeitPunkt.set(SECOND, 0);
		vonZeitPunkt.set(MINUTE, 0);
		vonZeitPunkt.set(HOUR_OF_DAY, 0);

		return vonZeitPunkt.getTimeInMillis();
	}

	/**
	 * Ermittelt den Zeitpunkt am Anfang der Minute ausgehend von einem beliebigen
	 * Zeitpunkt.
	 *
	 * @param zeitPunkt Zeitpunkt in ms seit 1.1.1970
	 * @return Zeitpunkt (GENAU!) am Anfang der Minute des übergebenen Zeitpunktes
	 *         <code>zeitPunkt</code>
	 */
	private long ermittleZeitpunktMinute(long zeitPunkt) {
		Calendar vonZeitPunkt = new GregorianCalendar();

		// vonZeitPunkt darf spätestens sein...
		vonZeitPunkt.setTimeInMillis(zeitPunkt);

		// Sekunden und Millisekunden auf 0 gesetzt werden.
		vonZeitPunkt.set(MILLISECOND, 0);
		vonZeitPunkt.set(SECOND, 0);

		return vonZeitPunkt.getTimeInMillis();
	}

	// ~ GET METHODEN ==========================================================

	/**
	 * Liefert den Intervallbeginn des letzten überwachten Pufferabrufs
	 *
	 * @return Intervallbeginn des letzten überwachten Pufferabrufs
	 */
	public long getLetzterIntervallBeginn() {
		return _letzterIntervallBeginn;
	}

	/**
	 * Liefert die Pid des dieser Verwaltungseinheit zugeordnete LZZS Systemobjekt.
	 *
	 * @return Pid des dieser Verwaltungseinheit zugeordnete LZZS Systemobjekt.
	 */
	public String getPid() {
		return _deLve255.getPid();
	}

	/**
	 * Diese Methode muss von der Applikation implementiert werden, um zu
	 * signalisieren, ob Sendesteuerungen erwünscht sind und mit der Methode
	 * <code>dataRequest</code> verarbeitet werden. In der Implementierung dieser
	 * Methode dürfen keine synchronen Aufrufe, die auf Telegramme vom
	 * Datenverteiler warten (wie z.B. Konfigurationsanfragen) durchgeführt werden,
	 * da ansonsten ein Deadlock entsteht.
	 *
	 * @param object          Das in der zugehörigen Sendeanmeldung angegebene
	 *                        System-Objekt.
	 * @param dataDescription Die in der zugehörigen Sendeanmeldung angegebenen
	 *                        beschreibenden Informationen der angemeldeten Daten.
	 * @return <code>true</code>, falls Sendesteuerungen gewünscht sind, sonst
	 *         <code>false</code>.
	 * @see #dataRequest
	 */
	@Override
	public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
		return false;
	}

	// ~ SET METHODEN ==========================================================

	/**
	 * Setzt den Intervallbeginn des letzten überwachten Pufferabrufs
	 *
	 * @param letzterIntervallBeginn Intervallbeginn des letzten überwachten
	 *                               Pufferabrufs
	 */
	public void setLetzterIntervallBeginn(long letzterIntervallBeginn) {
		_letzterIntervallBeginn = letzterIntervallBeginn;
	}

	// ~ METHODEN ==============================================================

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		for (SystemObject so : _mengeFs) {
			sb.append(so.getPid()).append("  ");
		}

		return sb.toString();
	}

	/**
	 * Aktualisierungsmethode, die nach Empfang eines angemeldeten Datensatzes von
	 * den Datenverteiler-Applikationsfunktionen aufgerufen wird. Diese Methode muss
	 * von der Applikation zur Verarbeitung der empfangenen Datensätze implementiert
	 * werden.
	 * <p>
	 * Für die empfangenen Datensätze aus Archivabfrage und Onlinedaten wird sich
	 * einfach der Zeitstempel jedes Datensatzes gemerkt. Für jeden Zeitstempel, der
	 * vorhanden ist, liegen somit Langzeitdaten vor und ein entsprechender
	 * Pufferabruf wird für dieses Intervall dann nicht ausgeführt.
	 *
	 * @param datasetResults Feld mit den empfangenen Ergebnisdatensätzen
	 */
	@Override
	public void update(Dataset[] datasetResults) {
		for (Dataset ergebnisDaten : datasetResults) {
			_setVorhandeneDatensaetze.add(ergebnisDaten.getDataTime());
		}
		alteEintraegeLoeschen();
	}
}
