/*
 * Copyright 2006 by inovat, Dipl.-Ing. H. C. Kni
 * ALL RIGHTS RESERVED.
 *
 * THIS SOFTWARE IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A PARTICULAR  PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL inovat OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES  (INCL., BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR  PROFITS;
 * OR BUSINESS INTERRUPTION)  HOWEVER  CAUSED  AND  ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,  OR TORT (INCL.
 * NEGLIGENCE OR OTHERWISE)  ARISING  IN  ANY  WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package de.bsvrz.kex.isis.isis;

/**
 * Legt das Interface zu einem Baustellenobjekt des BaustellenInformationsSystem
 * (BIS) fest. &Uuml;ber eine Klasse, die dieses Interface implementiert,
 * k&ouml;nnen auf die Baustellen des an die AK-VRZ Software angebunden
 * Baustelleninformationssystems zugegriffen werden.
 * 
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kni (HCK)
 * @version $Revision: 129 $ / $Date: 2006-12-15 23:24:31 +0100 (Fr, 15 Dez
 *          2006) $ / ($Author: HCK $)
 */

public interface BaustelleInterface {
	// ------------------------------------------------------------------------
	// SONSTIGE METHODEN
	// ------------------------------------------------------------------------

	/**
	 * Setzt (&auml;ndert) den G&uuml;ltigkeitszeitraum eines Baustellenobjekts
	 * auf Seiten der VRZ.
	 * 
	 * @param vonZeitpunkt
	 *            Zeitpunkt des Baustellenbeginns (in ms seit dem 1.1.1970 UTC).
	 * @param dauer
	 *            Dauer der Baustelle (in ms).
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	void aendereBaustellenGueltigkeitsZeitraum(long vonZeitpunkt, long dauer)
			throws BisInterfaceException;

	/**
	 * Setzt (&auml;ndert) das StraenSegment und die Richtung eines
	 * Baustellenobjekts auf Seiten der VRZ.
	 * 
	 * @param ortsReferenz
	 *            Ortsreferenzobjekt (StraenSegments mit Offset), auf der die
	 *            Baustelle beginnt.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	void aendereBaustellenOrt(
			OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenz)
			throws BisInterfaceException;

	/**
	 * Setzt (&auml;ndert) den Startoffset und die L&auml;nge eines
	 * Baustellenobjekts auf Seiten der VRZ.
	 * 
	 * @param startOffset
	 *            Positon des Baustellenanfangs auf der Strae.
	 * @param laenge
	 *            L&auml;nge des Baustelle stromabw&auml;rts.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	public void aendereBaustellenStartOffsetUndLaenge(long startOffset,
			long laenge) throws BisInterfaceException;

	/**
	 * Setzt (&auml;ndert) die Angaben zum Baustellenverantwortlichen eines
	 * Baustellenobjekts ({@link
	 * de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface}) .
	 * 
	 * @param baustellenVerantwortlicher
	 *            Der Baustellenverantwortliche.
	 * @throws BisInterfaceException
	 */
	public void aendereBaustellenVerantwortlichen(
			BaustellenVerantwortlicherInterface baustellenVerantwortlicher)
			throws BisInterfaceException;

	/**
	 * Setzt (&auml;ndert) den Zustand eines Baustellenobjekts auf Seiten der
	 * VRZ.
	 * 
	 * @param baustellenZustand
	 *            Neuer Zustand der Baustelle.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	void aendereBaustellenZustand(
			BisInterface.BaustellenZustand baustellenZustand)
			throws BisInterfaceException;

	/**
	 * Setzt (&auml;ndert) den Engpasskapazit&auml;t eines Baustellenobjekts auf
	 * Seiten der VRZ.
	 * 
	 * @param engpassKapazitaet
	 *            Restkapazit&auml;t w&auml;hrend der G&uuml;ltigkeitsdauer der
	 *            Baustelle.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	void aendereEngpassKapazitaet(int engpassKapazitaet)
			throws BisInterfaceException;

	/**
	 * Setzt/&auml;ndert den Informationstext zur Baustelle.
	 * 
	 * @param infoText
	 *            Informationstext zur Baustelle.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	void aendereInfo(String infoText) throws BisInterfaceException;

	/**
	 * Liefert Angaben zum Baustellenverantwortlichen eines Baustellenobjekts ({@link
	 * de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface}).
	 * 
	 * @return Angaben zum Baustellenverantwortlichen eines Baustellenobjekts.
	 */
	BaustellenVerantwortlicherInterface getBaustellenVerantwortlichen();

	/**
	 * Liefert den Zustand der Baustelle ({@link BisInterface.BaustellenZustand}).
	 * 
	 * @return Zustand der Baustelle.
	 */
	BisInterface.BaustellenZustand getBaustellenZustand();

	/**
	 * Liefert die Dauer der Baustelle (in ms).
	 * 
	 * @return Dauer der Baustelle (in ms).
	 */
	long getDauer();

	/**
	 * Liefert die Restkapazit&auml;t w&auml;hrend der G&uuml;ltigkeitsdauer der
	 * Baustelle in Fz/h.
	 * 
	 * @return Restkapazit&auml;t w&auml;hrend der G&uuml;ltigkeitsdauer der
	 *         Baustelle in Fz/h.
	 */
	int getEngpassKapazitaet();

	/**
	 * Liefert die Id der Baustelle.
	 * 
	 * @return Id der Baustelle.
	 */
	long getId();

	/**
	 * Liefert den Informationstext zur Baustelle.
	 * 
	 * @return Informationstext zur Baustelle.
	 */
	String getInfo();

	/**
	 * Liefert die L&auml;nge der Baustelle stromabw&auml;rts in Metern.<br>
	 * Es wird die L&auml;nge der Baustelle in der VRZ zur&uuml;ckgegeben, d.h.
	 * wenn die L&auml;nge beim Anlegen der Baustelle korrigiert werden musste
	 * (siehe {@link BisInterface#erzeugeBaustelle}), wird hier die korrigierte
	 * L&auml;nge geliefert
	 * 
	 * @return L&auml;nge der Baustelle stromabw&auml;rts in Metern.
	 */
	long getLaenge();

	/**
	 * Liefert den Namen der Baustelle.
	 * 
	 * @return Name der Baustelle.
	 */
	String getName();

	/**
	 * Liefert das Ortsreferenzobjekt (StraenSegments mit Offset), auf das die
	 * Baustelle beginnt.
	 * 
	 * @return Ortsreferenzobjekt (StraenSegments mit Offset), auf der die
	 *         Baustelle beginnt.
	 */
	OrtsReferenzStrassenSegmentUndOffsetInterface getOrtsReferenzStrassenSegmentUndOffset();

	/**
	 * Liefert die Pid der Baustelle (weltweit eindeutige PermanenteID).
	 * 
	 * @return Pid der Baustelle.
	 */
	String getPid();

	/**
	 * Liefert den Zeitpunkt des Baustellenbeginns (in ms seit dem 1.1.1970
	 * UTC).
	 * 
	 * @return Zeitpunkt des Baustellenbeginns (in ms seit dem 1.1.1970 UTC).
	 */
	long getStartZeitpunkt();

	/**
	 * Startet f&uuml;r die angegebene Baustelle die Baustellenprognose und
	 * liefert das Ergebnis der Prognose.
	 * 
	 * @param veranlasser
	 *            Veranlasser, der die Simulation angestoen hat.
	 * @param ursache
	 *            Ursache (Bemerkung) zu diesem Prognosestart.
	 * 
	 * @return BaustellenPrognoseObjekt ({@link de.bsvrz.kex.isis.isis.BaustellenPrognoseInterface})
	 *         mit den Ergebnissen der Baustellenprognose.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	BaustellenPrognoseInterface startePrognose(String veranlasser,
			String ursache) throws BisInterfaceException;
}
