/*
 * Copyright 2006 by inovat, Dipl.-Ing. H. C. Kni
 * ALL RIGHTS RESERVED.
 *
 * THIS SOFTWARE IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A PARTICULAR  PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL inovat OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES  (INCL., BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR  PROFITS;
 * OR BUSINESS INTERRUPTION)  HOWEVER  CAUSED  AND  ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,  OR TORT (INCL.
 * NEGLIGENCE OR OTHERWISE)  ARISING  IN  ANY  WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package de.bsvrz.kex.isis.isis;

import de.bsvrz.sys.funclib.bitctrl.modell.netz.NetzInterface;

/**
 * Legt das Interface zu einem BaustellenInformationsSystem (BIS) fest.
 * &Uuml;ber eine Klasse, die dieses Interface implementiert, k&ouml;nnen
 * bestehende Baustelleninformationssysteme an die AK-VRZ Software angebunden
 * werden.
 * 
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Hans Christian Kni (HCK)
 * @version $Revision: 144 $ / $Date: 2006-12-14 01:04:03 +0100 (Do, 14 Dez
 *          2006) $ / ($Author: HCK $)
 */

public interface BisInterface extends NetzInterface {
	// ------------------------------------------------------------------------
	// SONSTIGE METHODEN
	// ------------------------------------------------------------------------


	/**
	 * Aufz&auml;hlungstyp f&uuml;r die Stationierungsrichtung auf einem ASB
	 * Abschnitt.
	 */
//	public enum ASBStationierungsRichtung {
//		/**
//		 * ASB Abschnitt wird in Stationierungsrichtung betrachtet.
//		 */
//		IN_STATIONIERUNGSRICHTUNG,
//
//		/**
//		 * ASB Abschnitt wird entgegen der Stationierungsrichtung betrachtet.
//		 */
//		GEGEN_STATIONIERUNGSRICHTUNG
//	}

	/**
	 * Aufz&auml;hlungstyp f&uuml;r die m&ouml;glichen Zust&auml;nde einer
	 * Baustelle (siehe Datenkatalog Attributtyp: att.statusBaustelle).
	 */
	public enum BaustellenZustand {
		/**
		 * Baustelle ist im Entwurfsstadium.
		 */
		ENTWORFEN,
		/**
		 * Baustelle ist geplant.
		 */
		GEPLANT,
		/**
		 * Baustelle ist g&uuml;ltig.
		 */
		GUELTIG,
		/**
		 * Baustelle wurde storniert.
		 */
		STORNIERT
	}

	/**
	 * Aufz&auml;hlungstyp f&uuml;r den Kommunikationsstatus des BisInterface.
	 */
	public enum BisKommunikationsStatus {
		/**
		 * Kommunikationsstatus ist ok, BisInterface kann verwendet werden.
		 */
		BIS_KOMMUNIKATION_OK,

		/**
		 * Kommunikationsstatus ist fehlerhaft, BisInterface kann nicht werden.
		 */
		BIS_KOMMUNIKATION_FEHLER
	}

	/**
	 * Aufz&auml;hlungstyp f&uuml;r die Fahrtrichtung auf einer Stra&szlig;e.
	 */
//	public enum FahrtRichtung {
//		/**
//		 * Stra&szlig;e wird wie angegeben durchfahren.
//		 */
//		IN_RICHTUNG,
//
//		/**
//		 * Stra&szlig;e wird entgegen der Angaben durchfahren.
//		 */
//		GEGEN_RICHTUNG
//	}

	/**
	 * Registriert einen Listener (Beobachterklasse), der &uuml;ber den
	 * aktuellen Status der Kommunikation des BIS-Interfaces informiert werden
	 * soll.
	 * 
	 * @param listener
	 *            Beaobachterklasse, die das
	 *            {@link BisKommunikationsStatusListener} Interface implemtiert
	 *            hat.
	 */
	void addKommunikationsStatusListener(
			BisKommunikationsStatusListener listener);

	/**
	 * Erzeugt ein neues Baustellenobjekt (typ.baustelle) auf Seiten der VRZ.
	 * 
	 * @param pid
	 *            Pid der zu erzeugenden Baustelle (weltweit eindeutige
	 *            PermanenteID) oder <tt>null</tt>, falls die PID automatisch
	 *            vergeben werden soll.
	 * @param name
	 *            Name der Baustelle oder <tt>null</tt>, falls der Name
	 *            automatisch vergeben werden soll.
	 * @param info
	 *            Informationstext zur Baustelle oder <tt>null</tt>, falls
	 *            der Infotext automatisch vergeben werden soll.
	 * @param vonZeitpunkt
	 *            Zeitpunkt des Baustellenbeginns (in ms seit dem 1.1.1970 UTC).
	 * @param dauer
	 *            Dauer der Baustelle (in ms).
	 * @param ortsReferenz
	 *            Ortsreferenzobjekt (Stra&szlig;enSegments mit Offset), auf der
	 *            die Baustelle beginnt.
	 * @param laenge
	 *            L&auml;nge des Baustelle stromabw&auml;rts in Metern.
	 * @param baustellenZustand
	 *            Zustand der Baustelle.
	 * @param engpassKapazitaet
	 *            Restkapazit&auml;t w&auml;hrend der G&uuml;ltigkeitsdauer der
	 *            Baustelle in Fz/h.
	 * @param baustellenVerantwortlicher
	 *            Angaben zum Baustellenverantwortlichen eines
	 *            Baustellenobjekts.
	 * 
	 * @return Die Id des erzeugten Baustellenobjekts oder -1, wenn kein
	 *         Baustellenobjekt angelegt werden konnte.
	 * @throws BisInterfaceException
	 *             wenn die Baustelle nicht angelegt werden kann
	 */
	long erzeugeBaustelle(String pid, String name, String info,
			long vonZeitpunkt, long dauer,
			OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenz,
			long laenge, BaustellenZustand baustellenZustand,
			int engpassKapazitaet,
			BaustellenVerantwortlicherInterface baustellenVerantwortlicher)
			throws BisInterfaceException;

	/**
	 * Erzeugt einen Baustellenverantwortlichen, der einer Baustelle zugeordnet
	 * wird.
	 * 
	 * @param firma
	 *            Name der verantwortlichen Firma.
	 * @param nameBaustellenVerantwortlicher
	 *            Name des Baustellenverantwortlichen.
	 * @param telefonBaustellenVerantwortlicher
	 *            Telefonnummer des Baustellenverantwortlichen.
	 * @param telefonFirma
	 *            Telefonnummer der verantwortlichen Firma.
	 * @param telefonMobilBaustellenVerantwortlicher
	 *            Handynummer des Baustellenverantwortlichen.
	 * 
	 * @return Der Baustellenverantwortliche.
	 */
	BaustellenVerantwortlicherInterface erzeugeBaustellenVerantwortlichen(
			String firma, String nameBaustellenVerantwortlicher,
			String telefonBaustellenVerantwortlicher, String telefonFirma,
			String telefonMobilBaustellenVerantwortlicher);

	/**
	 * Erstellt eine Ortsreferenz, bei dem die Ortsangabe &uuml;ber eine
	 * Stra&szlig;e und den Betriebskilometer dargestellt wird.
	 * 
	 * @param strasse
	 *            Stra&szlig;e, auf der die OrtReferenz liegt.
	 * @param fahrtRichtung
	 *            Fahrtrichtung der Stra&szlig;e, auf der die Ortsreferenz
	 *            liegt.
	 * @param betriebsKilometer
	 *            Betriebskilometer auf der Stra&szlig;e, auf der die
	 *            Ortsreferenz liegt.
	 * @param blockNummer
	 *            Blocknummer des Betriebskilometers.
	 * 
	 * @return Ortsreferenz, bei dem die Ortsangabe &uuml;ber eine Stra&szlig;e
	 *         und den Betriebskilometer dargestellt wird.
	 * @throws BisInterfaceException
	 *             wenn keine Abbildung m&ouml;glich ist.
	 * 
	 */
	OrtsReferenzStrasseUndBetriebsKilometerInterface erzeugeOrtsReferenz(
			StrasseInterface strasse, BisInterface.FahrtRichtung fahrtRichtung,
			long betriebsKilometer, int blockNummer)
			throws BisInterfaceException;

	/**
	 * Erstellt ein OrtsreferenzObjekt, bei dem die Ortsangabe &uuml;ber ein
	 * Stra&szlig;enSegment und den Offset vom Anfang des Stra&szlig;enSegments
	 * dargestellt wird.
	 * 
	 * @param pidStrassenSegment
	 *            Pid des Stra&szlig;enSegments gem&auml;&szlig; Konfiguration
	 * @param startOffset
	 *            Offset vom Anfang des Stra&szlig;enSegments
	 * 
	 * @return Ortsreferenz, bei dem die Ortsangabe &uuml;ber ein
	 *         Stra&szlig;enSegment und den Offset vom Anfang des
	 *         Stra&szlig;enSegments dargestellt wird.
	 * 
	 * @throws BisInterfaceException
	 *             wenn keine Abbildung m&ouml;glich ist.
	 */
	OrtsReferenzStrassenSegmentUndOffsetInterface erzeugeOrtsReferenz(
			String pidStrassenSegment, long startOffset)
			throws BisInterfaceException;

	/**
	 * Erstellt eine Ortsreferenzen, bei dem die Ortsangabe &uuml;ber das
	 * ASB-Stationierungssystem angegeben wird (Anfangs- und Endpunkt
	 * (Netzknoten bzw. Nullpunkt), Stationierung im Wertebereich von 0 bis
	 * zugewiesene L&auml;nge sowie Angabe der Richtung)
	 * 
	 * @param anfangsKnoten
	 *            Anfangsknoten der ASB Stationierung. Anfangsknoten mit
	 *            eindeutiger Kennung (Anfangsnullpunkt) des Teilabschnittes
	 *            oder Astes. Die Nullpunktbezeichung hat die Form TTTTnnnB,
	 *            wobei TTTT die vierstellige TK25-Blattnummer und nnn die
	 *            dreistellige laufende Nummer ist, die zusammen die bundesweit
	 *            eindeutige Netzknotennummer darstellen. Durch die Kennung B
	 *            (ein Zeichen) wird zus&auml;tzlich der Nullpunkt des
	 *            Abschnitts oder Astes eindeutig festgelegt. Eine nicht
	 *            vorhandene Kennung wird als o(hne) eingetragen.
	 * @param endKnoten
	 *            Endknoten der ASB Stationierung. Endknoten mit eindeutiger
	 *            Kennung (Endnullpunkt) des Teilabschnittes oder Astes. Die
	 *            Nullpunktbezeichung hat die Form TTTTnnnB, wobei TTTT die
	 *            vierstellige TK25-Blattnummer und nnn die dreistellige
	 *            laufende Nummer ist, die zusammen die bundesweit eindeutige
	 *            Netzknotennummer darstellen. Durch die Kennung B (ein Zeichen)
	 *            wird zus&auml;tzlich der Nullpunkt des Abschnitts oder Astes
	 *            eindeutig festgelegt. Eine nicht vorhandene Kennung wird als
	 *            o(hne) eingetragen.
	 * @param stationierungsRichtung
	 *            Stationierungsrichtung f&uuml;r den ASB Abschnitt.
	 * @param stationierung
	 *            Stationierung (in Metern) auf dem ASB Abschnitt.
	 * 
	 * @return Ortsreferenz, bei dem die Ortsangabe &uuml;ber das
	 *         ASB-Stationierungssystem angegeben wird (Anfangs- und Endpunkt
	 *         (Netzknoten bzw. Nullpunkt) sowie der Stationierung im
	 *         Wertebereich von 0 bis zugewiesene L&auml;nge sowie der
	 *         Richtung).
	 * 
	 * @throws BisInterfaceException
	 *             wenn keine Abbildung m&ouml;glich ist.
	 */
	OrtsReferenzAsbStationierungInterface erzeugeOrtsReferenz(
			String anfangsKnoten, String endKnoten,
			ASBStationierungsRichtung stationierungsRichtung, long stationierung)
			throws BisInterfaceException;

	/**
	 * Liefert das Baustellenobjekt mit der zugeh&ouml;rigen ID.
	 * 
	 * @param baustellenId
	 *            Id der Baustelle, die bei {@link #erzeugeBaustelle}
	 *            zur&uuml;ckgegeben wurde.
	 * 
	 * @return Baustellenobjekt ({@link BaustelleInterface}), &uuml;ber die
	 *         Baustelleneigenschaften abgerufen werden k&ouml;nnen.
	 */
	BaustelleInterface getBaustelle(long baustellenId);

	// ------------------------------------------------------------------------
	// INNERE KLASSEN
	// ------------------------------------------------------------------------

	/**
	 * Liefert ein Stra&szlig;enobjekt zu einer PID.
	 * 
	 * @param strassenPid
	 *            Pid des Stra&szlig;enobjekts im DatKat.
	 *            <p>
	 *            Die Pid einer Stra&szlig;e hat folgenden Aufbau: Richtlinie
	 *            f&uuml;r die Vergabe der Pid:
	 *            strasse.typ.nummer.zusatz.LocCode' mit <p/> 'strasse': fester
	 *            Prefix. <p/> 'typ': Typ der Stra&szlig;e (erster Buchstabe der
	 *            Spalte ROADNUMBER der LCL). <p/> 'nummer': Nummer der
	 *            Stra&szlig;e (Ziffernkombination des Eintrags in der Spalte
	 *            ROADNUMBER der LCL im Anschluss an den Typ). <p/> 'zusatz':
	 *            Optional, falls vorhanden. Zusatzbezeichnung f&uuml;r die
	 *            Stra&szlig;e (eventuell vorhandene zus&auml;tzliche Zeichen
	 *            des Eintrags in der Spalte ROADNUMBER der LCL im Anschluss an
	 *            die Stra&szlig;ennummer). <p/> 'LocCode': Referenz auf das
	 *            Linienelement gem&auml;&szlig; LCL, welches die Stra&szlig;e
	 *            repr&auml;sentiert (PID der Linienreferenz hat die Form
	 *            tmclc.xxxxx, mit xxxxx gleich Location Code des
	 *            Linienelements, 5-stellig mit f&uuml;hrenden Nullen). <p/>
	 *            Existiert f&uuml;r eine Location keine Linienlocation, so wird
	 *            als PID der Wert strasse.S.0.tmclcl.00000 verwendet. <p/>
	 *            Beispiele f&uuml;r g&uuml;ltige PID einerStra&szlig;e: <p/>
	 *            strasse.L.150.a.tmclc.39886 <p/> strasse.A.562.tmclc.07145
	 *            <p/> 'Richtlinie f&uuml;r die Vergabe des Namens: Bezeichung
	 *            von AnschlussVon bis AnschlussBis (Zusatzinfo)' mit <p/>
	 *            'Bezeichnung': Eintrag der Spalte ROADNUMBER der LCL. <p/>
	 *            'AnschlussVon': Eintrag der Spalte FIRST_NAME der LCL. <p/>
	 *            'AnschlussBis': Eintrag der Spalte SECOND_NAME der LCL. <p/>
	 *            'Zusatzinfo': Optional, wenn nicht leer. Eintrag der Spalte
	 *            ROAD_NAME der LCL. <p/> Beispiele f&uuml;r g&uuml;ltige Namen
	 *            einerStra&szlig;e: <p/> L150a von Br&uuml;hl-Nord nach
	 *            K&ouml;ln-Godorf <p/> A562 von Bonn-Rheinaue nach Bonn-Ost
	 *            (Abzweig Bonn S&uuml;d)
	 *            </p>
	 * 
	 * @return Stra&szlig;enobjekt ({@link StrasseInterface}) mit der
	 *         entsprechenden Pid oder <tt>null</tt>, wenn die PID nicht
	 *         existiert.
	 * @throws BisInterfaceException
	 *             wenn die Stra&szlig;ssen nicht aus dem VRZ-System ausgelesen
	 *             werden konnten.
	 */
	StrasseInterface getStrasse(String strassenPid)
			throws BisInterfaceException;

	/**
	 * Gibt eine Menge von Stra&szlig;en des VRZ-Systems zur&uuml;ck, die
	 * bestimmten Kriterien gen&uuml;gen.
	 * 
	 * @param strassenTyp
	 *            Der Typ der Stra&szlig;e. Folgende Werte sind zul&auml;ssig:
	 *            <p>
	 *            <table>
	 *            <tr>
	 *            <td>SonstigeStra&szlig;e</td>
	 *            <td>0</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Autobahn</td>
	 *            <td>1</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Bundesstra&szlig;e</td>
	 *            <td>2</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Landstra&szlig;e</td>
	 *            <td>3</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Kreisstra&szlig;e</td>
	 *            <td>4</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Stadtstra&szlig;e</td>
	 *            <td>5</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Hauptverkehrsstra&szlig;e</td>
	 *            <td>6</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Sammelstra&szlig;e</td>
	 *            <td>7</td>
	 *            </tr>
	 *            <tr>
	 *            <td>Anliegerstra&szlig;e</td>
	 *            <td>8</td>
	 *            </tr>
	 *            </table>
	 *            </p>
	 * 
	 * <p>
	 * Beispiel: F&uuml;r die L128a ist der Typ die 3.
	 * </p>
	 * 
	 * @param strassenNummer
	 *            Stra&szlig;ennummer der Stra&szlig;e. Beispiel: Bei der L128a
	 *            also die 128.
	 * 
	 * @param zusatz
	 *            Zusatz zur Stra&szlig;ennummer der Stra&szlig;e. Beispiel: Bei
	 *            der L128a also "a".
	 * 
	 * @return Alle Stra&szlig;enobjekte ({@link StrasseInterface}), die auf
	 *         die entsprechenden Angaben passen, oder leeres Array, wenn keine
	 *         Stra&szlig;enobjekte gefunden werden.
	 * @throws BisInterfaceException
	 *             wenn die Stra&szlig;ssen nicht aus dem VRZ-System ausgelesen
	 *             werden konnten.
	 */
	StrasseInterface[] getStrassen(int strassenTyp, int strassenNummer,
			String zusatz) throws BisInterfaceException;

	/**
	 * L&ouml;scht das &uuml;bergebene Baustellenobjekt auf der VRZ. Die
	 * Baustelle bleibt f&uuml;r Auswertungen auf Seiten der VRZ weiterhin
	 * erhalten.
	 * 
	 * @param baustellenId
	 *            Id der Baustelle, die bei {@link #erzeugeBaustelle}
	 *            zur&uuml;ckgegeben wurde.
	 * @throws BisInterfaceException
	 *             wenn die Baustelle mit der angegebenen ID nicht existiert
	 *             oder wenn die Baustelle im VRZ-System nicht
	 *             ordnungsgem&auml&szlig; gel&ouml;scht werden konnte.
	 */
	void loescheBaustelle(long baustellenId) throws BisInterfaceException;

	/**
	 * L&ouml;scht die Registrierung eines Listener (Beobachterklasse), der
	 * &uuml;ber
	 * {@link #addKommunikationsStatusListener(BisKommunikationsStatusListener)}
	 * registirert wurde.
	 * 
	 * @param listener
	 *            Beaobachterklasse, die nicht weiter &uuml;ber den
	 *            Kommunikationsstatus informiert werden will.
	 */
	void removeKommunikationsStatusListener(
			BisKommunikationsStatusListener listener);
}
