/*
 * Segment 2 (KEx),  SWE 2.BW-ISIS
 * Copyright (C) 2007 BitCtrl Systems GmbH 
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.kex.isis.isis.impl;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.bsvrz.dav.daf.main.ApplicationCloseActionHandler;
import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.CommunicationError;
import de.bsvrz.dav.daf.main.ConnectionException;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DavConnectionListener;
import de.bsvrz.dav.daf.main.InconsistentLoginException;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.kex.isis.isis.BaustelleInterface;
import de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface;
import de.bsvrz.kex.isis.isis.BisInterface;
import de.bsvrz.kex.isis.isis.BisInterfaceException;
import de.bsvrz.kex.isis.isis.BisKommunikationsStatusListener;
import de.bsvrz.kex.isis.isis.OrtsReferenzAsbStationierungInterface;
import de.bsvrz.kex.isis.isis.OrtsReferenzStrasseUndBetriebsKilometerInterface;
import de.bsvrz.kex.isis.isis.OrtsReferenzStrassenSegmentUndOffsetInterface;
import de.bsvrz.kex.isis.isis.StrasseInterface;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.netz.NetzReferenzen;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.BaustellenListener;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.VerkehrModellNetz;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.zustaende.BaustellenVeranlasser;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.dynobj.DynamischeObjekte;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;

/**
 * Abstrakte Implementation des BisInterface.
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: AbstractBIS.java 21705 2010-02-23 14:22:32Z gieseler $
 * 
 */
public abstract class AbstractBIS implements BisInterface, BaustellenListener,
		ClientReceiverInterface {

	

	/**
	 * Thread zur Initialisierung und &Uuml;berwachung der
	 * Datenverteilerverbindung.
	 * 
	 * @author BitCtrl Systems GmbH, Gieseler
	 * @version $Id: AbstractBIS.java 21705 2010-02-23 14:22:32Z gieseler $
	 * 
	 */
	private class DavConnectionThread extends Thread implements
			DavConnectionListener, ApplicationCloseActionHandler {

		/**
		 * Wartezeit zwischen Anmeldeversuchen.
		 */
		private final int warteZeitNeuverbindung = 2000;

		/**
		 * Parent-Thread.
		 */
		private final AbstractBIS parent;

		/**
		 * Locking-Objekt.
		 */
		private final Object commlock = new Object();

		/**
		 * Konstruktor.
		 * 
		 * @param parent
		 *            Parent-Thread
		 */
		public DavConnectionThread(AbstractBIS parent) {
			this.parent = parent;
			start();
		}

		/**
		 * Benachrichtigt alle angemeldeten Listener &uuml;ber eine
		 * &Auml;nderung des Kommunikationsstatus.
		 */
		private void benachrichtigeListener() {
			synchronized (kommStatusListener) {
				for (BisKommunikationsStatusListener listener : kommStatusListener) {
					listener.setKommunikationsStatus(kommStatus);
				}
			}
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see de.bsvrz.dav.daf.main.ApplicationCloseActionHandler#close(java.lang.String)
		 */
		public void close(final String error) {
			System.err
					.println("Verbindung zum Datenverteiler geschlossen (ApplicationCloseActionHandler)");
			synchronized (commlock) {
				commlock.notify();
			}
		}

		/**
		 * Stellt die Verbindung zum Datenverteiler her.
		 * 
		 * @param davHost
		 *            Hostname des DAV
		 * @param davUsername
		 *            Nutzername am DAV
		 * @param davPassword
		 *            Passwort des Nutzers
		 * @return Datenverteilerverbindung
		 * @throws MissingParameterException
		 *             {@link de.bsvrz.dav.daf.main.ClientDavParameters#ClientDavParameters(ArgumentList argumentList)}
		 * @throws ConnectionException
		 *             {@link de.bsvrz.dav.daf.main.ClientDavConnection#connect()}
		 * @throws CommunicationError
		 *             {@link de.bsvrz.dav.daf.main.ClientDavConnection#connect()}
		 * @throws InconsistentLoginException
		 *             {@link de.bsvrz.dav.daf.main.ClientDavConnection#login(String userName, String password)}
		 */
		private ClientDavInterface connect(String davHost, String davUsername,
				String davPassword) throws MissingParameterException,
				CommunicationError, ConnectionException,
				InconsistentLoginException {
			ArgumentList argList = new ArgumentList(
					new String[] { "-datenverteiler=" + davHost });
			ClientDavParameters cdp = new ClientDavParameters(argList);

			cdp.setApplicationName(BISAPPNAME);
			if (dav == null) {
				dav = new ClientDavConnection(cdp);
			}
			dav.connect();
			dav.login(davUsername, davPassword);

			return dav;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see de.bsvrz.dav.daf.main.DavConnectionListener#connectionClosed(de.bsvrz.dav.daf.main.ClientDavInterface)
		 */
		public void connectionClosed(final ClientDavInterface connection) {
			System.err.println("Verbindung zum Datenverteiler geschlossen!");
			synchronized (commlock) {
				commlock.notify();
			}
		}

		/**
		 * Initialisiert die DAV-Parameter aus Properties-Datei.
		 * 
		 * @throws BisInterfaceException
		 *             bei Ausnahmen
		 */
		private void initDavConnection() throws BisInterfaceException {
			try {
				dav = (ClientDavConnection) connect(properties
						.getProperty(DAV_KEY), properties
						.getProperty(BENUTZER_KEY), properties
						.getProperty(PASSWORT_KEY));
				dav.setCloseHandler(this);
				config = dav.getDataModel();
				vdo = DynamischeObjekte.getInstanz(dav);
				objektTyp = (DynamicObjectType) dav.getDataModel().getType("typ.baustelle");
				if(objektTyp==null) {
					throw new RuntimeException("Der Typ '" + "'typ.baustelle'" + "' existiert nicht!");
				}

				MessageSender.getInstance().init(dav, BISAPPNAME, BISAPPNAME);

				// initialisiere den KV fr die ISIS-Parameterl
				kv = config.getObject(properties.getProperty(KV_KEY));
				if (kv == null) {
					throw new BisInterfaceException(
							"Der Konfigurationsverantwortliche '"
									+ properties.getProperty(KV_KEY)
									+ "' kann nicht initialisiert werden");
				}

				if (!kv.getType().getPid().equals("typ.isis")) {
					throw new BisInterfaceException(
							"Der Konfigurationsverantwortliche '"
									+ properties.getProperty(KV_KEY)
									+ "' ist nicht vom Typ 'typ.isis'!");
				}

				datenAnmeldenDaV();

				// initialisiere das Netzmodell
				netzObjekt = config.getObject(properties.getProperty(NETZ_KEY));
				if (netzObjekt == null) {
					throw new BisInterfaceException("Das Netz '"
							+ properties.getProperty(NETZ_KEY)
							+ "' kann nicht initialisiert werden");
				}

				ObjektFactory.getInstanz().setVerbindung(dav);
				ObjektFactory.getInstanz().registerStandardFactories();

				netzmodell = (VerkehrModellNetz) ObjektFactory.getInstanz()
						.getModellobjekt(netzObjekt);

				// initialisiere das Referenznetz
				NetzReferenzen.getInstanz().setNetzmodell(netzmodell);

				netzmodell.addBaustellenListener(parent);

				// aktuelle Baustellen einlesen
				for (de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle b : netzmodell
						.getBaustellen()) {
					if (istISISBaustelle(b)) {
						Logger.getLogger(logName).log(
								Level.INFO,
								"Die existierende BIS-Baustelle: "
										+ b.getSystemObject()
										+ " wird intialisiert");
						try{
							Baustelle bisbaustelle = new Baustelle(b);
							baustellenMap.put(new Long(b.getId()), bisbaustelle);
						}
						catch(BisInterfaceException be) {
							Logger.getLogger(logName).log(
									Level.WARNING,
									"Die existierende BIS-Baustelle: "
									+ b.getSystemObject()
									+ " kann nicht intialisiert werden (" + be.getMessage() + ")");
						}
					}
				}
				
				// Klasse zum Verwalten dynamischer Objekte initialisieren
				// vdo.setConnection(dav);
				
			} catch (Exception e) {
				if (dav != null) {
					dav.disconnect(false, "");
					dav = null;
				}
				throw new BisInterfaceException(e.getMessage());
			}
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see java.lang.Thread#run()
		 */
		@Override
		public void run() {
			initialisiert = false;

			try {
				while (davConnectionRun) {
					try {
						Logger
								.getLogger("BIS-Interface")
								.log(Level.INFO,
										"Verbindung zum Datenverteiler wird hergestellt");
						initDavConnection();
						Logger.getLogger("BIS-Interface").log(Level.INFO,
								"Verbindung zum Datenverteiler hergestellt");
						setKommStatus(BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_OK);
						if (!initialisiert) {
							synchronized (initLock) {
								initLock.notify();
							}
						}
						initialisiert = true;
						synchronized (commlock) {
							commlock.wait();
						}
						setKommStatus(BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER);
					} catch (Exception e) {
						System.err.println(e.getMessage());
						if (!initialisiert) {
							synchronized (initLock) {
								initLock.notify();
							}
							break;
						}
						initialisiert = true;
					}

					setKommStatus(BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER);
					try {
						Thread.sleep(warteZeitNeuverbindung);
					} catch (InterruptedException e1) {
						/** */
					}
				}
				
				if (dav != null) {
					Logger
					.getLogger("BIS-Interface")
					.log(Level.INFO,
							"Verbindung zum Datenverteiler wird beendet");
					dav.disconnect(false, "");
					setKommStatus(BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER);
					dav = null;
				}
				
			} catch (Exception e) {
				System.err.println(e.getMessage());
			} finally {
				if (dav != null
						&& kommStatus == BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_OK) {
					dav.disconnect(false, "");
				}
			}
		}

		/**
		 * Setzt den internen Kommunikationsstatus.
		 * 
		 * @param neuerstatus
		 *            neue Kommunikationsstatus
		 */
		private void setKommStatus(
				final BisInterface.BisKommunikationsStatus neuerstatus) {
			boolean notify = neuerstatus != kommStatus;

			kommStatus = neuerstatus;

			if (notify) {
				benachrichtigeListener();
			}
		}
	}

	/** Property Schl&uuml;ssel: IP-Adresse oder Rechnername des VRZ-Rechners. */
	protected static final String DAV_KEY = "Datenverteiler"; //$NON-NLS-1$

	/** Property Schl&uuml;ssel: Benutzername am Datenverteile. */
	protected static final String BENUTZER_KEY = "Benutzer"; //$NON-NLS-1$

	/** Property Schl&uuml;ssel: Passwort am Datenverteiler. */
	protected static final String PASSWORT_KEY = "Passwort"; //$NON-NLS-1$

	/**
	 * Property Schl&uuml;ssel: PID des Netzes, auf dem die Baustellen liegen
	 * sollen.
	 */
	private static final String NETZ_KEY = "Netz"; //$NON-NLS-1$

	/** DebugLogger fr Debug-Ausgaben. */
	// Debug _debug = Debug.getLogger();
	/** Property Schl&uuml;ssel: PID Konfigurationsverantwortlicher ISIS. */
	protected static final String KV_KEY = "Konfigurationsverantwortlicher"; //$NON-NLS-1$

	/** Name der Parameter-Attributgruppe. */
	private static final String ATTRIBUTGRUPPE_PARAMETER = "atg.parameterIsis";

	/** Name der Applikation, der f&uuml;r die Debug-Ausgaben benutzt wird. */
	public static final String BISAPPNAME = "BIS";

	/** HashMap der Baustellen. */
	private final Map<Long, Baustelle> baustellenMap = new HashMap<Long, Baustelle>();

	/**
	 * Properties.
	 */
	private final Properties properties = new Properties();

	/** Datenverteiler-Verbindung. */
	private ClientDavConnection dav = null;

	/** Datenmodell. */
	private DataModel config = null;

	/** die Liste der angemeldeten Listener. */
	List<BisKommunikationsStatusListener> kommStatusListener = new ArrayList<BisKommunikationsStatusListener>();

	/** der Konfigurationsverantwortliche. */
	private SystemObject kv;

	/**
	 * das Systemobjekt des Netzes.
	 */
	private SystemObject netzObjekt;

	/**
	 * das Verkehrsmodell.
	 */
	VerkehrModellNetz netzmodell;

	/**
	 * Synchronisationsobjekt f&uuml;r die Initialisierung.
	 */
	Object initLock = new Object();

	/**
	 * der Kommunikationsstatus des Interfaces.
	 */
	BisInterface.BisKommunikationsStatus kommStatus = BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER;

	/**
	 * Flag, ob das Interface initialisiert wurde.
	 */
	private boolean initialisiert = false;

	/**
	 * Datendefinition Parameterempfang.
	 */
	private DataDescription paramEmpfang;

	/** Wartezeit auf erfolgreiche Anmeldebest&auml;tigung. */
	private final long warteZeitAnmeldung = 10000;

	/** Name fr den Logger. */
	private final String logName = "BIS-Interface";

	/** Synchronisationsobjekt f&uuml;r die Baustellenanmeldung. */
	private Baustelle bLock;

	/**	Datenverteilerverbindung &Uuumlberwachungsthread.	*/
	DavConnectionThread connThread = null;
	
	/** Flag, ob die DAV-Verbindungs&uuml;berwachung laufen soll.	*/
	private boolean davConnectionRun = false;
	
	/** Klasse zum Verwalten dynamischer Objekte.	*/
	DynamischeObjekte vdo = null;
	
	/** Objekttyp der Meldungen. 	*/
	DynamicObjectType objektTyp = null;

	
	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#addKommunikationsStatusListener(de.bsvrz.kex.isis.isis.BisKommunikationsStatusListener)
	 */
	public void addKommunikationsStatusListener(
			BisKommunikationsStatusListener listener) {
		synchronized (kommStatusListener) {
			kommStatusListener.add(listener);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.BaustellenListener#baustelleAngelegt(de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.VerkehrModellNetz,
	 *      de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle)
	 */
	public void baustelleAngelegt(
			VerkehrModellNetz netz,
			de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle baustelle) {
		baustelle.getBaustellenEigenschaften().abrufenDatum();

		// ist es die gerade angelegte Baustelle?
		if (bLock != null && baustelle.equals(bLock.getModellBaustelle())) {
			bLock.setMengeAngemeldet(true);
			synchronized (bLock) {
				bLock.notify();
			}
			return;
		}

		// Irgend jemand hat eine Baustelle angelegt
		if (istISISBaustelle(baustelle)) {
			Baustelle bisbaustelle = null;
			try {
				bisbaustelle = new Baustelle(baustelle);
				baustellenMap.put(new Long(baustelle.getId()), bisbaustelle);
			} catch (BisInterfaceException e) {
				Logger.getLogger(logName).log(Level.WARNING, e.getMessage());
			}
		}
	}

	/**
	 * {@inheritDoc}.<br>
	 * empfngt und verarbeitet Meldungen ber entfernte Baustellen meldet sich
	 * fr die nderung der Daten der betreffenden Baustellen ab und leitet dies
	 * an die angemeldeten Listener weiter.
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.BaustellenListener#baustelleEntfernt(de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.VerkehrModellNetz,
	 *      de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle)
	 */
	public void baustelleEntfernt(
			VerkehrModellNetz netz,
			de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle baustelle) {
		// baustelleAbmelden(baustelle);
		// benachrichtigeBaustellenObserverListener(baustelle);
		// System.out.println("Baustelle entfernt: " + netz + ", " + baustelle);
	}

	/**
	 * Meldet alle ben&ouml;tigten Daten am Datenverteiler an.
	 */
	protected void datenAnmeldenDaV() {
		SystemObject[] objekte = null;
		AttributeGroup attributeGroup = null;
		Aspect aspect = null;

		// Anmeldungen zum Empfang
		// Kommunikationsparameter
		objekte = new SystemObject[] { kv };
		attributeGroup = config.getAttributeGroup(ATTRIBUTGRUPPE_PARAMETER);
		aspect = config.getAspect("asp.parameterSoll");
		paramEmpfang = new DataDescription(attributeGroup, aspect);

		// _debug.finer("Anmeldung zum Empfang von Daten beim DaV als SENKE :" +
		// "(Objekt: " + _kv.getName() +
		// ") (ATG: " + ATTRIBUTGRUPPE_PARAMETER +
		// ") (ASP: " + "asp.parameterSoll" +
		// ") (OT: " + _kv.getType().getName() + ")");
		//
		dav.subscribeReceiver(this, objekte, paramEmpfang, new ReceiveOptions(
				false), ReceiverRole.receiver());
		// }
	}


	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#erzeugeBaustelle(java.lang.String,
	 *      java.lang.String, java.lang.String, long, long,
	 *      de.bsvrz.kex.isis.isis.OrtsReferenzStrassenSegmentUndOffsetInterface,
	 *      long, de.bsvrz.kex.isis.isis.BisInterface.BaustellenZustand, int,
	 *      de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface)
	 */
	public long erzeugeBaustelle(final String pid, final String name,
			final String info, final long vonZeitpunkt, final long dauer,
			final OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenz,
			final long laenge, final BaustellenZustand baustellenZustand,
			final int engpassKapazitaet,
			final BaustellenVerantwortlicherInterface baustellenVerantwortlicher)
			throws BisInterfaceException {

		if (ortsReferenz == null) {
			throw new BisInterfaceException(
					"Die Ortsreferenz darf nicht null sein");
		}

		testeVerbindung();

		SystemObject baustellenobjekt = null;
		Baustelle baustelle = null;

		long id = -1;

		try {
//			ConfigurationArea area = dav.getLocalConfigurationAuthority()
//					.getConfigurationArea();

			String baustellepid = pid;

			if (baustellepid == null || baustellepid.length() == 0) {
				baustellepid = Baustelle.bildeBaustellePid();
			}

			String baustellename = name;
			// if (baustellename == null || baustellename.length() == 0) {
			// baustellename = Baustelle.bildeBaustelleName(baustellepid);
			// }

			String baustelleinfo = info;
			if (baustelleinfo == null || baustelleinfo.length() == 0) {
				baustelleinfo = Baustelle.bildeBaustelleInfo();
			}

			if (config.getObject(baustellepid) != null) {
				throw new BisInterfaceException(
						"Die Baustelle kann nicht angelegt werden, da bereits ein Objekt mit dieser PID existiert");
			}

// Gieseler, 25.11.08: benutze de.bsvrz.sys.funclib.vewdynobj zum Anlegen dynamischer
// Objekte
//			baustellenobjekt = area.createDynamicObject(
//					(DynamicObjectType) config.getType("typ.baustelle"),
//					baustellepid, baustellename);
//			if (vdo.erzeugeObjekt("typ.baustelle", baustellepid, baustellename, null)) {
//				baustellenobjekt = dav.getDataModel().getObject(baustellepid);
//			} else {
//				throw new BisInterfaceException("Die Baustelle '" 
//						+ baustellepid + "' kann ber die Verwaltung dynamischer Objekte nicht angelegt werden");
//			}
			
// Gieseler, 17.02.10: neue Bibilothek zur Verwaltung dynamischer Objekte
			baustellenobjekt = vdo.erzeugeObjekt(objektTyp, baustellename, baustellepid);
			
			baustelle = new Baustelle(baustelleinfo, vonZeitpunkt, dauer,
					ortsReferenz, laenge, baustellenZustand, engpassKapazitaet,
					baustellenVerantwortlicher, dav, baustellenobjekt);

			bLock = baustelle;
			synchronized (bLock) {
				netzmodell.baustelleHinzufuegen(baustelle.getDavObjekt());
				bLock.wait(warteZeitAnmeldung);
			}

			if (baustelle.istMengeAngemeldet()) {
				baustellenMap.put(new Long(baustelle.getId()), baustelle);
			}

			baustelle.publiziereDAV();

			id = baustelle.getId();
		} catch (Exception e) {
			// Logger.getLogger(logName).log(Level.WARNING, e.getMessage());
			if (baustelle != null) {
				loescheBaustelle(baustelle.getId());
			} else if (baustellenobjekt != null) {
				try {
					baustellenobjekt.invalidate();
				} catch (ConfigurationChangeException e1) {
					e1.printStackTrace();
				}
			}
			throw new BisInterfaceException(e);
		}

		return id;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#erzeugeBaustellenVerantwortlichen(java.lang.String,
	 *      java.lang.String, java.lang.String, java.lang.String,
	 *      java.lang.String)
	 */
	public BaustellenVerantwortlicherInterface erzeugeBaustellenVerantwortlichen(
			final String firma, final String nameBaustellenVerantwortlicher,
			final String telefonBaustellenVerantwortlicher,
			final String telefonFirma,
			final String telefonMobilBaustellenVerantwortlicher) {

		return new BaustellenVerantwortlicher(firma,
				nameBaustellenVerantwortlicher,
				telefonBaustellenVerantwortlicher, telefonFirma,
				telefonMobilBaustellenVerantwortlicher);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#erzeugeOrtsReferenz(de.bsvrz.kex.isis.isis.StrasseInterface, de.bsvrz.sys.funclib.bitctrl.modell.netz.NetzInterface.FahrtRichtung, long, int)
	 */
	public OrtsReferenzStrasseUndBetriebsKilometerInterface erzeugeOrtsReferenz(
			final StrasseInterface strasse, final FahrtRichtung fahrtRichtung,
			final long betriebsKilometer, final int blockNummer)
			throws BisInterfaceException {

		testeVerbindung();

		return new OrtsReferenzStrasseUndBetriebsKilometer(strasse,
				fahrtRichtung, betriebsKilometer, blockNummer);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#erzeugeOrtsReferenz(java.lang.String,
	 *      long)
	 */
	public OrtsReferenzStrassenSegmentUndOffsetInterface erzeugeOrtsReferenz(
			final String pidStrassenSegment, final long startOffset)
			throws BisInterfaceException {

		testeVerbindung();

		return new OrtsReferenzStrassenSegmentUndOffset(pidStrassenSegment,
				startOffset);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#erzeugeOrtsReferenz(java.lang.String, java.lang.String, de.bsvrz.sys.funclib.bitctrl.modell.netz.NetzInterface.ASBStationierungsRichtung, long)
	 */
	public OrtsReferenzAsbStationierungInterface erzeugeOrtsReferenz(
			final String anfangsKnoten, final String endKnoten,
			final ASBStationierungsRichtung stationierungsRichtung,
			final long stationierung) throws BisInterfaceException {

		testeVerbindung();

		return new OrtsReferenzAsbStationierung(anfangsKnoten, endKnoten,
				stationierungsRichtung, stationierung);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#getBaustelle(long)
	 */
	public BaustelleInterface getBaustelle(final long baustellenId) {
		return baustellenMap.get(new Long(baustellenId));
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#getStrasse(java.lang.String)
	 */
	public StrasseInterface getStrasse(final String strassenPid)
			throws BisInterfaceException {

		testeVerbindung();

		SystemObject co = config.getObject(strassenPid);

		if (co == null) {
			throw new BisInterfaceException("Zur PID '" + strassenPid
					+ "' existiert kein Strassenobjekt");
		}

		return new Strasse(co);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#getStrassen(int, int,
	 *      java.lang.String)
	 */
	public StrasseInterface[] getStrassen(final int strassenTyp,
			final int strassenNummer, final String zusatz)
			throws BisInterfaceException {

		testeVerbindung();

		List<StrasseInterface> strassen = new ArrayList<StrasseInterface>();

		for (SystemObject so : config.getType("typ.strae").getElements()) {
			StrasseInterface strasse = new Strasse(so);

			if (((Strasse) strasse).testeStrassenDaten(strassenTyp,
					strassenNummer, zusatz)) {
				strassen.add(strasse);
			}
		}

		return strassen.toArray(new StrasseInterface[strassen.size()]);
	}

	/**
	 * Initialisiert die DAV-Parameter aus Properties-Datei.
	 * 
	 * 
	 * @param parameterdatei
	 *            Datei mit Parametern
	 * @throws IOException
	 *             wenn Fehler beim Lesen der Datei aufgetreten sind
	 * @throws FileNotFoundException
	 *             wenn die Datei nicht gefunden wurde
	 * @throws BisInterfaceException
	 *             bei sonstigen Ausnahmen
	 */
	public void init(String parameterdatei) throws FileNotFoundException,
			IOException, BisInterfaceException {
		properties.load(new FileInputStream(parameterdatei));
		davConnectionRun = true;

		try {
			connThread = new DavConnectionThread(this);
			synchronized (initLock) {
				initLock.wait();
			}

			if (kommStatus == BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER) {
				connThread.interrupt();
				throw new BisInterfaceException(
						"Die Datenverteilerverbindung konnte nicht initialisiert werden");
			}
			initialisiert = true;
		} catch (Exception e) {
			throw new BisInterfaceException(e.getMessage());
		}
	}

	/**
	 * Beendet die Verbindung zum Datenverteiler. Damit werden alle vom DAV-Interface
	 * gestarteten Threads beendet.
	 */
	public void disconnect() {
		davConnectionRun = false;
        connThread.interrupt();
        try {
        	connThread.join();
        } catch (final InterruptedException e) {
        	/** */
        }

        // Holzhammer: Wenn sich ImplicitUnsubscriber nicht beendet, wird
        // er hier unterbrochen

        try {
        	Thread.sleep(3000L);
        } catch (final InterruptedException ex) { //
        }

        final long timeOutTime = 5000L;
        final int timeOutTimeStep = 500;
        final Thread[] threadsArray = new Thread[Thread.activeCount() * 5];
        Thread.enumerate(threadsArray);
        Thread implicitUnsubscriberThread;
        implicitUnsubscriberThread = null;
        String log;
        log = "";
        for (final Thread thread : threadsArray) {
        	if (thread == null) {
        		break;
        	}
        	
            if (thread.getName().equals("ImplicitUnsubscriber")) {
            	implicitUnsubscriberThread = thread;
            }
            log += "  " + thread + " ("
            	+ (thread.isAlive() ? "alive" : "not alive") + ")\n";
        }

        Logger.getLogger("ImplicitUnsubscriberTerminator").severe(
        		"Alle Threads\n" + log);

        if (implicitUnsubscriberThread != null
        		&& implicitUnsubscriberThread.isAlive()) {
        	Logger.getLogger("ImplicitUnsubscriberTerminator").severe(
        			"ImplicitUnsubscriber: " + implicitUnsubscriberThread);
            final Random r = new Random();
            long timeout;
            timeout = timeOutTime;
            while (timeout > 0) {
            	try {
            		implicitUnsubscriberThread.checkAccess();
                    timeout = -1;
                    Logger.getLogger("ImplicitUnsubscriberTerminator").severe(
                    		"Beim ersten mal unterbrochen");
            	} catch (final SecurityException ex) {
            		Logger.getLogger("ImplicitUnsubscriberTerminator").severe(
                    	"Warte...");
            		final long nextSleep = r.nextInt(timeOutTimeStep);
                    timeout -= nextSleep;
                    try {
                    	Thread.sleep(nextSleep);
                    } catch (final InterruptedException ex1) {
                    	ex1.printStackTrace();
                    }
            	}
            }
            
            implicitUnsubscriberThread.interrupt();
        }
	}
	
	/**
	 * Testet, ob es sich bei einer Baustelle um eine &uuml;ber das
	 * BIS-Interface angelegte handelt.
	 * 
	 * @param b
	 *            zu testende Baustelle
	 * 
	 * @return true, wenn der Veranlasser der Baustelle 'BIS' ist, sonst false
	 */
	boolean istISISBaustelle(
			de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle b) {
		b.getBaustellenEigenschaften().abrufenDatum();
		return b.getBaustellenEigenschaften().getDatum().getVeranlasser() == BaustellenVeranlasser.BIS;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#loescheBaustelle(long)
	 */
	public void loescheBaustelle(final long baustellenId)
			throws BisInterfaceException {

		testeVerbindung();

		Long key = new Long(baustellenId);

		Baustelle baustelle = baustellenMap.get(key);

		if (baustelle != null) {
			netzmodell.baustelleEntfernen(baustelle.getDavObjekt());
			baustelle.loeschen();
			baustellenMap.remove(key);
		} else {
			throw new BisInterfaceException("Zur angeforderten ID '"
					+ baustellenId + "' existiert kein Objekt!");
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BisInterface#removeKommunikationsStatusListener(de.bsvrz.kex.isis.isis.BisKommunikationsStatusListener)
	 */
	public void removeKommunikationsStatusListener(
			final BisKommunikationsStatusListener listener) {
		synchronized (kommStatusListener) {
			kommStatusListener.remove(listener);
		}
	}

	/**
	 * Testet, ob das Interface initialisiert wurde und eine
	 * Datenverteiler-Verbindung besteht.
	 * 
	 * @throws BisInterfaceException
	 *             wenn noch keine Initialisierung erfolgt ist oder wenn keine
	 *             Verbindung besteht
	 * 
	 */
	private void testeVerbindung() throws BisInterfaceException {
		if (!initialisiert) {
			throw new BisInterfaceException(
					"Das Interface wurde noch nicht initialisiert!");
		}

		if (kommStatus == BisInterface.BisKommunikationsStatus.BIS_KOMMUNIKATION_FEHLER) {
			throw new BisInterfaceException(
					"Die Funktion kann nicht ausgefhrt werden, da keine Datenverteilerverbindung besteht!");
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.dav.daf.main.ClientReceiverInterface#update(de.bsvrz.dav.daf.main.ResultData[])
	 */
	public void update(final ResultData[] results) {
		for (ResultData datensatz : results) {
			if (datensatz.hasData()) {
				if (datensatz.getDataDescription().equals(paramEmpfang)) {
					Logger.getLogger(logName).log(Level.FINE,
							"Parameterdatensatz empfangen");
					ISISParameter.getInstance().aktualisieren(
							datensatz.getData());
					for (Baustelle baustelle : baustellenMap.values()) {
						baustelle.aktualisiereEreignisse();
					}
				}
			}
		}
	}
}
