/*
 * Segment 2 (KEx),  SWE 2.BW-ISIS
 * Copyright (C) 2007 BitCtrl Systems GmbH 
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.kex.isis.isis.impl;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.kex.isis.isis.BaustelleInterface;
import de.bsvrz.kex.isis.isis.BaustellenPrognoseInterface;
import de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface;
import de.bsvrz.kex.isis.isis.BisInterface;
import de.bsvrz.kex.isis.isis.BisInterfaceException;
import de.bsvrz.kex.isis.isis.OrtsReferenzStrassenSegmentUndOffsetInterface;
import de.bsvrz.kex.isis.isis.BisInterface.BaustellenZustand;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.Datum;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.kalender.objekte.Ereignis;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.onlinedaten.OdBaustellenSimulation;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.zustaende.BaustellenStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.zustaende.BaustellenVeranlasser;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.operatingMessage.MessageGrade;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.bsvrz.sys.funclib.operatingMessage.MessageType;

/**
 * BIS-Baustelle.
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: Baustelle.java 21573 2010-02-17 13:35:08Z gieseler $
 */
public class Baustelle implements BaustelleInterface, ClientSenderInterface,
		DatensatzUpdateListener {

	/**
	 * Erzeugt die Info f&uuml;r eine Baustelle.
	 * 
	 * @return Info
	 */
	public static String bildeBaustelleInfo() {
		return "Baustelle angelegt durch ISIS-Interface";
	}

	/**
	 * Erzeugt den Namen f&uuuml;r eine anzulegende Baustelle. Der Name wird auf
	 * der Basis der PID folgt gebildet: <br>
	 * NAME = PID
	 * 
	 * @param baustellepid
	 *            PID der Baustelle
	 * 
	 * @return Name
	 */
	public static String bildeBaustelleName(String baustellepid) {
		return baustellepid;
	}

	/**
	 * Erzeugt eine PID f&uuuml;r eine anzulegende Baustelle. Die PID wird wie
	 * folgt gebildet: <br>
	 * baustelle.isis.aktuelle-zeit-millisekunden
	 * 
	 * @return PID
	 */
	public static String bildeBaustellePid() {
		return "baustelle.isis." + System.currentTimeMillis();
	}

	/** Name fr den Logger. */
	private final String logName = "BIS-Interface";

	/**
	 * Ortsreferenzobjekt (Stra&szlig;ensegment mit Offset), auf der die
	 * Baustelle beginnt.
	 */
	private OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenzBaustelle = null;

	/** L&auml;nge der Baustelle stromabw&auml;rts. */
	private long laengeBaustelle = 0;

	/** Der Baustellenverantwortliche. */
	// private BaustellenVerantwortlicherInterface verantwortlicherBaustelle;
	/** Neuer Zustand der Baustelle. */
	private BaustellenZustand zustandBaustelle = null;

	/** Informationstext zur Baustelle. */
	private String infoTextBaustelle = "";

	/** Datenverteiler-Verbindung. */
	private ClientDavInterface dav = null;

	/** das zugeordnete DAV-Objekt. */
	private SystemObject davObjekt = null;

	/** Synchronisationsobjekt DaV-Anmeldung. */
	private final Object anmeldungsLock = new Object();

	/** das zugeordnete Modell-Objekt. */
	private final de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle modellBaustelle;

	/** die zugeordneten Situationseigenschaften des Modell-Objektes. */
	private de.bsvrz.sys.funclib.bitctrl.modell.verkehr.parameter.PdSituationsEigenschaften.Daten baustellenSituationModell;

	/** die zugeordneten Baustelleneigenschaften des Modell-Objektes. */
	private de.bsvrz.sys.funclib.bitctrl.modell.verkehr.parameter.PdBaustellenEigenschaften.Daten baustellenEigenschaftenModell;

	/** die zugeordneten Verantwortlicher-Eigenschaften des Modell-Objektes. */
	private de.bsvrz.sys.funclib.bitctrl.modell.verkehr.parameter.PdBaustellenVerantwortlicher.Daten baustellenVerantwortlicherModell;

	/** Flag, ob alle Daten erfolgreich beim DaV angemeldet wurden. */
	private boolean datenAngemeldet = false;

	/** Flag, ob die Baustelle erfolgreich an der Menge angemeldet wurde. */
	private boolean mengeAngemeldet = false;

	/** Synchronisationsobjekt Prognoseanfrage. */
	private final Object prognoseLock = new Object();

	/** Timeout Prognoseeanfrage. */
	private final long warteZeitPrognose = 100000;

	/** Flag, ob Prognosedaten empfangen wurden. */
	private boolean prognoseEmpfangen = false;

	/** Startzeit Prognoseeanfrage. */
	private long startZeitPrognose;

	/** Liste der angelegten Ereignisse. */
	private final List<BaustellenEreignis> ereignisse = new ArrayList<BaustellenEreignis>();

	/**
	 * Erzeugt ein neues Baustellenobjekt (typ.baustelle) auf Seiten der VRZ.
	 * 
	 * @param dav
	 *            Datenverteilerverbindung
	 * @param objekt
	 *            Systemobjekt der Baustelle
	 * 
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	public Baustelle(ClientDavInterface dav, SystemObject objekt)
			throws BisInterfaceException {
		this.dav = dav;
		this.davObjekt = objekt;
		modellBaustelle = (de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle) ObjektFactory
				.getInstanz().getModellobjekt(objekt);
		anmeldenDav();
	}

	/**
	 * Erzeugt eine neue Baustelle auf der Basis einer Baustelle des
	 * Netzmodells.
	 * 
	 * @param baustelle
	 *            Baustelle des Netzmodells.
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	public Baustelle(
			de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle baustelle)
			throws BisInterfaceException {
		modellBaustelle = baustelle;
		baustellenEigenschaftenModell = modellBaustelle
				.getBaustellenEigenschaften().abrufenDatum();
		baustellenSituationModell = modellBaustelle
				.getSituationsEigenschaften().abrufenDatum();
		baustellenVerantwortlicherModell = modellBaustelle
				.getBaustellenVerantwortlicher().abrufenDatum();

		if(baustellenSituationModell.getSegment(0) == null) {
			throw new BisInterfaceException("Die Baustelle besitzt kein zugeordnetes Strassensegment");
		}
		
		ortsReferenzBaustelle = new OrtsReferenzStrassenSegmentUndOffset(
				baustellenSituationModell.getSegment(0).getPid(),
				baustellenSituationModell.getStartOffset());
		davObjekt = baustelle.getSystemObject();
		anmeldenDav();
	}

	/**
	 * Erzeugt ein neues Baustellenobjekt (typ.baustelle) auf Seiten der VRZ.
	 * 
	 * @param info
	 *            Informationstext zur Baustelle oder <tt>null</tt>, falls
	 *            der Infotext automatisch vergeben werden soll.
	 * @param vonZeitpunkt
	 *            Zeitpunkt des Baustellenbeginns (in ms seit dem 1.1.1970 UTC).
	 * @param dauer
	 *            Dauer der Baustelle (in ms).
	 * @param ortsReferenz
	 *            Ortsreferenzobjekt (Stra&szlig;enSegments mit Offset), auf der
	 *            die Baustelle beginnt.
	 * @param laenge
	 *            L&aumlnge des Baustelle stromabwrts in Metern.
	 * @param baustellenZustand
	 *            Zustand der Baustelle.
	 * @param engpassKapazitaet
	 *            Restkapazit&auml;t w&auml;hrend der G&uuml;ltigkeitsdauer der
	 *            Baustelle in Fz/h.
	 * @param baustellenVerantwortlicher
	 *            Angaben zum Baustellenverantwortlichen eines
	 *            Baustellenobjekts.
	 * @param dav
	 *            Datenverteilerverbindung
	 * @param objekt
	 *            DAV-Systemobjekt der Baustelle
	 * @throws BisInterfaceException
	 *             bei Fehlern in Zusammenhang mit der Erzeugung der Baustelle
	 */
	public Baustelle(String info, long vonZeitpunkt, long dauer,
			OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenz,
			long laenge, BaustellenZustand baustellenZustand,
			int engpassKapazitaet,
			BaustellenVerantwortlicherInterface baustellenVerantwortlicher,
			ClientDavInterface dav, SystemObject objekt)
			throws BisInterfaceException {
		this(dav, objekt);
		if (laenge < 0) {
			throw new BisInterfaceException(
					"Eine negative Lnge wird nicht untersttzt");
		}
		this.laengeBaustelle = laenge;
		baustellenEigenschaftenModell = modellBaustelle
				.getBaustellenEigenschaften().erzeugeDatum();
		baustellenEigenschaftenModell.setRestKapazitaet(engpassKapazitaet);
		baustellenEigenschaftenModell
				.setStatus(getDavBaustellenStatus(baustellenZustand));
		baustellenEigenschaftenModell.setVeranlasser(BaustellenVeranlasser.BIS);

		baustellenSituationModell = modellBaustelle
				.getSituationsEigenschaften().erzeugeDatum();
		baustellenSituationModell.setDauer(dauer);
		baustellenSituationModell.setStartZeit(vonZeitpunkt);

		ortsReferenzBaustelle = ortsReferenz;
		setzeOrtsreferenz();

		baustellenVerantwortlicherModell = modellBaustelle
				.getBaustellenVerantwortlicher().erzeugeDatum();
		modellBaustelle.getBaustellenVerantwortlicher().erzeugeDatum();

		setzeBaustellenVerantwortlicherModell(baustellenVerantwortlicher);
		setBaustellenZustand(baustellenZustand);
		aendereInfo(info);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereBaustellenGueltigkeitsZeitraum(long,
	 *      long)
	 */
	public void aendereBaustellenGueltigkeitsZeitraum(final long vonZeitpunkt,
			final long dauer) throws BisInterfaceException {
		baustellenSituationModell.setDauer(dauer);
		baustellenSituationModell.setStartZeit(vonZeitpunkt);
		try {
			publiziereDAVSituation();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereBaustellenOrt(de.bsvrz.kex.isis.isis.OrtsReferenzStrassenSegmentUndOffsetInterface)
	 */
	public void aendereBaustellenOrt(
			final OrtsReferenzStrassenSegmentUndOffsetInterface ortsReferenzNeu)
			throws BisInterfaceException {
		this.ortsReferenzBaustelle = ortsReferenzNeu;
		setzeOrtsreferenz();
		try {
			publiziereDAVSituation();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereBaustellenStartOffsetUndLaenge(long,
	 *      long)
	 */
	public void aendereBaustellenStartOffsetUndLaenge(long startOffset,
			final long laenge) throws BisInterfaceException {
		ortsReferenzBaustelle = new OrtsReferenzStrassenSegmentUndOffset(
				ortsReferenzBaustelle.getStrassenSegment().getPid(),
				startOffset);
		this.laengeBaustelle = laenge;
		setzeOrtsreferenz();
		try {
			publiziereDAVSituation();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereBaustellenVerantwortlichen(de.bsvrz.kex.isis.isis.BaustellenVerantwortlicherInterface)
	 */
	public void aendereBaustellenVerantwortlichen(
			BaustellenVerantwortlicherInterface baustellenVerantwortlicher)
			throws BisInterfaceException {
		setzeBaustellenVerantwortlicherModell(baustellenVerantwortlicher);
		try {
			publiziereDAVBaustellenVerantwortlicher();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereBaustellenZustand(de.bsvrz.kex.isis.isis.BisInterface.BaustellenZustand)
	 */
	public void aendereBaustellenZustand(BaustellenZustand baustellenZustand)
			throws BisInterfaceException {
		setBaustellenZustand(baustellenZustand);
		baustellenEigenschaftenModell
				.setStatus(getDavBaustellenStatus(this.zustandBaustelle));
		try {
			publiziereDAVBaustelle();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereEngpassKapazitaet(int)
	 */
	public void aendereEngpassKapazitaet(int engpassKapazitaet)
			throws BisInterfaceException {
		if (engpassKapazitaet < 0) {
			throw new BisInterfaceException(
					"Die Engpasskapazitt muss positiv sein");
		}

		baustellenEigenschaftenModell.setRestKapazitaet(engpassKapazitaet);
		try {
			publiziereDAVBaustelle();
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Fehler beim Senden zum Datenverteiler: " + e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#aendereInfo(java.lang.String)
	 */
	public void aendereInfo(final String infoText) throws BisInterfaceException {
		this.infoTextBaustelle = infoText;
		try {
			publiziereDAVObjektDaten();
		} catch (ConfigurationChangeException e) {
			throw new BisInterfaceException(e.getMessage());
		}
	}

	/**
	 * Aktualisiert die Ereignisse der Baustelle.
	 * <p>
	 * Die aktuellen Ereignisse werden auf der Basis der aktuellen Parameter in
	 * der Art angepasst, dass die dann resultierenden Ereigniseintr&auml;ge zum
	 * aktuellen Parametersatz passen:
	 * <p>- Wird f&uuml;r einen Zustand, bei dem bisher ein Ereignis angelegt
	 * werden musste, dies jetzt nicht mehr gefordert, werden die entsprechenden
	 * Ereignisse gel&ouml;scht.
	 * </p>
	 * <p>- Wird f&uuml;r einen Zustand, bei dem bisher kein Ereignis angelegt
	 * werden musste, dies jetzt gefordert, werden die entsprechenden Ereignisse
	 * angelegt.
	 * </p>
	 */
	public void aktualisiereEreignisse() {
		List<BaustellenEreignis> geloescht = new ArrayList<BaustellenEreignis>();

		// bestehende Ereignisse berprfen
		for (BaustellenEreignis ereignis : ereignisse) {
			if (!BaustellenEreignisFactory.getInstance().istEreignis(
					ereignis.getZustand())) {
				try {
					ereignis.loeschen();
				} catch (ConfigurationChangeException e) {
					e.printStackTrace();
				}
				geloescht.add(ereignis);
			}
		}
		ereignisse.removeAll(geloescht);

		// Ereignisse fr aktuellen Zustand anlegen
		try {
			erzeugeEreignis();
		} catch (BisInterfaceException e) {
			Logger.getLogger(logName).log(Level.WARNING, e.getMessage());
		}
	}

	/**
	 * Meldet alle ben&ouml;tigten Daten beim Datenverteiler an.
	 * 
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	private void anmeldenDav() throws BisInterfaceException {
		try {
			modellBaustelle.getSituationsEigenschaften().anmeldenSender();
			modellBaustelle.getBaustellenEigenschaften().anmeldenSender();
			modellBaustelle.getBaustellenVerantwortlicher().anmeldenSender();
			modellBaustelle.getSituationsEigenschaften()
					.addUpdateListener(this);
			modellBaustelle.getBaustellenEigenschaften()
					.addUpdateListener(this);
			modellBaustelle.getBaustellenVerantwortlicher().addUpdateListener(
					this);

			synchronized (anmeldungsLock) {
				try {
					anmeldungsLock.wait(10000);
				} catch (InterruptedException e) {
					/** */
				}
			}

			if (!datenAngemeldet) {
				throw new BisInterfaceException(
						"Die Baustelle konnte nicht erfolgreich beim DaV angemeldet werden");
			}

		} catch (AnmeldeException e) {
			throw new BisInterfaceException(e.getMessage());
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.dav.daf.main.ClientSenderInterface#dataRequest(de.bsvrz.dav.daf.main.config.SystemObject,
	 *      de.bsvrz.dav.daf.main.DataDescription, byte)
	 */
	public void dataRequest(SystemObject object,
			DataDescription dataDescription, byte state) {
		/** */
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener#datensatzAktualisiert(de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent)
	 */
	public void datensatzAktualisiert(DatensatzUpdateEvent event) {
		if (event.getDatensatz() instanceof OdBaustellenSimulation) {
			if (event.getDatum().isValid()
					&& (event.getDatum().getZeitstempel() > startZeitPrognose)) {
				prognoseEmpfangen = true;
				synchronized (prognoseLock) {
					prognoseLock.notify();
				}
			}
		}

		// teste komplette Anmeldung
		datenAngemeldet = ((modellBaustelle.getBaustellenEigenschaften()
				.getDatum() != null) && (modellBaustelle
				.getBaustellenEigenschaften().getDatum().isValid() || modellBaustelle
				.getBaustellenEigenschaften().getDatum().getDatenStatus() == Datum.Status.KEINE_DATEN))
				&& ((modellBaustelle.getBaustellenVerantwortlicher().getDatum() != null) && (modellBaustelle
						.getBaustellenVerantwortlicher().getDatum().isValid() || modellBaustelle
						.getBaustellenVerantwortlicher().getDatum()
						.getDatenStatus() == Datum.Status.KEINE_DATEN))
				&& ((modellBaustelle.getSituationsEigenschaften().getDatum() != null) && (modellBaustelle
						.getSituationsEigenschaften().getDatum().isValid() || modellBaustelle
						.getSituationsEigenschaften().getDatum()
						.getDatenStatus() == Datum.Status.KEINE_DATEN));

		if (datenAngemeldet) {
			synchronized (anmeldungsLock) {
				anmeldungsLock.notify();
			}
		}
	}

	/**
	 * Erzeugt ein Baustellenereignis.
	 * 
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	private void erzeugeEreignis() throws BisInterfaceException {
		Ereignis ereignis = BaustellenEreignisFactory.getInstance()
				.anlegenEreignis(this);
		if (ereignis != null) {
			ereignisse.add(new BaustellenEreignis(zustandBaustelle, ereignis));
		}

	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getBaustellenVerantwortlichen()
	 */
	public BaustellenVerantwortlicherInterface getBaustellenVerantwortlichen() {
		return new BaustellenVerantwortlicher(baustellenVerantwortlicherModell
				.getFirma(), baustellenVerantwortlicherModell
				.getNameBaustellenVerantwortlicher(),
				baustellenVerantwortlicherModell
						.getTelefonBaustellenVerantwortlicher(),
				baustellenVerantwortlicherModell.getTelefonFirma(),
				baustellenVerantwortlicherModell
						.getTelefonMobilBaustellenVerantwortlicher());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getBaustellenZustand()
	 */
	public BaustellenZustand getBaustellenZustand() {
		return zustandBaustelle;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getDauer()
	 */
	public long getDauer() {
		return baustellenSituationModell.getDauer();
	}

	/**
	 * Konvertiert den Baustellzustand des BIS-Interface in den zugeh&ouml;rigen
	 * Zustand am DAV.<br>
	 * Die Umsetzung ist wie folgt definiert: <table>
	 * <tr>
	 * <td>Zustand im BIS-ISIS</td>
	 * <td>Zustand in der VRZ [DaK]</td>
	 * <td>vordefinierte Konstante im BIS-Interface</td>
	 * </tr>
	 * <tr>
	 * <td>Grobplanung</td>
	 * <td>entworfen (0)</td>
	 * <td>BaustellenZustand.ENTWORFEN</td>
	 * </tr>
	 * <tr>
	 * <td>Feinplanung</td>
	 * <td>geplant (1)</td>
	 * <td>BaustellenZustand.GEPLANT</td>
	 * </tr>
	 * <tr>
	 * <td>Umsetzung</td>
	 * <td>gltig (2)</td>
	 * <td>BaustellenZustand.GUELTIG</td>
	 * </tr>
	 * <tr>
	 * <td>Gelscht</td>
	 * <td>storniert (3)</td>
	 * <td>BaustellenZustand.STORNIERT</td>
	 * </tr>
	 * </table>
	 * 
	 * @param bisZustand
	 *            der Zustand der Baustelle entsprechend BIS-Interface
	 * @return zugeh&ouml;riger Zustand des DAV
	 * @throws BisInterfaceException
	 *             wen der Zustand nicht abgebildet werden kann.
	 */
	public BaustellenStatus getDavBaustellenStatus(BaustellenZustand bisZustand)
			throws BisInterfaceException {
		if (bisZustand == BaustellenZustand.ENTWORFEN) {
			return BaustellenStatus.ENTWORFEN;
		}

		if (bisZustand == BaustellenZustand.GUELTIG) {
			return BaustellenStatus.GUELTIG;
		}

		if (bisZustand == BaustellenZustand.GEPLANT) {
			return BaustellenStatus.GEPLANT;
		}

		if (bisZustand == BaustellenZustand.STORNIERT) {
			return BaustellenStatus.STORNIERT;
		}

		throw new BisInterfaceException("Der Baustellenzustand '" + bisZustand
				+ "' kann nicht auf einen DAV-Zustand abgebildet werden");
	}

	/**
	 * Gibt das zur Baustelle gehrende DaV-Objekt zur&uuml;ck.
	 * 
	 * @return Datenverteiler-Objekt der Baustelle
	 */
	public SystemObject getDavObjekt() {
		return davObjekt;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getEngpassKapazitaet()
	 */
	public int getEngpassKapazitaet() {
		return (int) baustellenEigenschaftenModell.getRestKapazitaet();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getId()
	 */
	public long getId() {
		return davObjekt.getId();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getInfo()
	 */
	public String getInfo() {
		return infoTextBaustelle;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getLaenge()
	 */
	public long getLaenge() {
		return laengeBaustelle;
	}

	/**
	 * Gibt die zugeh&ouml;rige Baustelle des Verkehrsmodells zur&uuml;ck.
	 * 
	 * @return zugeh&ouml;rige Baustelle des Verkehrsmodells
	 */
	public de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.Baustelle getModellBaustelle() {
		return modellBaustelle;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getName()
	 */
	public String getName() {
		return davObjekt.getName();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getOrtsReferenzStrassenSegmentUndOffset()
	 */
	public OrtsReferenzStrassenSegmentUndOffsetInterface getOrtsReferenzStrassenSegmentUndOffset() {
		return ortsReferenzBaustelle;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getPid()
	 */
	public String getPid() {
		return davObjekt.getPid();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#getStartZeitpunkt()
	 */
	public long getStartZeitpunkt() {
		// return modellBaustelle.getSituationsEigenschaften().getDatum()
		// .getStartZeit();
		return baustellenSituationModell.getStartZeit();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.dav.daf.main.ClientSenderInterface#isRequestSupported(de.bsvrz.dav.daf.main.config.SystemObject,
	 *      de.bsvrz.dav.daf.main.DataDescription)
	 */
	public boolean isRequestSupported(SystemObject object,
			DataDescription dataDescription) {
		return false;
	}

	/**
	 * Test, ob die Baustelle bei der Baustellenmenge angemeldet ist.
	 * 
	 * @return true wenn angemeldet, sonst false
	 */
	public boolean istMengeAngemeldet() {
		return mengeAngemeldet;
	}

	/**
	 * L&ouml;scht die Baustelle.
	 * 
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	public void loeschen() throws BisInterfaceException {
		try {
			davObjekt.invalidate();
			zustandBaustelle = BisInterface.BaustellenZustand.STORNIERT;
			BaustellenEreignisFactory.getInstance().anlegenEreignis(this);
		} catch (ConfigurationChangeException e) {
			throw new BisInterfaceException(e.getMessage());
		}
	}

	/**
	 * Publiziert alle Baustellendaten zum DaV.
	 * 
	 * @throws DatensendeException
	 *             bei Ausnahmen
	 * @throws BisInterfaceException
	 *             wenn das Baustellenereignis nicht angelegt werden kann
	 */
	public void publiziereDAV() throws DatensendeException,
			BisInterfaceException {
		publiziereDAVBaustelle();
		publiziereDAVSituation();
		publiziereDAVBaustellenVerantwortlicher();
	}

	/**
	 * Publiziert die Baustelleneigenschaften zum DaV.
	 * 
	 * @throws DatensendeException
	 *             bei Ausnahmen
	 * @throws BisInterfaceException
	 *             wenn das Baustellenereignis nicht angelegt werden kann
	 */
	private void publiziereDAVBaustelle() throws DatensendeException,
			BisInterfaceException {
		modellBaustelle.getBaustellenEigenschaften().sendeDaten(
				baustellenEigenschaftenModell);
		erzeugeEreignis();
	}

	/**
	 * Publiziert den Baustellenverantwortlichen zum DaV.
	 * 
	 * @throws DatensendeException
	 *             bei Ausnahmen
	 */
	private void publiziereDAVBaustellenVerantwortlicher()
			throws DatensendeException {
		modellBaustelle.getBaustellenVerantwortlicher().sendeDaten(
				baustellenVerantwortlicherModell);
	}

	/**
	 * Publiziert die Objektdaten der Baustelle zum DaV.
	 * 
	 * @throws ConfigurationChangeException
	 *             bei Ausnahmen
	 * 
	 */
	private void publiziereDAVObjektDaten() throws ConfigurationChangeException {
		// typ.dynamischesObjekt, atg.info ist konfigurierend
		dav = ObjektFactory.getInstanz().getVerbindung();
		AttributeGroup atg = dav.getDataModel().getAttributeGroup("atg.info");
		Data data = dav.createData(atg);
		data.getTextValue("kurzinfo").setText(infoTextBaustelle);
		data.getTextValue("beschreibung").setText(infoTextBaustelle);

		davObjekt.setConfigurationData(atg, data);
	}

	/**
	 * Publiziert die Situationsdaten der Baustelle zum DaV.
	 * 
	 * @throws DatensendeException
	 *             bei Ausnahmen
	 */
	private void publiziereDAVSituation() throws DatensendeException {
		modellBaustelle.getSituationsEigenschaften().sendeDaten(
				baustellenSituationModell);
		synchronized (anmeldungsLock) {
			try {
				anmeldungsLock.wait(10000);
			} catch (InterruptedException e) {
				/** */
			}
		}
	}

	/**
	 * Setzt den Zustand der Baustelle. Wenn der Zustand g&uuml;ltig wird oder
	 * war, werden die VRZ-Benutzer durch eine Betriebsmeldung informiert, wenn
	 * der Zeitpunkt des Baustellenbeginns innerhalb der n&auml;chsten n Tage
	 * liegt, wobei n dem aktuellen Wert des Parameters
	 * atg.parameterIsis.WarnZeitraum entspricht.
	 * 
	 * @param neuerZustand
	 *            neuer Zustand der Baustelle
	 */
	private void setBaustellenZustand(
			final BisInterface.BaustellenZustand neuerZustand) {
		if (zustandBaustelle == neuerZustand) {
			return;
		}

		// if (((zustandBaustelle == BisInterface.BaustellenZustand.GUELTIG) ||
		// (neuerZustand == BisInterface.BaustellenZustand.GUELTIG))
		// && (this.getStartZeitpunkt() > System.currentTimeMillis())
		// && (this.getStartZeitpunkt() < (System.currentTimeMillis() +
		// ISISParameter.getInstance().warnZeitraum))) {
		if ((zustandBaustelle == BisInterface.BaustellenZustand.GUELTIG)
				|| (neuerZustand == BisInterface.BaustellenZustand.GUELTIG)) {
			if ((this.getStartZeitpunkt() > System.currentTimeMillis())
					&& (this.getStartZeitpunkt() < (System.currentTimeMillis() + ISISParameter
							.getInstance().warnZeitraum))) {
				DateFormat formatter = new SimpleDateFormat();

				String nachricht = "Die Baustelle " + this.getName()
						+ " wird am "
						+ formatter.format(this.getStartZeitpunkt()) + " "
						+ neuerZustand;
				MessageSender.getInstance().sendMessage(
						MessageType.APPLICATION_DOMAIN,
						MessageGrade.INFORMATION, nachricht);
			}
		}

		zustandBaustelle = neuerZustand;
	}

	/**
	 * Setzt das Flag, ob die Baustelle bei der Baustellenmenge angemeldet ist.
	 * 
	 * @param angemeldet
	 *            angemeldet ja/nein
	 */
	public void setMengeAngemeldet(boolean angemeldet) {
		mengeAngemeldet = angemeldet;
	}

	/**
	 * Setzt die Daten des Baustellenverantwortlichen.
	 * 
	 * @param verantwortlicher
	 *            Baustellenverantwortlicher
	 */
	private void setzeBaustellenVerantwortlicherModell(
			final BaustellenVerantwortlicherInterface verantwortlicher) {
		baustellenVerantwortlicherModell.setFirma(verantwortlicher.getFirma());
		baustellenVerantwortlicherModell
				.setNameBaustellenVerantwortlicher(verantwortlicher
						.getNameBaustellenVerantwortlicher());
		baustellenVerantwortlicherModell
				.setTelefonBaustellenVerantwortlicher(verantwortlicher
						.getTelefonBaustellenVerantwortlicher());
		baustellenVerantwortlicherModell.setTelefonFirma(verantwortlicher
				.getTelefonFirma());
		baustellenVerantwortlicherModell
				.setTelefonMobilBaustellenVerantwortlicher(verantwortlicher
						.getTelefonMobilBaustellenVerantwortlicher());
	}

	/**
	 * Setzt die Ortsreferenzdaten der Baustelle. Ausgehend von der Ortsreferenz
	 * des Beginns werden alle folgenden Stra&szlig;ensegmente gesucht, bis die
	 * komplette L&auml;nge der Baustelle abgebildet ist. Wenn keine
	 * Folgesegmente bestimmt werden k&ouml;nnen, wird die L&auml;nge der
	 * Baustelle entsprechend korrigiert.
	 * 
	 * @throws BisInterfaceException
	 *             bei Ausnahmen
	 */
	private void setzeOrtsreferenz() throws BisInterfaceException {
		baustellenSituationModell.setStartOffset(ortsReferenzBaustelle
				.getStartOffset());
		StrassenSegment segment = ((OrtsReferenzStrassenSegmentUndOffset) ortsReferenzBaustelle)
				.getModelSegment();
		LinkedList<StrassenSegment> segmente = new LinkedList<StrassenSegment>();

		if (ortsReferenzBaustelle.getStartOffset() > segment.getLaenge()) {
			throw new BisInterfaceException(
					"Der Offset auf dem Straensegment ist grer als seine Lnge");
		}

		// Diese Einschrnkung ist nicht mehr reproduzierbar!? In irgend einem
		// Dokument glaube ich mal sowas gelsen zu haben!
		// Mglicherweise hing das mit der alten Referenzierung der Baustellen
		// ber Strae und Offset zusammen.
		// Wahrscheinlich deshalb, weil die Baustellen in einer frheren Version
		// der Modellierung ber eine Strae und den Offset auf der Strae
		// verortet waren. In diesem Fall ist es schwierig, die Verortung aus
		// einem inneren Straensegment zu bilden, da die Strae ber mehrere
		// innere Segmente laufen kann.
		//
		// if (segment instanceof InneresStrassenSegment) {
		// throw new BisInterfaceException(
		// "Die Ortsreferenz liegt auf einem inneren Straensegment");
		// }

		segmente.add(segment);
		long restlaenge = (long) (laengeBaustelle - (segment.getLaenge() - ortsReferenzBaustelle
				.getStartOffset()));
		baustellenSituationModell.setEndOffset((ortsReferenzBaustelle
				.getStartOffset() + laengeBaustelle));

		// suche Folgesegmente bis die komplette Lnge abgedeckt ist
		while ((segment != null) && (restlaenge > 0)) {
			// bestimme Folgesegment
			if (segment instanceof InneresStrassenSegment) {
				segment = ((InneresStrassenSegment) segment).getNachSegment();
			} else {
				// AeusseresStrassenSegment
				StrassenKnoten knoten = ((AeusseresStrassenSegment) segment)
						.getNachKnoten();
				if (knoten != null) {
					segment = knoten
							.getInnereVerbindungDanach((AeusseresStrassenSegment) segment);
				} else {
					segment = null;
				}
			}

			if (segment != null) {
				segmente.add(segment);
				restlaenge -= segment.getLaenge();

				if (restlaenge <= 0) {
					baustellenSituationModell.setEndOffset((long) (segment
							.getLaenge() + restlaenge));
				}
			}
		}

		// korrigiere Lnge
		if (restlaenge > 0) {
			baustellenSituationModell.setEndOffset((long) segmente.getLast()
					.getLaenge());
			laengeBaustelle -= restlaenge;
		}

		baustellenSituationModell.setSegmente(segmente);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.BaustelleInterface#startePrognose(java.lang.String,
	 *      java.lang.String)
	 */
	public BaustellenPrognoseInterface startePrognose(String veranlasser,
			String ursache) throws BisInterfaceException {
		try {
			OdBaustellenSimulation antwort = modellBaustelle
					.getOnlineDatensatz(OdBaustellenSimulation.class);
			antwort.abrufenDatum(OdBaustellenSimulation.Aspekte.PrognoseNormal
					.getAspekt());
			antwort.addUpdateListener(
					OdBaustellenSimulation.Aspekte.PrognoseNormal.getAspekt(),
					this);

			if (antwort.getDatum(OdBaustellenSimulation.Aspekte.PrognoseNormal
					.getAspekt()) == null) {
				startZeitPrognose = 0;
			} else {
				startZeitPrognose = antwort.getDatum(
						OdBaustellenSimulation.Aspekte.PrognoseNormal
								.getAspekt()).getZeitstempel();
			}

			prognoseEmpfangen = false;

			// System.out.println("Prognose anfordern: " + startZeitPrognose
			// + " um " + System.currentTimeMillis());
			modellBaustelle.simuliereBaustelle(veranlasser, ursache);

			synchronized (prognoseLock) {
				try {
					prognoseLock.wait(warteZeitPrognose);
				} catch (InterruptedException e) {
					/** */
				}
			}

			if (!prognoseEmpfangen) {
				throw new BisInterfaceException(
						"Die Prognose konnte nicht erfolgreich ausgefhrt werden");
			}

			return new BaustellenPrognose(antwort
					.getDatum(OdBaustellenSimulation.Aspekte.PrognoseNormal
							.getAspekt()));

		} catch (AnmeldeException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (DatensendeException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return null;
	}

}
