/*
 * Segment 2 (KEx),  SWE 2.BW-ISIS
 * Copyright (C) 2007 BitCtrl Systems GmbH 
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.kex.isis.isis.impl;

import com.bitctrl.util.Interval;

import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.kex.isis.isis.BisInterfaceException;
import de.bsvrz.kex.isis.isis.BisInterface.BaustellenZustand;
import de.bsvrz.sys.funclib.bitctrl.kalender.Ereigniskalender;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.kalender.objekte.Ereignis;
import de.bsvrz.sys.funclib.bitctrl.modell.kalender.objekte.EreignisTyp;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;

/**
 * Factory fr die Baustellenereignisse.
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: BaustellenEreignisFactory.java 10546 2008-07-17 11:25:17Z gieseler $
 */
public final class BaustellenEreignisFactory {

	/** das Singleton der Klasse. */
	private static BaustellenEreignisFactory singleton;

	/** die PID des Ereignistyps f&uuml;r die Baustellenereignisse. */
	public static final String PID_BAUSTELLENEREIGNIS = "ereignisTyp.Baustelle";

	/**
	 * Gibt das einzige Objekt der Klasse zur&uuml;ck.
	 * 
	 * @return das Singleton der Klasse.
	 */
	public static BaustellenEreignisFactory getInstance() {
		if (singleton == null) {
			singleton = new BaustellenEreignisFactory();
		}

		return singleton;
	}

	/** Diese Klasse versendet die Betriebsmeldungen. */
	protected MessageSender nachrichtenSender = null;

	/** das Typobjekt der Baustellenereignisse. */
	private EreignisTyp ereignisTyp = null;

	/**
	 * Privater Konstruktor.
	 */
	private BaustellenEreignisFactory() {
		SystemObject etdav = ObjektFactory.getInstanz().getVerbindung()
				.getDataModel().getObject(
						BaustellenEreignisFactory.PID_BAUSTELLENEREIGNIS);
		if (etdav != null) {
			ereignisTyp = (EreignisTyp) ObjektFactory.getInstanz()
					.getModellobjekt(etdav);
		} else {
			ereignisTyp = null;
		}
	}

	/**
	 * Legt ein Baustellenereignis f&uuml;r die Baustelle an, wenn das f&uuml;r
	 * den aktuellen Zustand parametriert ist.
	 * 
	 * @param baustelle
	 *            Baustelle
	 * @return angelegtes Ereignis oder null, wenn kein Ereignis angelegt wurde
	 * @throws BisInterfaceException
	 *             wenn das Ereignis nicht angelegt wrden konnte
	 */
	public Ereignis anlegenEreignis(final Baustelle baustelle)
			throws BisInterfaceException {
		Ereignis ereignis = null;

		if (ereignisTyp == null) {
			throw new BisInterfaceException("Der EreignisTyp '"
					+ BaustellenEreignisFactory.PID_BAUSTELLENEREIGNIS
					+ "' existiert nicht im Datenkatalog!");
		}

		if (!istEreignis(baustelle.getBaustellenZustand())) {
			return null;
		}

		try {
			String pid = bildePIDEreignis(baustelle);
			String name = bildeNameEreignis(baustelle);
			String beschreibung = bildeBeschreibungEreignis(baustelle);
			String quelle = AbstractBIS.BISAPPNAME;
			Interval intervall = new Interval(baustelle.getStartZeitpunkt(),
					baustelle.getStartZeitpunkt() + baustelle.getDauer());
			ereignis = Ereigniskalender.getInstanz().anlegenEreignis(pid, name,
					beschreibung, ereignisTyp, intervall, quelle);
		} catch (ConfigurationChangeException e) {
			throw new BisInterfaceException(
					"Das Ereignis konnte nicht angelegt werden: "
							+ e.getMessage());
		} catch (AnmeldeException e) {
			throw new BisInterfaceException(
					"Das Ereignis konnte nicht angelegt werden: "
							+ e.getMessage());
		} catch (DatensendeException e) {
			throw new BisInterfaceException(
					"Das Ereignis konnte nicht angelegt werden: "
							+ e.getMessage());
		}

		return ereignis;
	}

	/**
	 * Erzeugt einen Beschreibungstext f&uuml;r das Ereignis.
	 * 
	 * @param baustelle
	 *            Baustelle
	 * @return Beschreigungstext
	 * @throws BisInterfaceException
	 *             bei ung&uuml;ltigem Baustellenzustand
	 */
	private String bildeBeschreibungEreignis(Baustelle baustelle)
			throws BisInterfaceException {
		String statustext = baustelle.getDavBaustellenStatus(
				baustelle.getBaustellenZustand()).getName();

		return "Die Baustelle " + baustelle.getName()
				+ " wurde in den Zustand " + statustext + " berfhrt";
	}

	/**
	 * Bildet den Namen f&uuml;r das Ereignis. Der Name wird aus dem Namen der
	 * Baustelle und der Beschreibung des Zustandes gebildet.
	 * 
	 * @param baustelle
	 *            Baustelle
	 * @return Name f&uuml;r das Baustellenereignis
	 * @throws BisInterfaceException
	 *             bei ung&uuml;ltigem Baustellenzustand
	 */
	private String bildeNameEreignis(Baustelle baustelle)
			throws BisInterfaceException {
		return baustelle.getName()
				+ " "
				+ baustelle.getDavBaustellenStatus(
						baustelle.getBaustellenZustand()).getName();
	}

	/**
	 * Bildet die PID f&uuml;r das Baustellen-Ereignis. DIe PID wird wie folgt
	 * gebildet.
	 * <p>
	 * 'ereignis.baustellenstatus.Baustelle-PID'
	 * </p>
	 * mit <p/> 'ereignis': fester Prefix. <p/> 'baustellenstatus': texttuelle
	 * Beschreibung des Status des Baustelle <p/> 'Baustelle-PID': PID der
	 * Baustelle
	 * <p>
	 * Beispiel: ereignis.g&uuml;ltig.BaustelleA5<p/>
	 * </p>
	 * 
	 * @param baustelle
	 *            Baustelle
	 * @return PID Baustellen-Ereignis
	 * @throws BisInterfaceException
	 *             bei ung&uuml;ltigem Baustellenzustand
	 */
	private String bildePIDEreignis(Baustelle baustelle)
			throws BisInterfaceException {
		return "ereignis."
				+ baustelle.getDavBaustellenStatus(
						baustelle.getBaustellenZustand()).getName() + "."
				+ baustelle.getPid() + "." + System.currentTimeMillis();
	}

	/**
	 * Test, ob der Baustellenzustand das Anlegen eines Ereignisses erfordert.
	 * 
	 * @param zustand
	 *            Baustellenzustand
	 * @return true, wenn Ereignis sonst false
	 */
	public boolean istEreignis(BaustellenZustand zustand) {
		if (zustand == BaustellenZustand.ENTWORFEN) {
			return ISISParameter.getInstance().ereignisBeiEntworfen;
		}

		if (zustand == BaustellenZustand.GUELTIG) {
			return ISISParameter.getInstance().ereignisBeiGueltig;
		}

		if (zustand == BaustellenZustand.GEPLANT) {
			return ISISParameter.getInstance().ereignisBeiGeplant;
		}

		if (zustand == BaustellenZustand.STORNIERT) {
			return ISISParameter.getInstance().ereignisBeiStorniert;
		}

		return false;
	}
}
