/*
 * Segment 2 (KEx),  SWE 2.BW-ISIS
 * Copyright (C) 2007 BitCtrl Systems GmbH 
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.kex.isis.isis.impl;

import com.bitctrl.util.Interval;

import de.bsvrz.iav.gllib.gllib.Ganglinie;
import de.bsvrz.iav.gllib.gllib.dav.GanglinieMQ;
import de.bsvrz.kex.isis.isis.GanglinienInterface;

/**
 * Implementation des {@link GanglinienInterface}<br>
 * 
 * Folgende Besonderheit ist zu beachten: Da die Ganglinienprognose nur auf
 * St&uuml;tzstellen basiert, die Momentaufnahmen zum Zeitpunkt der
 * St&uuml;tzstelle sind, wird zur Herstellung der geforderten Daten auf der
 * Basis der Approximation eine Mittellung &uuml;ber die Intervalle
 * durchgef&uuml;hrt und der berechnete Mittelwert zur&uuml;ckgeliefert, d.h.
 * Die Ganglinie wird &uuml;ber die Ganglinienbibliothek f&uuml;r die geforderte
 * Intervallbreite (1h) interpoliert. Der von der Klasse gelieferte Wert[n]
 * ergibt sich aus dem Mittelwert der Ganglinie Wert[n-1] bis Wert[n]. Die zur
 * Bildung des Mittelwertes benutzte Granularit&auml;t ist &uuml;ber die
 * Konstante MITTELWERT_GRANULARITAET festgelegt.
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: Ganglinien.java 7485 2008-03-17 14:49:20Z gieseler $
 * 
 */

public class Ganglinien implements GanglinienInterface {

	/** die Intervallbreite der Ganglinien (1h). */
	private static final long INTERVALLBREITE = 60 * 60 * 1000;

	/** von der Ganglinienprognose gelieferte Ganglinie . */
	private final GanglinieMQ prognoseganglinie;

	/**
	 * Konstruktor aus von der Ganglinienprognose gelieferten Ganglinie.
	 * 
	 * @param mqganglinie
	 *            von der Ganglinienprognose gelieferte Ganglinie
	 */
	public Ganglinien(final GanglinieMQ mqganglinie) {
		this.prognoseganglinie = mqganglinie;
	}

	/**
	 * Erzeugt die bereitzustellenden St&uuml;tzstellen aus den
	 * St&uuml;tzstellen der Ganglinienbibliothek.
	 * 
	 * @param ganglinie
	 *            St&uuml;tzstellen der Ganglinienprognose
	 * @return Feld von int
	 */
	private int[] erzeugeBISStuetzstellen(Ganglinie<Double> ganglinie) {
		Interval aintervall = prognoseganglinie.getIntervall();

		int anz = (int) (aintervall.getLength() / INTERVALLBREITE);

		int[] bisstuetzstellen = new int[anz];

		for (int i = 0; i < anz; i++) {
			Interval bintervall = new Interval(aintervall.getStart()
					+ (i * INTERVALLBREITE), aintervall.getStart()
					+ ((i + 1) * INTERVALLBREITE));
			if (ganglinie.isValid(bintervall)) {
				bisstuetzstellen[i] = (int) (ganglinie.getApproximation()
						.integral(bintervall) / INTERVALLBREITE);
			} else {
				bisstuetzstellen[i] = -1;
			}
		}

		return bisstuetzstellen;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getDatum()
	 */
	public long getDatum() {
		return prognoseganglinie.getIntervall().getStart();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getQKfz()
	 */
	public int[] getQKfz() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieQKfz());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getQLkw()
	 */
	public int[] getQLkw() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieQLkw());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getQPkw()
	 */
	public int[] getQPkw() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieQPkw());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getVKfz()
	 */
	public int[] getVKfz() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieVKfz());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getVLkw()
	 */
	public int[] getVLkw() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieVLkw());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.GanglinienInterface#getVPkw()
	 */
	public int[] getVPkw() {
		return erzeugeBISStuetzstellen(prognoseganglinie.getGanglinieVPkw());
	}

}
