/*
 * Segment 2 (KEx),  SWE 2.BW-ISIS
 * Copyright (C) 2007 BitCtrl Systems GmbH 
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weienfelser Strae 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.kex.isis.isis.impl;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.GregorianCalendar;
import java.util.List;

import com.bitctrl.Constants;
import com.bitctrl.util.Interval;

import de.bsvrz.iav.gllib.gllib.dav.GanglinieMQ;
import de.bsvrz.iav.gllib.gllib.dav.Ganglinienprognose;
import de.bsvrz.iav.gllib.gllib.dav.GlProgAnfrage;
import de.bsvrz.iav.gllib.gllib.dav.GlProgAntwortEvent;
import de.bsvrz.iav.gllib.gllib.dav.GlProgAntwortListener;
import de.bsvrz.kex.isis.isis.BisInterfaceException;
import de.bsvrz.kex.isis.isis.GanglinienInterface;
import de.bsvrz.kex.isis.isis.MessQuerschnittInterface;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.verkehr.objekte.MessQuerschnittAllgemein;

/**
 * Implementierung MessQuerschnittinterface.
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: MessQuerschnitt.java 9185 2008-05-26 12:38:47Z gieseler $
 * 
 */

public class MessQuerschnitt implements MessQuerschnittInterface,
		GlProgAntwortListener {

	/**
	 * Das zugeordnete Objekt der Konfiguration.
	 */
	MessQuerschnittAllgemein messQuerschnitt;

	/**
	 * Synchronisationsobjekt f&uuml;r die Ganglinienprognoseanfrage.
	 */
	private final Object progWait = new Object();

	/**
	 * /** von der Ganglinienprognose gelieferte Ganglinien.
	 */
	private Collection<GanglinieMQ> mqganglinien = null;

	/**
	 * Konstruktor auf der Basis eines Modellobjektes.
	 * 
	 * @param mq
	 *            Das zugeordnete Objekt der Konfiguration.
	 */
	public MessQuerschnitt(MessQuerschnittAllgemein mq) {
		super();
		this.messQuerschnitt = mq;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.iav.gllib.gllib.dav.GlProgAntwortListener#antwortEingetroffen(de.bsvrz.iav.gllib.gllib.dav.GlProgAntwortEvent)
	 */
	public void antwortEingetroffen(final GlProgAntwortEvent e) {
		mqganglinien = e.getGanglinien();
		synchronized (progWait) {
			progWait.notify();
		}
	}

	/**
	 * Bildet ein Abfrageintervall f&uuml; die Ganglinienprognose. Der
	 * vorgegebene Zeitbereich wird auf ganze Tage beschr&auml;nkt, da das
	 * BIS-Interface nur Tagesganglinien bereitstellt.
	 * 
	 * @param vonZeitpunkt
	 *            Startzeitpunkt, ab dem die Ganglinien ermittelt werden sollen.
	 *            (in ms seit dem 1.1.1970 UTC)
	 * @param bisZeitpunkt
	 *            Endzeitpunkt, bis zu dem die Ganglinien ermittelt werden
	 *            sollen. (in ms seit dem 1.1.1970 UTC)
	 * @return Intervall vom 1. vollstndigen Tag 0:00 Uhr bis zum letzten
	 *         vollstndigen Tag 24:00 Uhr innerhalb des vorgegebenen
	 *         Zeitbereiches
	 */
	private List<Interval> bildeAbfrageIntervalle(long vonZeitpunkt,
			long bisZeitpunkt) {

		List<Interval> intervalle = new ArrayList<Interval>();

		GregorianCalendar calvon = new GregorianCalendar();
		calvon.setTimeInMillis(vonZeitpunkt);
		calvon.set(Calendar.HOUR_OF_DAY, 0);
		calvon.set(Calendar.MINUTE, 0);
		calvon.set(Calendar.SECOND, 0);
		calvon.set(Calendar.MILLISECOND, 0);

		if (vonZeitpunkt != calvon.getTimeInMillis()) {
			// Startzeit ist nicht Tagesanfang
			calvon.add(Calendar.DAY_OF_MONTH, +1);
		}

		GregorianCalendar calbis = new GregorianCalendar();
		calbis.setTimeInMillis(bisZeitpunkt);
		calbis.set(Calendar.HOUR_OF_DAY, 24);
		calbis.set(Calendar.MINUTE, 0);
		calbis.set(Calendar.SECOND, 0);
		calbis.set(Calendar.MILLISECOND, 0);
		if (bisZeitpunkt != calbis.getTimeInMillis()) {
			// Endezeit ist nicht Tagesende
			calbis.add(Calendar.DAY_OF_MONTH, -1);
		}

		// Sonderbehandlung DAV kann keine '0' als Zeitstempel
		if (calvon.getTimeInMillis() == 0) {
			calvon.setTimeInMillis(1);
		}

		if (calbis.getTimeInMillis() == 0) {
			calbis.setTimeInMillis(1);
		}

		for (long t = calvon.getTimeInMillis(); t < calbis.getTimeInMillis(); t += Constants.MILLIS_PER_DAY) {
			intervalle.add(new Interval(t, t + Constants.MILLIS_PER_DAY));
		}

		return intervalle;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws BisInterfaceException
	 * 
	 * @see de.bsvrz.kex.isis.isis.MessQuerschnittInterface#getGanglinien(long,
	 *      long)
	 */
	public GanglinienInterface[] getGanglinien(final long vonZeitpunkt,
			final long bisZeitpunkt) throws BisInterfaceException {

		mqganglinien = null;
		List<GanglinienInterface> ganglinien = new ArrayList<GanglinienInterface>();

		List<Interval> abfrageIntervalle = bildeAbfrageIntervalle(vonZeitpunkt,
				bisZeitpunkt);

		// if (abfrageIntervall.getLength() < Constants.MILLIS_PER_DAY) {
		// throw new BisInterfaceException(
		// "Der Abfragezeitraum umfasst keinen ganzen Tag!");
		// }

		List<GlProgAnfrage> anfragen = new ArrayList<GlProgAnfrage>();

		for (Interval intervall : abfrageIntervalle) {
			GlProgAnfrage anfrage = new GlProgAnfrage(messQuerschnitt,
					intervall, false);
			anfragen.add(anfrage);
		}

		String absender = "BIS-INTERFACE";
		Ganglinienprognose prog = Ganglinienprognose.getInstanz();
		prog.addAntwortListener(this);
		try {
			prog.sendeAnfrage(absender, anfragen);
			synchronized (progWait) {
				progWait.wait(60000);
			}

			if (mqganglinien != null) {
				for (GanglinieMQ mqganglinie : mqganglinien) {
					if (istGueltig(mqganglinie)) {
						// TODO
						// fr alle if(mqganglinie.getIntervalle())

						ganglinien.add(new Ganglinien(mqganglinie));
						// ganglinien[index] = new Ganglinien(mqganglinie);
					}
				}
			}
		} catch (DatensendeException e) {
			throw new BisInterfaceException(e.getMessage());
		} catch (InterruptedException e) {
			throw new BisInterfaceException(
					"Timeout beim Warten auf die Antwort der Ganglinienprognose");
		}

		return ganglinien.toArray(new GanglinienInterface[ganglinien.size()]);
	}

	/**
	 * Gibt das Messquerschnittsobjekt zur&uuml;ck.
	 * 
	 * @return Messquerschnittsobjekt
	 */
	public MessQuerschnittAllgemein getMessQuerschnitt() {
		return messQuerschnitt;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.MessQuerschnittInterface#getName()
	 */
	public String getName() {
		return messQuerschnitt.getName();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see de.bsvrz.kex.isis.isis.MessQuerschnittInterface#getPid()
	 */
	public String getPid() {
		return messQuerschnitt.getPid();
	}

	/**
	 * Test, ob die Ganglinie g&uuml;ltige Werte enth&auml;lt.
	 * 
	 * @param mqganglinie
	 *            Ganglinie
	 * @return true, wenn g&uuml;ltig sonst false
	 */
	private boolean istGueltig(GanglinieMQ mqganglinie) {
		return !mqganglinie.getGanglinieQKfz().getIntervalle().isEmpty();
	}
}
