/*
 * @(#)Auftrag.java   11.07.12
 *
 * Copyright (c) 2008 by inovat, Dipl.-Ing. H. C. Kni
 * ALL RIGHTS RESERVED.
 *
 * THIS SOFTWARE IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A PARTICULAR  PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL inovat OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES  (INCL., BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR  PROFITS;
 * OR BUSINESS INTERRUPTION)  HOWEVER  CAUSED  AND  ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,  OR TORT (INCL.
 * NEGLIGENCE OR OTHERWISE)  ARISING  IN  ANY  WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */



package de.bsvrz.kex.tls.deabruf;

//~ NICHT JDK IMPORTE =========================================================

import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

import java.text.SimpleDateFormat;
import java.util.Date;

//~ JDK IMPORTE ===============================================================

//~ KLASSEN ===================================================================

/**
 * Enthlt alle Informationen fr einen DE-Abruf-Auftrag.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kni (HCK)
 * @version $Revision: 630 $ / $Date: 2012-07-17 16:53:25 +0200 (Di, 17 Jul 2012) $ / ($Author: LG $)
 */
public class Auftrag implements ClientSenderInterface {
    private static final String ASPEKT = "asp.tlsAbruf";
    private static final String ATG    = "atg.tlsGloDeBlockAbruf";

    /** DebugLogger fr Debug-Ausgaben. */
    private static final Debug debug = Debug.getLogger();

    /** Datumsformatierung fr Ausgabe toString(). */
    private static SimpleDateFormat DATUM_FORMATIERER = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss,SSS");

    /** Datenverteilerverbindung. */
    private static ClientDavInterface _dav;

    /** Die Konfiguration des DaV. */
    private static DataModel _davKonf;

    //~ FELDER ================================================================

    /** Zeitpunkt der geplanten Ausfhrung fr diesen Auftrag in Millisekunden. */
    private long _ausfuehrungsZeitPunkt = 0;

    /** Legt fest, ob das Sendeerlaubnis fr den Auftrag vorhanden ist. */
    private boolean _sendeErlaubnis = false;

    /** Legt fest, ob der Auftragt aktiviert ist (Wert == true). */
    private boolean _aktiviert;

    /** DE PID. */
    private String _dePid;

    /** ID des Parameters. */
    private int _id;

    /** Typ des Parameters. */
    private int _typ;

    /** ZeitIntervall in Millis zwischen den Ausfhren der Auftrge. */
    private long _zeitIntervallInMillis;

    //~ KONSTRUKTOREN  (und vom Konstruktor verwendete Methoden) ==============

    /**
     * Erzeugt eine Instanz der Klasse Auftrag.
     *
     * @param dav Datenverteilerverbindung.
     * @param dePid De PID.
     * @param id ID des Parameters.
     * @param typ Typ des Parameters.
     * @param ausfuehrungsZeitPunkt Zeitpunkt der geplanten Ausfhrung fr diesen Auftrag in Millisekunden.
     * @param zeitIntervallInMillis ZeitIntervall in Millis zwischen den Ausfhren der Auftrge.
     * @param aktiviert Legt fest, ob der Auftragt aktiviert ist (Wert == true).
     */
    public Auftrag(ClientDavInterface dav, String dePid, int id, int typ, long ausfuehrungsZeitPunkt, long zeitIntervallInMillis, boolean aktiviert) {
        _dav                   = dav;
        _davKonf               = _dav.getDataModel();
        _dePid                 = dePid;
        _id                    = id;
        _typ                   = typ;
        _ausfuehrungsZeitPunkt = ausfuehrungsZeitPunkt;
        _zeitIntervallInMillis = zeitIntervallInMillis;
        _aktiviert             = aktiviert;
    }

    //~ METHODEN ==============================================================

    /**
     * Auftrag wird ausgefhrt:
     * <p>
     *     deSys-Objekt wird als Sender an DaV angemeldet;
     * <p>
     *     DE-Abruf-Daten werden an DaV gesendet;
     * <p>
     *     deSys-Objekt wird von DaV abgemeldet.
     *
     *
     * @return true, falls die Daten erfolgreich gesendet wurden, false - sonst.
     */
    public boolean ausfuehren() {

        // -------------------------------------------------
        // Falls der Auftrag nicht aktiviert ist:
        // -------------------------------------------------
        if (!_aktiviert) {
            debug.info(String.format("Der Auftrag ist nicht aktiviert: %s", this));

            return false;
        }

        // -------------------------------------------------
        // Initialisiere DaV-Daten:
        // -------------------------------------------------
        SystemObject    so                = _davKonf.getObject(_dePid);
        AttributeGroup  atg               = _davKonf.getAttributeGroup(ATG);
        Aspect          asp               = _davKonf.getAspect(ASPEKT);
        DataDescription datenBeschreibung = new DataDescription(atg, asp);

        // -------------------------------------------------
        // Anmeldung als Sender:
        // -------------------------------------------------
        try {
            _dav.subscribeSender(this, so, datenBeschreibung, SenderRole.sender());
        }
        catch (OneSubscriptionPerSendData e) {
            debug.warning(String.format("Datenidentifikation ist bereits angemeldet. Eventuell Senke nicht vorhanden: <%s>, <%s>", so, datenBeschreibung), e);

            return false;
        }
        catch (Exception e) {
            debug.warning(String.format("Fehler bei der Anmeldung der Daten als Sender: <%s>, <%s>", so, datenBeschreibung), e);

            return false;
        }

        // -------------------------------------------------
        // Warte auf SendeErlaubnis:
        // Ein paar mal versuchen, ob Sendeerlaubnis bereits vorliegt...
        // -------------------------------------------------
        int i = 0;

        while (i < 10) {
            if (_sendeErlaubnis) {
                break;
            } else {

                // Pause:
                try {
                    Thread.sleep(500);
                }
                catch (Exception ignored) {}
            }

            i++;
        }

        // -------------------------------------------------
        // Sende DaV-Daten:
        // -------------------------------------------------
        boolean erfolgreich = false;

        // Sende die Daten:
        if (!_sendeErlaubnis) {
            debug.warning(String.format("Sendeerlaubnis ist nicht vorhanden: <%s>, <%s>", so, datenBeschreibung));
        } else {

            // Erstelle DaV-Daten:
            Data daten = _dav.createData(atg);

            daten.getItem("JobNummer").asUnscaledValue().set(1);
            daten.getItem("ID").asUnscaledValue().set(_id);
            daten.getItem("Typ").asUnscaledValue().set(_typ);

            ResultData ergebnis = new ResultData(so, datenBeschreibung, System.currentTimeMillis(), daten, false);

            // Sende DaV-Daten:
            try {
                _dav.sendData(ergebnis);
                erfolgreich = true;
            }
            catch (Exception e) {
                debug.warning(String.format("Fehler beim Senden der Daten an DaV: <%s>, <%s>", so, datenBeschreibung), e);
            }
        }

        // -------------------------------------------------
        // Abmeldung als Sender:
        // -------------------------------------------------
        try {
            _dav.unsubscribeSender(this, so, datenBeschreibung);
        }
        catch (Exception e) {
            debug.warning(String.format("Fehler bei der Abmeldung der Daten als Sender: <%s>, <%s>", so, datenBeschreibung), e);
        }

        return erfolgreich;
    }

    @Override
    public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
        _sendeErlaubnis = (state == START_SENDING);

        String status = "unbekannter Status";

        if (state == START_SENDING) {
            status = "START_SENDING";
        } else if (state == STOP_SENDING) {
            status = "STOP_SENDING";
        } else if (state == STOP_SENDING_NO_RIGHTS) {
            status = "STOP_SENDING_NO_RIGHTS";
        } else if (state == STOP_SENDING_NOT_A_VALID_SUBSCRIPTION) {
            status = "STOP_SENDING_NOT_A_VALID_SUBSCRIPTION";
        }

        debug.fine(String.format("SO: %s%nDatenbeschreibung: %s%nStatus: %s%n", object, dataDescription, status));
    }

    //~ GET METHODEN ==========================================================

    /**
     * Liefert den Zeitpunkt der geplanten Ausfhrung fr diesen Auftrag in Millisekunden seit 1.1.1970
     *
     * @return Zeitpunkt der geplanten Ausfhrung fr diesen Auftrag in Millisekunden seit 1.1.1970
     */
    public long getAusfuehrungsZeitPunkt() {
        return _ausfuehrungsZeitPunkt;
    }

    public String getDePid() {
        return _dePid;
    }

    public int getId() {
        return _id;
    }

    public int getTyp() {
        return _typ;
    }

    /**
     * Stellt die Zeitdauer als String dar.
     *
     * @param zeitDauer Zeitdauer, die als String dargestellt werden soll.
     *
     * @return Zeitdauer als String "hh:mm:ss".
     */
    private static String getZeitdauerAlsString(long zeitDauer) {
        long  zeitHr  = zeitDauer / (1000 * 60 * 60);
        long  zeitMin = (zeitDauer - zeitHr * 60 * 60 * 1000) / (1000 * 60);
        float zeitSek = (zeitDauer - zeitHr * 60 * 60 * 1000 - zeitMin * 60 * 1000) / 1000f;

        return String.format("%02d:%02d:%02d", zeitHr, zeitMin, Math.round(zeitSek));
    }

    public long getZeitIntervallInMillis() {
        return _zeitIntervallInMillis;
    }

    public boolean isAktiviert() {
        return _aktiviert;
    }

    @Override
    public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
        return true;
    }

    //~ METHODEN ==============================================================

    /**
     * Vergleicht diesen Auftrag mit einem anderen Auftrag auf Gleichheit wobei der  AusfuehrungsZeitPunkts des neuen Auftrags
     * weiter in der Zukunft liegen muss, damit sich keine Auftraege mit alten Zeitstempeln (z. B. nachgelieferte Daten)
     * einreihen koennen.
     *
     * @param auftragNeu Der zu vergleichende Auftrag
     * @return <code>true</code>, wenn dieser Auftrag gleich aber aelter als zu vergleichendene Auftrag ist, sonst <code>false</code>.
     */
    public boolean istAelterAls(Auftrag auftragNeu) {
        boolean istGleichenAuftrag = ((this._id == auftragNeu._id) &&    //
                                      (this._typ == auftragNeu._typ) &&  //
                                      (this._dePid.equals(auftragNeu._dePid)));

        if ((istGleichenAuftrag) && (this._ausfuehrungsZeitPunkt < auftragNeu._ausfuehrungsZeitPunkt)) {
            return true;
        }

        return false;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();

        sb.append("Auftrag");
        sb.append("{_ausfuehrungsZeitPunkt=").append(DATUM_FORMATIERER.format(new Date(_ausfuehrungsZeitPunkt)));
        sb.append(", _dePid=").append(_dePid);
        sb.append(", _id=").append(_id);
        sb.append(", _typ=").append(_typ);
        sb.append(", _zeitIntervallInMillis='").append(getZeitdauerAlsString(_zeitIntervallInMillis)).append('\'');
        sb.append(", _aktiviert=").append(_aktiviert ? "ja" : "nein");
        sb.append('}');

        return sb.toString();
    }
}


//~Formatiert mit 'inovat Kodierkonvention' am 16.07.12
