/*
 * @(#)AuftragsListe.java   11.07.12
 *
 * Copyright (c) 2008 by inovat, Dipl.-Ing. H. C. Kni
 * ALL RIGHTS RESERVED.
 *
 * THIS SOFTWARE IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A PARTICULAR  PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL inovat OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES  (INCL., BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR  PROFITS;
 * OR BUSINESS INTERRUPTION)  HOWEVER  CAUSED  AND  ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,  OR TORT (INCL.
 * NEGLIGENCE OR OTHERWISE)  ARISING  IN  ANY  WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */



package de.bsvrz.kex.tls.deabruf;

//~ JDK IMPORTE ===============================================================

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

//~ KLASSEN ===================================================================

/**
 * Singletonklasse, welche die Auftragsliste fr den Abruf DE_Fehler verwaltet.
 * <p/>
 * Die Auftragsliste ermglicht das Hinzufgen von Auftrgen ({@link Auftrag} sowie die Entnahme des aktuell
 * auszufhrenden Auftrags. Die Zugriffe auf die Auftragsliste ist intern synchronisiert.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Dipl.-Ing. Hans Christian Kni (HCK)
 * @version $Revision: 630 $ / $Date: 2012-07-17 16:53:25 +0200 (Di, 17 Jul 2012) $ / ($Author: LG $)
 */
public class AuftragsListe {
    private static AuftragsListe       dieEinzigeInstanz = new AuftragsListe();
    private static final List<Auftrag> _listeAuftraege   = new ArrayList<Auftrag>();

    //~ KONSTRUKTOREN  (und vom Konstruktor verwendete Methoden) ==============

    /**
     * Privater Konstruktor.
     */
    private AuftragsListe() {}

    //~ METHODEN ==============================================================

    /**
     * Fgt einen neuen Auftrag zur Liste der Auftrge hinzu. Die Liste wird intern so sortiert, dass die Methode {@link
     * #holeNaechstenAuftrag()} immer den zum aktuellen Zeitpunkt nchsten auszufhrenden Auftrag zurckliefert.
     *
     * @param auftrag Neuer Auftrag, der in die Liste der Auftrge eingetragen werden soll.
     */
    public void addAuftrag(Auftrag auftrag) {
        synchronized (_listeAuftraege) {

            // Zuerst bisherigen (zeitlich alten) Auftrag entfernen...
            for (Auftrag auftragAlt : _listeAuftraege) {
                if (auftragAlt.istAelterAls(auftrag)) {
                    _listeAuftraege.remove(auftragAlt);

                    break;
                }
            }

            // .. und dann den zeitlich aktuellen Auftrag eintragen
            _listeAuftraege.add(auftrag);
            Collections.sort(_listeAuftraege, new Comparator<Auftrag>() {
                public int compare(Auftrag auftrag1, Auftrag auftrag2) {

                    // Zeitliche Sortierung der Auftrge nach Ausfhrungszeitpunkt
                    if (auftrag1.getAusfuehrungsZeitPunkt() < auftrag2.getAusfuehrungsZeitPunkt()) {
                        return -1;
                    }

                    if (auftrag1.getAusfuehrungsZeitPunkt() > auftrag2.getAusfuehrungsZeitPunkt()) {
                        return 1;
                    }

                    return 0;
                }
            });
        }
    }

    //~ GET METHODEN ==========================================================

    /**
     * Liefert die Instanz des Singletons.
     *
     * @return Instanz des Singletons
     */
    public static AuftragsListe getInstanz() {
        return dieEinzigeInstanz;
    }

    //~ METHODEN ==============================================================

    /**
     * Gibt zurck, ob noch Auftrge (aktuell oder zuknftig) anstehen.
     *
     * @return <code>true</code>, wenn noch Auftrge (aktuell oder zuknftig) anstehen, andernfalls <code>false</code>.
     */
    public boolean hatNochAuftraege() {
        synchronized (_listeAuftraege) {
            return _listeAuftraege.size() != 0;
        }
    }

    /**
     * Liefert eine Kopie der Auftragsliste, z.B. fr Tests oder Debugausgaben des aktuellen Stands zu einem
     * bestimmten Zeitpunkt.
     *
     * @return Kopie der Auftragsliste.
     */
    public List<Auftrag> holeKopieAuftragsListe() {
        List<Auftrag> listeKopieAuftraege = new ArrayList<Auftrag>();

        synchronized (_listeAuftraege) {
            for (Auftrag auftrag : _listeAuftraege) {
                listeKopieAuftraege.add(auftrag);
            }
        }

        return listeKopieAuftraege;
    }

    /**
     * Liefert den, bezogen auf den aktuellen Zeitpunkt, nchsten auszufhrenden Auftrag zurck.
     *
     * @return Nchster auszufhrender Auftrag oder <code>null</code>, wenn kein Auftrag aktuell auszufhren ist. Der
     *         zurckgegebene Auftrag wird entfernt und kann nicht noch einmal abgerufen werden.
     */
    public Auftrag holeNaechstenAuftrag() {
        Auftrag naechsterAuftrag = null;

        synchronized (_listeAuftraege) {
            if ((_listeAuftraege.size() > 0) && (_listeAuftraege.get(0).getAusfuehrungsZeitPunkt() < System.currentTimeMillis())) {
                naechsterAuftrag = _listeAuftraege.get(0);
                _listeAuftraege.remove(0);
            }
        }

        return naechsterAuftrag;
    }

    /**
     * Lsche alle Auftrge aus der Liste.
     */
    public void loescheAlleAuftraege() {
        _listeAuftraege.clear();
    }
}


//~Formatiert mit 'inovat Kodierkonvention' am 11.07.12
