/*
 * @(#)DeAbruf.java   11.07.12
 *
 * Copyright (c) 2008 by inovat, Dipl.-Ing. H. C. Kni
 * ALL RIGHTS RESERVED.
 *
 * THIS SOFTWARE IS  PROVIDED  "AS IS"  AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF  MERCHANTABILITY  AND  FITNESS  FOR  A PARTICULAR  PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL inovat OR ITS CONTRIBUTORS BE
 * LIABLE FOR ANY  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES  (INCL., BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;  LOSS OF USE, DATA, OR  PROFITS;
 * OR BUSINESS INTERRUPTION)  HOWEVER  CAUSED  AND  ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,  OR TORT (INCL.
 * NEGLIGENCE OR OTHERWISE)  ARISING  IN  ANY  WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */



package de.bsvrz.kex.tls.deabruf;

//~ NICHT JDK IMPORTE =========================================================

import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

//~ KLASSEN ===================================================================

/**
 * Klasse verwaltet den Prozess fr den Abruf DE_Fehler.
 * <p/>
 * Die zu bearbeitenden DEs, sowie ID, Typ und Intervall
 * sind im DaV-Objekt VewTlsDeAbruf, das ber den Aufrufparameter bergeben
 * wird, definiert.
 * <p/>
 * Prozess prft in regelmigen Abstnden (wird ber den Aufrufparameter definiert)
 * nach den DE-Abruf-Auftrgen und fhrt die aktuellen aus.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Liliya Givorgizova (LG)
 * @version $Revision: 630 $ / $Date: 2012-07-17 16:53:25 +0200 (Di, 17 Jul 2012) $ / ($Author: LG $)
 */
public class DeAbruf implements StandardApplication, ClientReceiverInterface {
    private static final String ASPEKT = "asp.parameterSoll";
    private static final String ATG    = "atg.paramTlsDeAbruf";

    /** DebugLogger fr Debug-Ausgaben. */
    private static Debug           debug = Debug.getLogger();
    private static DataDescription datenBeschreibung;

    //~ FELDER ================================================================

    /** Datenverteilerverbindung. */
    private ClientDavInterface _dav;

    /** Die Konfiguration des DaV. */
    private DataModel _davKonf;

    /** Parameter: Wartezeit fr die Auftragsberwachung in Sekunden. */
    private int _parPauseInSekunden;

    /** Parameter: SystemObjekt-PID fr den DE-Abruf. */
    private String _parSoPid;

    /** Wartezeit fr die Auftragsberwachung in Millisekunden. */
    private long _pauseInMillis;

    //~ METHODEN ==============================================================

    /**
     * Fgt einen neuen Auftrag. ltere Auftrage fr eine solche Datenidentifikation werden
     * automatisch ersetzt, so dass je Datenidentifikation maximal ein Auftrag anliegt.
     * Eingefgt werden nur aktivierte Auftrge.
     *
     * @param dePid DE PID.
     * @param id ID.
     * @param typ Typ.
     * @param ausfuehrungsZeitPunkt Zeitpunkt der geplanten Ausfhrung fr diesen Auftrag in Millisekunden.
     * @param zeitIntervallInMillis ZeitIntervall in Millis zwischen den Ausfhren der Auftrge.
     * @param aktiviert Legt fest, ob der Auftragt aktiviert ist (Wert == true).
     */
    private void addAuftrag(String dePid, int id, int typ, long ausfuehrungsZeitPunkt, long zeitIntervallInMillis, boolean aktiviert) {

        // Eingefgt werden nur aktivierte Auftrge:
        if (!aktiviert) {
            return;
        }

        // Auftrag nur erzeugen, wenn zeitIntervallInMillis > 0 ist:
        if (zeitIntervallInMillis > 0) {
            Auftrag auftrag = new Auftrag(_dav, dePid, id, typ, ausfuehrungsZeitPunkt, zeitIntervallInMillis, aktiviert);

            AuftragsListe.getInstanz().addAuftrag(auftrag);

            //
            debug.info(String.format("Eingefgter Auftrag: %s", auftrag));
        }
    }

    @Override
    public void initialize(ClientDavInterface dav) throws Exception {

        // --------------------------------------------------------------
        // DaV-Zugriff:
        _dav     = dav;
        _davKonf = _dav.getDataModel();

        // --------------------------------------------------------------
        // Initialisiere DE-Abruf-Objekt:
        SystemObject soDeAbruf = _davKonf.getObject(_parSoPid);

        if (soDeAbruf == null) {
            debug.error(String.format("Das Objekte mit dem PID <%s> kann nicht ermittelt werden!. Bitte prfen Sie den Aufrufparameter <-soDeAbrufPid>. Programm wird beendet.", _parSoPid));

            throw new RuntimeException("Null-Objekt");
        }

        // --------------------------------------------------------------
        // Initialisiere DaV-Daten:
        AttributeGroup atg = _davKonf.getAttributeGroup(ATG);
        Aspect         asp = _davKonf.getAspect(ASPEKT);

        datenBeschreibung = new DataDescription(atg, asp);

        // --------------------------------------------------------------
        // DaV-Anmeldung als Empfnger:
        try {
            _dav.subscribeReceiver(this, soDeAbruf, datenBeschreibung, new ReceiveOptions(false), ReceiverRole.receiver());
        }
        catch (Exception e) {
            debug.error(String.format("Fehler bei der Anmeldung der Daten als Empfnger: <%s>, <%s>", soDeAbruf, datenBeschreibung), e);

            throw new RuntimeException(e);
        }
    }

    /**
     * Main-Methode.
     *
     * @param args Parameter.
     */
    public static void main(String[] args) {

        //
        DeAbruf deAbruf = new DeAbruf();

        StandardApplicationRunner.run(deAbruf, args);

        // Debug erst hier initialisieren, da erst ab hier verfgbar !!
        debug = Debug.getLogger();

        // Starte den berwachungsprozess:
        deAbruf.starteProzess();
    }

    @Override
    public void parseArguments(ArgumentList argumentList) throws Exception {
        String par;

        // --------------------------------------------------------------
        // Hole den Parameter 'SystemObjekt PID fr den DE-Abruf':
        par = "-soDeAbrufPid";

        try {
            _parSoPid = argumentList.fetchArgument(par).asNonEmptyString();
        }
        catch (Exception e) {
            debug.error(String.format("Bitte die SystemObjekt-PID fr den DE-Abruf als Parameter <%s> bergeben!", par), e);

            throw new RuntimeException(e);
        }

        // --------------------------------------------------------------
        // Hole den Parameter 'Pause in Sekunden':
        par = "-ueberwachungsZeitInSekunden";

        try {
            _parPauseInSekunden = argumentList.fetchArgument(par).intValue();
            _pauseInMillis      = _parPauseInSekunden * 1000;
        }
        catch (Exception e) {
            debug.error(String.format("Bitte die Pause in Sekunden fr die DE-Abruf-berwachung als Parameter <%s> bergeben!", par), e);

            throw new RuntimeException(e);
        }
    }

    /**
     * Starte den berwachungsprozess:
     * regelmig prfe die Auftragsliste und fhre aktuelle Auftrge aus.
     */
    private void starteProzess() {

        // --------------------------------------------------------------
        // debug-Info:
        debug.info(String.format("%n=================================================================%n" + "DE-Abruf gestartet...%n" + "=================================================================%n"));

        // --------------------------------------------------------------
        // Endlosschleife mit der Prfung der Auftragsliste:
        while (true) {
            Auftrag auftrag;

            // --------------------------------------------------------------
            // Falls es einen Auftrag zum Ausfhren in der Liste gibt:
            if ((auftrag = AuftragsListe.getInstanz().holeNaechstenAuftrag()) != null) {

                // -------------------------------------
                // Fge neuen Auftrag mit gleicher Datenidentifikation und nchstem Ausfhrungspunkt:
                long ausfuehrungsZeitPunkt = System.currentTimeMillis() + auftrag.getZeitIntervallInMillis();

                addAuftrag(auftrag.getDePid(), auftrag.getId(), auftrag.getTyp(), ausfuehrungsZeitPunkt, auftrag.getZeitIntervallInMillis(), auftrag.isAktiviert());

                // -------------------------------------
                // Ausfhren den aktuellen Auftrag:
                if (auftrag.ausfuehren()) {
                    debug.info(String.format("Ausgefhrter Auftrag: %s", auftrag));
                } else {
                    debug.warning(String.format("Folgender Auftrag wurde nicht ausgefhrt: %s%n(siehe Meldungen oben)", auftrag));
                }
            }

            // --------------------------------------------------------------
            // Zurzeit ist nichts zu tun, also warten und es dann noch mal versuchen...
            else {
                debug.fine(String.format(                                                                      //
                    "Zur Zeit keine sofort auszufhrenden Auftrge in der Auftragsliste. Warte %d Sekunden.",  //
                    _parPauseInSekunden));

                // Pause
                try {
                    Thread.sleep(_pauseInMillis);
                }
                catch (InterruptedException e) {
                    debug.info("Sleep der Auftragsbearbeitung des DE-Abrufs wurde vorzeitig unterbrochen: ", e);
                }
            }
        }
    }

    @Override
    public void update(ResultData[] results) {
        debug.info(String.format("%n=================================================================%n" + "Erstelle die Liste der Auftrage aus DaV-Daten neu...%n" + "=================================================================%n"));

        // --------------------------------------------------------------
        // Lsche alle Auftrge:
        AuftragsListe.getInstanz().loescheAlleAuftraege();

        // --------------------------------------------------------------
        // Erstelle die Liste der Auftrage aus DaV-Daten neu:
        for (ResultData daten : results) {
            if (daten.hasData()) {
                String  dePid;
                int     id;
                int     typ;
                long    ausfuehrungsZeitPunkt = System.currentTimeMillis();
                long    zeitIntervallInMillis;
                boolean aktiviert;

                //
                Data.Array ar = daten.getData().getArray("DeAbrufParameter");
                Data       data;

                for (int i = 0; i < ar.getLength(); i++) {
                    data                  = ar.getItem(i);
                    dePid                 = data.getReferenceValue("DeReferenz").getSystemObjectPid();
                    id                    = data.getUnscaledValue("ID").intValue();
                    typ                   = data.getUnscaledValue("Typ").intValue();
                    zeitIntervallInMillis = data.getTimeValue("Intervall").getMillis();
                    aktiviert             = (data.getTextValue("aktiviert").getText().equals("Ja"));
                    addAuftrag(dePid, id, typ, ausfuehrungsZeitPunkt, zeitIntervallInMillis, aktiviert);
                }
            }
        }

        debug.info(String.format("%n=================================================================%nMomentan sind %d Auftrge in der Liste erstellt.%n=================================================================%n",  //
                                 AuftragsListe.getInstanz().holeKopieAuftragsListe().size()));
    }
}


//~Formatiert mit 'inovat Kodierkonvention' am 16.07.12
