/*
 * Copyright (c) 2012 by inovat, innovative systeme - verkehr - tunnel - technik, Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.bsvrz.kex.tls.zeitsync.par.CsvLeser
 *
 * de.bsvrz.kex.tls.zeitsync.par.CsvLeser is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.kex.tls.zeitsync.par.CsvLeser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.kex.tls.zeitsync.par.CsvLeser; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * Koelner Strasse 30
 * D-50859 Koeln
 * +49 (0)2234 4301 800
 * info@invat.de
 * www.inovat.de
 */


package de.bsvrz.kex.tls.deabruf.par;

//~ NICHT JDK IMPORTE =========================================================

import de.bsvrz.sys.funclib.debug.Debug;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

//~ KLASSEN ===================================================================

/**
 * Ermglicht den lesenden Zugriff auf Excel-Tabellen im *.csv-Format.
 * <p/>
 * Eine Klasse, mit der mit einem Trennzeichen separierte Werte eingelesen werden knnen. Fr jede Zeile werden die
 * Werte in einem Hashtable abgelegt. Eine Klasse, die die Daten weiterverarbeiten soll, kann dann unter dem Namen der
 * jeweiligen Spalte die Daten aus dem Hashtable auslesen.
 *
 * @author inovat
 * @author Dipl.-Ing. Hans Chrstian Kni (HCK)
 * @version $Revision: 630 $ / $Date: 2012-07-17 16:53:25 +0200 (Di, 17 Jul 2012) $ / ($Author: LG $)
 */
public class CsvLeser {

    // ------------------------------------------------------------------------
    // VARIABLENDEKLARATIONEN
    // ------------------------------------------------------------------------

    /** DebugLogger fr Debug-Ausgaben. */
    private static final Debug debug = Debug.getLogger();

    //~ FELDER ================================================================

    /** Inhalt der aktuell gelesenen Zeile */
    private String _naechsteZeile = null;

    /** Trennzeichen (i.d.R. ';'), durch welches einzelne Zellen innerhalb einer Zeile getrennt sind. */
    private char _begrenzungsZeichen;

    /** Reader, mit der die CSV-Datei gelesen wird. */
    private BufferedReader _csvLeser;

    /** Inhalt der ersten Zeile. */
    private List<String> _ersteZeile;

    //~ KONSTRUKTOREN  (und vom Konstruktor verwendete Methoden) ==============

    // ------------------------------------------------------------------------
    // KONSTRUKTOREN  (und vom Konstruktor verwende Methoden)
    // ------------------------------------------------------------------------

    /**
     * Konstruktor fr den CSV-Datei-Leser.
     *
     * @param leser              Der zu verwendende Leser.
     * @param begrenzungsZeichen Trennzeichen (i.d.R. ';'), durch welches einzelne Zellen innerhalb einer Zeile getrennt
     *                           sind.
     */
    public CsvLeser(Reader leser, char begrenzungsZeichen) {
        this._begrenzungsZeichen = begrenzungsZeichen;
        this._csvLeser           = new BufferedReader(leser);
        this._ersteZeile         = ermitteleSpaltenNamen();
    }

    //~ METHODEN ==============================================================

    /** Schliet die *.csv-Datei. */
    public void close() {
        try {
            _csvLeser.close();
        }

        // kann ignoriert werden, da hasNochMehrZeilen den Reader bereits schliesst
        catch (IOException e) {
            debug.finest("Datei bereits geschlossen (eventuell schon letzte Zeile eingelesen: " + e);
        }
    }

    /**
     * Ermittelt die Spaltennamen der Tabelle (Zelleneintrge der ersten Zeile).
     *
     * @return Spaltennamen der Tabelle.
     */
    private List<String> ermitteleSpaltenNamen() {
        List<String> spaltenNamen = null;

        try {
            String zeile = _csvLeser.readLine();

            spaltenNamen = ermittleZellenEinerZeile(zeile);
        }
        catch (Exception e) {
            e.printStackTrace(System.out);
        }

        return spaltenNamen;
    }

    /**
     * Zerlegt die Zeile in die einzelnen Zellen unter Bercksichtigung von Anfhrungszeichen in Zellen
     *
     * @param zeile aktuelle Zeile der *.csv-Datei.
     *
     * @return Liste mit den Zelleninhalten.
     */
    private List<String> ermittleZellenEinerZeile(String zeile) {
        List<String> zellen = new Vector<String>();
        boolean      anfuehrungsZeichen;
        int          start, ende,
                     index           = 0,
                     maxZeilenLaenge = zeile.length() - 1;

        try {

            // Alle Spalten durchlaufen
            while (index <= maxZeilenLaenge) {
                start              = index;
                anfuehrungsZeichen = false;

                // Inhalt einer Spalte extrahieren
                while (index <= maxZeilenLaenge) {
                    char pruefZeichen = zeile.charAt(index);

                    // Nun wird der nchste Delimiter gesucht, der NICHT
                    // innerhalb von Anfhrungszeichen (") steht. Wenn ein
                    // Anfhrungszeichen gefunden wurde, dann muss der Merker
                    // getoggled werden.
                    if (pruefZeichen == '"') {
                        anfuehrungsZeichen = !anfuehrungsZeichen;

                        // Es befindet sich auf jeden Fall eine gerade Anzahl von
                        // "-Zeichen zwischen den Quotes, so dass nur die am Anfang
                        // und Ende bercksichtigt werden.
                    } else if ((pruefZeichen == _begrenzungsZeichen) &&!anfuehrungsZeichen) {

                        // noinspection BreakStatement
                        break;
                    }

                    index++;
                }

                ende = index;

                // Anfhrungszeichen am Anfang und am Ende gehren nicht zum
                // String und werden deshalb auch nicht beachtet.
                if ((zeile.charAt(start) == '"') && (zeile.charAt(ende - 1) == '"')) {
                    start++;
                    ende--;
                }

                // Der gefundene Text wird in dem Vector gespeichert, ohne jedoch zu
                // vergessen, dass zwei auf einander folgende Anfhrungszeichen ("")
                // durch ein Einzelnes (") zu ersetzen sind.
                zellen.add(zeile.substring(start, ende).replaceAll("\"\"", "\""));
                index++;
            }
        }
        catch (Exception e) {
            debug.error("Fehler beim zerlegen einer Zeile der *.csv-Datei: " + e, e);
        }

        return zellen;
    }

    //~ GET METHODEN ==========================================================

    // ------------------------------------------------------------------------
    // GETTER / SETTER METHODEN
    // ------------------------------------------------------------------------

    /**
     * Liefert die erste Zeile der *.csv-Datei mit der Liste der Spaltennamen.
     *
     * @return Liste der Spaltennamen aus der ersten Zeile der *.csv-Datei.
     */
    public List<String> getErsteZeile() {
        return _ersteZeile;
    }

    // ------------------------------------------------------------------------
    // SONSTIGE METHODEN
    // ------------------------------------------------------------------------

    /**
     * Liefert die Zellen einer Zeile.
     *
     * @return Die Zellen einer Zeile in einer Map. Schlssel ist der Spaltenname, Objekt ist der Inhalt der entsprechenden
     *         Zelle der aktuellen Zeile.
     *         <p/>
     *         Ist keine Zeile mehr vorhanden, wird eine leere Map zurckgeliefert.
     */
    public Map<String, String> getNaechsteZeile() {

        // Map mit den Zellen einer Zeile
        Map<String, String> zellenDerZeile = new HashMap<String, String>();

        // Liest auf jeden Fall die neue Zeile, wenn es eine gibt.

        if (hasNochMehrZeilen()) {

            // Aus der Zeile wird die Hashtable erzeugt.
            List<String> zellen = ermittleZellenEinerZeile(_naechsteZeile.trim());

            for (int i = zellen.size() - 1; i >= 0; i--) {
                zellenDerZeile.put(_ersteZeile.get(i), zellen.get(i));
            }

            // Lscht die Zeile, damit hasNochMehrZeilen auf jeden Fall
            // eine neue Zeile einliest.
            _naechsteZeile = "";
        }

        return zellenDerZeile;
    }

    /**
     * Prft, ob die *.csv-Datei noch weitere Zeilen zum EInlesen hat.
     *
     * @return <code>true</code>, wenn noch Zeilen eingelesen werden knnen, sonst <code>false</code>.
     */
    public boolean hasNochMehrZeilen() {
        boolean hasNochMehrZeilen;

        try {
            if ((_naechsteZeile == null) || _naechsteZeile.trim().equals("")) {
                _naechsteZeile = _csvLeser.readLine();
            }
        }
        catch (Exception e) {
            debug.finest("Ignorierte Exception beim Einlesen: " + e);
        }

        if ((_naechsteZeile == null) || _naechsteZeile.trim().equals("")) {
            close();
            hasNochMehrZeilen = false;
        } else {
            hasNochMehrZeilen = true;
        }

        return hasNochMehrZeilen;
    }

    //~ METHODEN ==============================================================

    // ------------------------------------------------------------------------
    // KANONISCHE METHODEN
    // ------------------------------------------------------------------------

    /**
     * Gibt die String-Reprsentation dieses Objekts aus.
     * <p/>
     * Die Formatierung der Ausgabe kann sich ndern und wird nicht garantiert.
     *
     * @return Aktueller Zustand der Klassenvariablen dieses Objekts.
     */
    public String toString() {
        return "CsvLeser{" + "_begrenzungsZeichen=" + _begrenzungsZeichen + ", _csvLeser=" + _csvLeser + ", _ersteZeile=" + _ersteZeile + ", _naechsteZeile='" + _naechsteZeile + '\'' + '}';
    }
}


//~Formatiert mit 'inovat Kodierkonvention' am 22.07.09
