/*
 * Copyright (c) 2012 by inovat, innovative systeme - verkehr - tunnel - technik, Dipl.-Ing. H. C. Kniss
 *
 * This file is part of de.bsvrz.kex.tls.deabruf.par.VewDeAbrufParameter
 *
 * de.bsvrz.kex.tls.deabruf.par.VewDeAbrufParameter is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.kex.tls.deabruf.par.VewDeAbrufParameter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.kex.tls.deabruf.par.VewDeAbrufParameter; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Contact Information:
 * inovat, Dipl.-Ing. H. C. Kniss
 * Koelner Strasse 30
 * D-50859 Koeln
 * +49 (0)2234 4301 800
 * info@invat.de
 * www.inovat.de
 */



package de.bsvrz.kex.tls.deabruf.par;

//~ NICHT JDK IMPORTE =========================================================

import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.*;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.*;
import java.util.*;

//~ JDK IMPORTE ===============================================================

//~ KLASSEN ===================================================================

/**
 * Klasse erstellt/ergnzt den Parametersatz fr den DE-Abruf.
 * <p/>
 * Die Parameter-Eintrge werden aus einer ber den Aufrufparameter bergebenden
 * csv-Datei ausgelesen.
 * <p/>
 * ber den Aufrufparameter wird auch festgelegt, ob der Parametersatz berschrieben werden soll.
 * Falls ja, wird der DE-Abruf-Parameter neu aus der csv-Datei erstellt.
 * Falls nein, wird der DE-Abruf-Parameter mit Daten aus der csv-Datei ergnzt.
 * <p/>
 * Die Parameter-Eintrge werden nach Doppel-Eintrge berprft (s.Methode {@link Parameter#equals(Object)}).
 * Falls die Eintrge gleich sind, wird nur letzter hinzugefgt (der letzte gewinnt).
 * Beim gleichen Eintrag aus DaV und aus der Datei gewinnt der Eintrag aus der Datei.
 * <p/>
 * Jeder Eintrag in der Datei besteht aus folgenden Informationen:
 * <li/> SO_PID: PID des SystemObjekts von Typ typ.de;
 * <li/> ID: ID des Eintrags;
 * <li/> TYP: Typ des Eintrags;
 * <li/> INTERVALL_SEKUNDEN Intervall des DE-Abrufs als Sekundenangabe.
 * <li/> AKTIVIERT: ja oder nein.
 *
 * @author inovat, innovative systeme - verkehr - tunnel - technik
 * @author Liliya Givorgizova (LG)
 * @version $Revision: 653 $ / $Date: 2012-11-02 11:59:36 +0100 (Fr, 02 Nov 2012) $ / ($Author: LG $)
 */
public class VewDeAbrufParameter implements StandardApplication, ClientSenderInterface {
    private static final String ASPEKT_SOLL    = "asp.parameterSoll";
    private static final String ASPEKT_VORGABE = "asp.parameterVorgabe";
    private static final String ATG            = "atg.paramTlsDeAbruf";
    private static final String TRENN_LINIE    = "=======================================================================================\n";

    /** DebugLogger fr Debug-Ausgaben. */
    private static Debug debug = Debug.getLogger();

    //~ FELDER ================================================================

    /** Legt fest, ob das Sendeerlaubnis fr den Parameter vorhanden ist. */
    private boolean         _sendeErlaubnis = false;
    private DataDescription _datenBeschreibung;

    /** Datenverteilerverbindung. */
    private ClientDavInterface _dav;

    /** Die Konfiguration des DaV. */
    private DataModel _davKonf;

    /** Parameter: csv-datei mit Daten fr den Parameter DE-Abruf. */
    private String _parDatei;

    /** Parameter: SystemObjekt-PID fr den Parameter DE-Abruf. */
    private String _parSoPid;

    /**
     * Parameter: Legt fest ob Parametersatz berschrieben werden soll
     * (ja - Parameter wird berschrieben, nein - Parameter wird ergnzt).
     */
    private boolean _parUeberschreiben;

    /** SystemObjekt fr den Parameter DE-Abruf. */
    private SystemObject _soDeAbruf;

    //~ METHODEN ==============================================================

    /**
     * Debug-Info-Ausgabe der ParameterListe.
     *
     * @param titel Titel.
     * @param listePar ParameterListe.
     */
    private void ausgabe(String titel, List<Parameter> listePar) {
        StringBuilder txt = new StringBuilder(TRENN_LINIE);

        txt.append(titel).append(":\n");

        for (Parameter p : listePar) {
            txt.append(p).append("\n");
        }

        txt.append(TRENN_LINIE);
        debug.info(txt.toString());
    }

    @Override
    public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
        _sendeErlaubnis = (state == START_SENDING);

        String status = "unbekannter Status";

        if (state == START_SENDING) {
            status = "START_SENDING";
        } else if (state == STOP_SENDING) {
            status = "STOP_SENDING";
        } else if (state == STOP_SENDING_NO_RIGHTS) {
            status = "STOP_SENDING_NO_RIGHTS";
        } else if (state == STOP_SENDING_NOT_A_VALID_SUBSCRIPTION) {
            status = "STOP_SENDING_NOT_A_VALID_SUBSCRIPTION";
        }

        debug.fine(String.format("SO: %s%nDatenbeschreibung: %s%nStatus: %s%n", object, dataDescription, status));
    }

    @Override
    public void initialize(ClientDavInterface dav) throws Exception {

        // --------------------------------------------------------------
        // DaV-Zugriff:
        _dav     = dav;
        _davKonf = _dav.getDataModel();

        // --------------------------------------------------------------
        // Initialisiere DE-Abruf-Objekt:
        _soDeAbruf = _davKonf.getObject(_parSoPid);

        if (_soDeAbruf == null) {
            debug.error(String.format("Das Objekte mit dem PID <%s> kann nicht ermittelt werden!. Bitte prfen Sie den Aufrufparameter <-soDeAbrufPid>. Programm wird beendet.", _parSoPid));

            throw new RuntimeException("Null-Objekt");
        }
    }

    //~ GET METHODEN ==========================================================

    @Override
    public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
        return true;
    }

    //~ METHODEN ==============================================================

    /**
     * Lade den Parameter DE-Abruf aus Dav.
     *
     * @return Liste der aus DaV geladenen Parameter.
     */
    private List<Parameter> ladeParameterAusDav() {

        //
        debug.info(String.format("%s Lade  den Parametersatz aus DaV ...\n%s", TRENN_LINIE, TRENN_LINIE));

        //
        List<Parameter> listePar = new ArrayList<Parameter>();

        // --------------------------------------------------------------
        // Initialisiere DaV-Daten:
        AttributeGroup  atg               = _davKonf.getAttributeGroup(ATG);
        Aspect          asp               = _davKonf.getAspect(ASPEKT_SOLL);
        DataDescription datenBeschreibung = new DataDescription(atg, asp);

        // --------------------------------------------------------------
        ResultData daten = _dav.getData(_soDeAbruf, datenBeschreibung, 0);

        // Lese den DE-Abruf-Parameter aus DaV:
        if (daten.hasData()) {
            String  dePid;
            int     id;
            int     typ;
            long    zeitIntervallInMillis;
            boolean aktiviert;

            //
            Data.Array   ar = daten.getData().getArray("DeAbrufParameter");
            Data         data;
            SystemObject de;

            for (int i = 0; i < ar.getLength(); i++) {

                // Ermittle Daten:
                data                  = ar.getItem(i);
                dePid                 = data.getReferenceValue("DeReferenz").getSystemObjectPid();
                id                    = data.getUnscaledValue("ID").intValue();
                typ                   = data.getUnscaledValue("Typ").intValue();
                zeitIntervallInMillis = data.getTimeValue("Intervall").getMillis();
                aktiviert             = (data.getTextValue("aktiviert").getText().equals("Ja"));

                // Ermittle SystemObject:
                de = _davKonf.getObject(dePid);

                // Fge den Parameter in die Liste ein:
                if (de != null) {
                    listePar.add(new Parameter(de, id, typ, zeitIntervallInMillis, aktiviert));
                }

                // Warnung:
                else {
                    debug.warning(String.format("DE mit der PID <%s> ist nicht ermittelbar", dePid));
                }
            }
        }

        // Warnung:
        else {
            debug.warning(String.format("Parametersatz DE-Abruf ist leer."));
        }

        // --------------------------------------------------------------
        // Ausgabe:
        ausgabe(String.format("Es wurden folgende <%d> Parameter-Eintrge aus DaV geladen", listePar.size()), listePar);

        //
        return listePar;
    }

    /**
     * Lese die Parameter-Eintrge aus der csv-Datei.
     *
     * @return Liste der aus der csv-Datei ausgelesenen Parameter.
     */
    private List<Parameter> leseParameterAusDerCsvDatei() {
        debug.info(String.format("%s Leser  den Parametersatz aus der Datei \n<%s>...\n%s", TRENN_LINIE, _parDatei, TRENN_LINIE));

        //
        List<Parameter> listePar = new ArrayList<Parameter>();
        boolean         fehler   = false;

        // --------------------------------------------------------------
        // Lese die csv-Datei:
        File datei = new File(_parDatei);

        //
        String spSoPid            = "SO_PID";
        String spId               = "ID";
        String spTyp              = "TYP";
        String spIntevallSekunden = "INTERVALL_SEKUNDEN";
        String spAktiviert        = "AKTIVIERT";

        //
        String       soPid;
        SystemObject so;
        int          id;
        int          typ;
        long         intervall;
        boolean      aktiviert;
        Parameter    par;

        //
        CsvLeser csvDatei;
        int      zeilenNr = 1;

        try {

            // Datei einlesen...
            csvDatei = new CsvLeser(new FileReader(datei), ';');

            Map<String, String> zeile;

            // --------------------------------------------------------------
            // fr jede Zeile:
            while (csvDatei.hasNochMehrZeilen()) {
                zeile = csvDatei.getNaechsteZeile();

                // --------------------------------------------------------------
                // lese und prfe Daten der Zeile:
                soPid     = zeile.get(spSoPid).trim();
                id        = Integer.valueOf(zeile.get(spId).trim());
                typ       = Integer.valueOf(zeile.get(spTyp).trim());
                intervall = Integer.valueOf(zeile.get(spIntevallSekunden).trim()) * 1000;
                aktiviert = (zeile.get(spAktiviert).trim().equals("ja"));
                so        = _davKonf.getObject(soPid);

                if (so == null) {
                    debug.error(String.format("Unbekanntes SystemObject <%s> in der Zeile <%d>", soPid, zeilenNr));
                    fehler = true;

                    break;
                }

                // --------------------------------------------------------------
                // Einfge  Parameter in die Liste:
                par = new Parameter(so, id, typ, intervall, aktiviert);
                listePar.add(par);

                //
                zeilenNr++;
            }

            csvDatei.close();
        }
        catch (FileNotFoundException e) {
            debug.error(String.format("Datei <%s> kann nicht gefunden werden!", datei), e);
            fehler = true;
        }
        catch (NumberFormatException e) {
            debug.error(String.format("Fehler beim Einlesen der Datei <%s> in der Zeile <%d>.", datei, zeilenNr), e);
            fehler = true;
        }
        catch (Exception e) {
            debug.error(String.format("Fehler beim Einlesen der Datei <%s> in der Zeile <%d>.", datei, zeilenNr), e);
            fehler = true;
        }

        // --------------------------------------------------------------
        // Falls _Fehler aufgetreten sind:
        if (fehler) {
            return new ArrayList<Parameter>();
        }

        // --------------------------------------------------------------
        // Ausgabe:
        ausgabe(String.format("Es wurden folgende <%d> Parameter-Eintrge aus der Datei gelesen", listePar.size()), listePar);

        // --------------------------------------------------------------
        return listePar;
    }

    /**
     * Main-Methode.
     *
     * @param args Parameter.
     */
    public static void main(String[] args) {

        //
        VewDeAbrufParameter vewPar = new VewDeAbrufParameter();

        StandardApplicationRunner.run(vewPar, args);

        // Debug erst hier initialisieren, da erst ab hier verfgbar !!
        debug = Debug.getLogger();

        // Sende den Parametersatz:
        vewPar.sendeDeAbrufParameter();

        // Ende:
        System.exit(0);
    }

    @Override
    public void parseArguments(ArgumentList argumentList) throws Exception {
        String par;

        // --------------------------------------------------------------
        // Hole den Parameter 'SystemObjekt PID fr den DE-Abruf':
        par = "-soDeAbrufPid";

        try {
            _parSoPid = argumentList.fetchArgument(par).asNonEmptyString();
        }
        catch (Exception e) {
            debug.error(String.format("Bitte die SystemObjekt-PID fr den DE-Abruf als Parameter <%s> bergeben!", par), e);

            throw new RuntimeException(e);
        }

        // --------------------------------------------------------------
        // Hole den Parameter 'csv-Datei mit Parameter-Daten':
        par = "-csvDatei";

        try {
            _parDatei = argumentList.fetchArgument(par).asNonEmptyString();
        }
        catch (Exception e) {
            debug.error(String.format("Bitte die csv-Datei mit Parameter-Daten als Parameter <%s> bergeben!", par), e);

            throw new RuntimeException(e);
        }

        // --------------------------------------------------------------
        // Hole den Parameter 'Parameter berschreiben'
        // (ja - Parameter wird berschrieben, nein - Parameter wird ergnzt):
        par = "-parameterUeberschreiben";

        try {
            String parUeberschreiben = argumentList.fetchArgument(par).asNonEmptyString().toLowerCase();

            if (parUeberschreiben.equals("ja")) {
                _parUeberschreiben = true;
            } else if (parUeberschreiben.equals("nein")) {
                _parUeberschreiben = false;
            } else {
                debug.error(String.format("Unzulssiger Wert fr den Parameter <%s=%s> (darf nur <ja> oder <nein> sein)!", par, parUeberschreiben));

                throw new RuntimeException();
            }
        }
        catch (Exception e) {
            debug.error(String.format("Bitte den Parameter <%s> mit dem Wert <ja> oder <nein> bergeben!", par), e);

            throw new RuntimeException(e);
        }
    }

    /**
     * Erstelle den DE-Abruf-Parameter aus den DaV-Daten und der csv-Datei
     * und sende den an DaV.
     */
    private void sendeDeAbrufParameter() {

        // --------------------------------------------------------------
        // Lade Parameter aus DaV, falls den Parameter ergnszt werden soll:
        // --------------------------------------------------------------
        List<Parameter> listeParGelesen;

        if (!_parUeberschreiben) {
            listeParGelesen = ladeParameterAusDav();
        } else {
            listeParGelesen = new ArrayList<Parameter>();
        }

        // --------------------------------------------------------------
        // Erstelle DaV-Parameter aus csv-Datei und fge die in der Liste:
        // --------------------------------------------------------------
        {
            List<Parameter> listeParAusDatei = leseParameterAusDerCsvDatei();

            if (listeParAusDatei.isEmpty()) {
                debug.error("Die Liste der zu erstellten Parameter ist leer!");

                throw new RuntimeException();
            }

            // Fge die erstellten Parameter in die Liste:
            listeParGelesen.addAll(listeParAusDatei);
        }

        // --------------------------------------------------------------
        // Erstelle die neue Liste der Parameter aus der Eintrgen aus DaV und aus der Datei.
        //
        // Falls die Eintrge gleich sind (s. Methode Parameter.equal()),
        // wird nur letzter hinzugefgt (der letzte gewinnt).
        //
        // Beim gleichen Eintrag aus DaV und aus der Datei
        // gewinnt der Eintrag aus der Datei.
        // --------------------------------------------------------------
        debug.info(String.format("%s Erstelle neuen Parametersatz ...\n%s", TRENN_LINIE, TRENN_LINIE));

        //
        List<Parameter> listeParNeu                 = new ArrayList<Parameter>();
        List<Parameter> listeParNichtBercksichtigt = new ArrayList<Parameter>();

        // Betrachte in der umgekehrten Reihenfolge um die Prioritt
        // der Eintrge aus der Datei vor den Eintrge aus DaV und
        // die Prioritt des letzten Eintrags zu schafen:
        for (int i = listeParGelesen.size() - 1; i >= 0; i--) {
            Parameter p = listeParGelesen.get(i);

            if (listeParNeu.contains(p)) {
                listeParNichtBercksichtigt.add(0, p);
            } else {
                listeParNeu.add(p);
            }
        }

        // Sortiere die Liste:
        Collections.sort(listeParNeu, new Comparator<Parameter>() {
            @Override
            public int compare(Parameter p1, Parameter p2) {
                return p1._de.getPidOrNameOrId().compareTo(p2._de.getPidOrNameOrId());
            }
        });

        // Ausgabe:
        if (!listeParNichtBercksichtigt.isEmpty()) {
            ausgabe(String.format("Folgende <%d> Parameter-Eintrge wurden nicht bercksichtigt (da sie mehrmals vorhanden sind)", listeParNichtBercksichtigt.size()), listeParNichtBercksichtigt);
        }

        ausgabe(String.format("Es sind folgende neue <%d> Parameter-Eintrge erstellt", listeParNeu.size()), listeParNeu);

        // --------------------------------------------------------------
        // Initialisiere DaV-Daten:
        // --------------------------------------------------------------
        AttributeGroup atg = _davKonf.getAttributeGroup(ATG);
        Aspect         asp = _davKonf.getAspect(ASPEKT_VORGABE);

        _datenBeschreibung = new DataDescription(atg, asp);

        // --------------------------------------------------------------
        // Anmeldung als Sender:
        // --------------------------------------------------------------
        try {
            _dav.subscribeSender(this, _soDeAbruf, _datenBeschreibung, SenderRole.sender());
        }
        catch (OneSubscriptionPerSendData e) {
            debug.error(String.format("Datenidentifikation ist bereits angemeldet. Eventuell Empfnger nicht vorhanden: <%s>, <%s>. Programm wird beendet.", _soDeAbruf, _datenBeschreibung), e);

            throw new RuntimeException(e);
        }
        catch (Exception e) {
            debug.error(String.format("Fehler bei der Anmeldung der Daten als Sender: <%s>, <%s>. Programm wird beendet.", _soDeAbruf, _datenBeschreibung), e);

            throw new RuntimeException(e);
        }

        // -------------------------------------------------
        // Warte auf SendeErlaubnis:
        // Ein paar mal versuchen, ob Sendeerlaubnis bereits vorliegt...
        // -------------------------------------------------
        int i = 0;

        while (i < 10) {
            if (_sendeErlaubnis) {
                break;
            } else {

                // Pause:
                try {
                    Thread.sleep(500);
                }
                catch (Exception ignored) {}
            }

            i++;
        }

        if (!_sendeErlaubnis) {
            debug.error(String.format("Sendeerlaubnis ist nicht vorhanden: <%s>, <%s>", _soDeAbruf, _datenBeschreibung));

            throw new RuntimeException("Sendeerlaubnis ist nicht vorhanden");
        }

        // -------------------------------------------------
        // Erstelle DaV-Daten aus der Liste der Parameter:
        // -------------------------------------------------
        Data daten = _dav.createData(atg);

        // Urlasser-Daten:
        String ursache;

        if (_parUeberschreiben) {
            ursache = "Neue Anlage des Parametersatzes";
        } else {
            ursache = "Ergnzung des Parametersatzes";
        }

        daten.getItem("Urlasser").getTextValue("Ursache").setText(ursache);
        daten.getItem("Urlasser").getTextValue("Veranlasser").setText("Tool zur Verwaltung des DE-Abruf-Parameters <de.bsvrz.kex.tls.deabruf.par>");

        // DE-Abruf-Parameter
        Data.Array ar = daten.getArray("DeAbrufParameter");

        ar.setLength(listeParNeu.size());

        Parameter par;
        Data      data;

        for (i = 0; i < listeParNeu.size(); i++) {
            par  = listeParNeu.get(i);
            data = ar.getItem(i);
            data.getReferenceValue("DeReferenz").setSystemObject(par._de);
            data.getUnscaledValue("ID").set(par._id);
            data.getUnscaledValue("Typ").set(par._typ);
            data.getTimeValue("Intervall").setMillis(par._intervall);
            data.getTextValue("aktiviert").setText(par._aktiviert ? "Ja" : "Nein");
        }

        ResultData ergebnis = new ResultData(_soDeAbruf, _datenBeschreibung, System.currentTimeMillis(), daten, false);

        // -------------------------------------------------
        // Sende DaV-Daten:
        // -------------------------------------------------
        try {
            _dav.sendData(ergebnis);

            // Ausgabe:
            debug.info(String.format("%s Es sind <%d> Eintrge fr den DE-Abruf-Parameter erfolgreich gesendet.\n%s", TRENN_LINIE, listeParNeu.size(), TRENN_LINIE));
        }
        catch (Exception e) {
            debug.error(String.format("Fehler beim Senden der Daten an DaV: <%s>, <%s>", _soDeAbruf, _datenBeschreibung), e);

            throw new RuntimeException(e);
        }
    }

    //~ INNERE KLASSEN ========================================================

    /**
     * Daten fr den DE-Abruf-Parameter.
     */
    private class Parameter {
        private boolean      _aktiviert;
        private SystemObject _de;
        private int          _id;
        private long         _intervall;
        private int          _typ;

        //~ KONSTRUKTOREN  (und vom Konstruktor verwendete Methoden) ==========

        /**
         * Erstellt eine Instanz der Klasse Parameter.
         *
         * @param de DE.
         * @param id ID des Parameters.
         * @param typ Typ des Parameters.
         * @param intervall ZeitIntervall in Millis zwischen den Ausfhren der Auftrge.
         * @param aktiviert Legt fest, ob der Auftragt aktiviert ist (Wert == true).
         */
        private Parameter(SystemObject de, int id, int typ, long intervall, boolean aktiviert) {
            _de        = de;
            _id        = id;
            _typ       = typ;
            _intervall = intervall;
            _aktiviert = aktiviert;
        }

        //~ METHODEN ==========================================================

        /**
         * Stellt fest, ob die Parameter gleich sind.
         *
         * @param obj 2.Parameter.
         * @return true, falls DE,ID und Typ gleich sind,
         * false sonst.
         */
        @Override
        public boolean equals(Object obj) {

            // Parameter sind gleich, falls DE, ID und Typ gleich sind:
            if (obj instanceof Parameter) {
                Parameter par = (Parameter) obj;

                return ((_de.equals(par._de)) && (_id == par._id) && (_typ == par._typ));
            }

            return false;
        }

        //~ GET METHODEN ======================================================

        /**
         * Stellt die Zeitdauer als String dar.
         *
         * @param zeitDauer Zeitdauer, die als String dargestellt werden soll.
         *
         * @return Zeitdauer als String "hh:mm:ss".
         */
        private String getZeitdauerAlsString(long zeitDauer) {
            long  zeitHr  = zeitDauer / (1000 * 60 * 60);
            long  zeitMin = (zeitDauer - zeitHr * 60 * 60 * 1000) / (1000 * 60);
            float zeitSek = (zeitDauer - zeitHr * 60 * 60 * 1000 - zeitMin * 60 * 1000) / 1000f;

            return String.format("%02d:%02d:%02d", zeitHr, zeitMin, Math.round(zeitSek));
        }

        //~ METHODEN ==========================================================

        @Override
        public String toString() {
            return String.format("%-65s  %3d  %3d  %s  %s", _de.getPidOrNameOrId(), _id, _typ, getZeitdauerAlsString(_intervall), _aktiviert);
        }
    }
}


//~Formatiert mit 'inovat Kodierkonvention' am 17.07.12
