/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import java.util.HashMap;
import java.util.Map;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data.NumberValue;
import de.bsvrz.dav.daf.main.Data.TextValue;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.MutableSetChangeListener;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse zur Verwaltung von TMC-Meldungen einer LMS.<br>
 * Beim Instanziieren eines Klassenobjekts werden aus einem Objekt vom Typ
 * typ.tmcLandesMeldeStelle die TMC-Meldungen der zugeordneten Menge "
 * TMCMeldungen" zur Verwaltung eingelesen. Die Menge der aktuellen
 * TMC-Meldungen wird bezüglich neuen Elementen und Löschung vorhandener
 * Elemente überwacht.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class TMCVerwaltung implements MutableSetChangeListener, InvalidationListener, ClientReceiverInterface {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Konfigurationsobjekt der LMS.
	 */
	private ConfigurationObject lmsObjekt;

	/**
	 * Name der dynamische Menge, die für das Konfigurationsobjekt verwendet
	 * wird.
	 */
	public static final String TMCMELDUNGEN = "TMCMeldungen";

	/**
	 * Datenverteiler-Verbindung.
	 */
	private ClientDavInterface dav;

	/**
	 * Löschverfahren für die dynamischen TMC-Meldungsobjekte.
	 */
	// private int loeschVerfahren;

	/**
	 * Flag zur Unterdrückung der Korrektur, wenn ein Zeitstempel einer Meldung
	 * in der Vergangenheit liegt.
	 */
	// private boolean ohneZeitstempelKorrektur;

	/**
	 * Datenbeschreibung der ATG, die die verwendeten Parameter enthält.
	 */
	private DataDescription datenBeschreibung;

	/**
	 * Alle aktuell verwalteten TMC-Meldungen.
	 */
	private Map<SystemObject, TMCVewMeldung> aktuelleMeldungen = new HashMap<SystemObject, TMCVewMeldung>();

	/**
	 * Instanziiert die Verwaltung für TMC-Meldungen.
	 * 
	 * @param connection
	 *            die Datenverteiler-Verbindung
	 * @param konfigObjekt
	 *            das Konfigurationsobjekt, das die LMS darstellt
	 * @param optionen
	 *            Optionen als Bitmaske
	 */
	public TMCVerwaltung(final ClientDavInterface connection, final ConfigurationObject konfigObjekt, final int optionen) {
		dav = connection;
		lmsObjekt = konfigObjekt;
		// ohneZeitstempelKorrektur = (optionen & TMCVerwaltungApp.MASKE_OHNE_ZEITSTEMPEL_KORREKTUR) != 0;

		final AttributeGroup atg = dav.getDataModel().getAttributeGroup("atg.tmcMeldungsVerwaltung");
		final Aspect asp = dav.getDataModel().getAspect("asp.parameterSoll");
		datenBeschreibung = new DataDescription(atg, asp);
		dav.subscribeReceiver(this, lmsObjekt, datenBeschreibung, ReceiveOptions.normal(),
				ReceiverRole.receiver());
		DEBUG.config(datenBeschreibung + " zum Empfang angemeldet für " + lmsObjekt);

		final MutableSet menge = lmsObjekt.getMutableSet(TMCMELDUNGEN);
		for (SystemObject obj : menge.getElements()) {
			if (obj.isValid()) {
				initialisiereMeldung(obj);
			}
		}
		menge.addChangeListener(this);
		DEBUG.config("An Menge \"" + TMCMELDUNGEN + "\" angemeldet");

		DEBUG.config(aktuelleMeldungen.size() + " TMC-Meldungsobjekt(e) initial verwaltet");
	}

	/**
	 * Instanziiert die Verwaltung für TMC-Meldungen.
	 * 
	 * @param connection
	 *            die Datenverteiler-Verbindung
	 * @param konfigObjekt
	 *            das Konfigurationsobjekt, das die LMS darstellt
	 */
	public TMCVerwaltung(final ClientDavInterface connection, final ConfigurationObject konfigObjekt) {
		this(connection, konfigObjekt, 0);
	}

	/**
	 * Gibt belegte Resourcen frei.
	 */
	public void dispose() {
		lmsObjekt.getMutableSet(TMCMELDUNGEN);
		DEBUG.config("Von Menge \"" + TMCMELDUNGEN + "\" abgemeldet");

		for (TMCVewMeldung meldung : aktuelleMeldungen.values()) {
			if (meldung.getDavObjekt() instanceof DynamicObject) {
				((DynamicObject) meldung.getDavObjekt()).removeListenerForInvalidation(this);
				DEBUG.fine("Invalidation-Listener abgemeldet für " + meldung.getDavObjekt().getPid());
			}
			meldung.dispose();

		}
		aktuelleMeldungen.clear();

		dav.unsubscribeReceiver(this, lmsObjekt, datenBeschreibung);
		DEBUG.config("Empfang abgemeldet von " + datenBeschreibung + " für " + lmsObjekt);
	}

	/**
	 * Gibt die die aktuelle DAV-Verbindung zur&uuml;ck.
	 * 
	 * @return {@link ClientDavInterface}
	 */
	public ClientDavInterface getConnection() {
		return dav;
	}

	/**
	 * Gibt das aktuelle Konfigurationsobjekt vom Typ typ.landesMeldeStelle zur&uuml;ck.
	 * 
	 * @return {@link ConfigurationObject}
	 */
	public ConfigurationObject getKonfigObjekt() {
		return lmsObjekt;
	}


	@Override
	public void update(final MutableSet set, final SystemObject[] addedObjects, final SystemObject[] removedObjects) {
		DEBUG.fine("MutableSetChangeListener.update() von \"" + set.getName() + "\"");

		if (addedObjects.length > 0) {
			DEBUG.config(
					"Dynamische Menge \"" + set.getName() + "\" um " + addedObjects.length + " Objekte vergrössert");

			for (SystemObject obj : addedObjects) {
				if (obj.isValid()) {
					initialisiereMeldung(obj);
				}
			}
		}

		if (removedObjects.length > 0) {
			DEBUG.config(
					"Dynamische Menge \"" + set.getName() + "\" um " + removedObjects.length + " Objekte verkleinert");

			for (SystemObject obj : removedObjects) {
				bereinigeMeldung(obj);
			}
		}
	}

	/**
	 * Bereinigt die Verwaltung für eine neu entfernte (gelöschte, invalidierte)
	 * TMC-Meldung.
	 * 
	 * @param obj
	 *            Das Stellvertreter-Objekt der TMC-Meldung
	 */
	private void bereinigeMeldung(final SystemObject obj) {
		final TMCVewMeldung meldung = aktuelleMeldungen.remove(obj);
		if (meldung != null) {
			meldung.dispose();
		}
	}

	/**
	 * Bereinigt die Verwaltung für eine neu entfernte (gelöschte, invalidierte)
	 * TMC-Meldung. Diese Methode nimmt die Bereinigung für die Meldung in
	 * ihrer Eigenschaft als dynmaisches Objekt vor und ruft dann
	 * {@link de.bsvrz.ste.tmcvew.TMCVerwaltung#bereinigeMeldung(SystemObject)}
	 * auf.
	 * 
	 * @param obj
	 *            Das dynamische Stellvertreter-Objekt der TMC-Meldung
	 */
	private void bereinigeMeldung(final DynamicObject obj) {
		obj.removeListenerForInvalidation(this);
		DEBUG.fine("Invalidation-Listener abgemeldet für " + obj.getPid());

		/*
		 * Falls obj vor dem löschen nicht aus der dyn. Menge entfernt wurde:
		 */
		bereinigeMeldung((SystemObject) obj);
	}

	/**
	 * Initialisiert die Verwaltung für eine neu hinzugekommene TMC-Meldung.
	 * 
	 * @param obj
	 *            Das Stellvertreter-Objekt der TMC-Meldung
	 */
	private void initialisiereMeldung(final SystemObject obj) {
		final TMCVewMeldung meldung = new TMCVewMeldung(obj, this);
		aktuelleMeldungen.put(obj, meldung);
		if (obj instanceof DynamicObject) {
			((DynamicObject) obj).addListenerForInvalidation(this);
			DEBUG.fine("Invalidation-Listener angemeldet für " + obj.getPid());
		}
	}

	@Override
	public void invalidObject(final DynamicObject dynamicObject) {
		DEBUG.config("Objekt " + dynamicObject + " gelöscht/invalidiert => wird nicht weiter verwaltet");

		bereinigeMeldung(dynamicObject);
	}

	@Override
	public void update(final ResultData[] resultData) {
		for (ResultData data : resultData) {
			final Aspect asp = data.getDataDescription().getAspect();
			final AttributeGroup atg = data.getDataDescription().getAttributeGroup();

			if (atg.equals(datenBeschreibung.getAttributeGroup()) && asp.equals(datenBeschreibung.getAspect())) {
				if (data.hasData()) {
					final NumberValue nv = data.getData().getUnscaledValue("RDSMeldungsRückhaltung");
					rdsMeldungsRueckhaltung = nv == null ? -1 : nv.longValue();
					final TextValue tv = data.getData().getTextValue("RDSMeldungsAktion");
					rdsMeldungsAktion = tv == null ? null : tv.getText();
				} else {
					rdsMeldungsRueckhaltung = -1;
					rdsMeldungsAktion = null;
				}
				DEBUG.config("Update Daten " + atg.getPid() + " für " + lmsObjekt.getName()
						+ "\nRDSMeldungsRückhaltung auf " + rdsMeldungsRueckhaltung + "\nRDSMeldungsAktion auf "
						+ rdsMeldungsAktion);
			}
		}
	}

	/**
	 * Parameter SendeVerzoegerungAufhebung.
	 */
	private long rdsMeldungsRueckhaltung = -1;

	/**
	 * Gibt den Wert des Parameters RDSMeldungsRückhaltung zur&uuml;ck.
	 * 
	 * @return Wert des Parameters RDSMeldungsRückhaltung
	 */
	public long getRDSMeldungsRueckhaltung() {
		return rdsMeldungsRueckhaltung;
	}

	/**
	 * Parameter MeldungsErhaltung.
	 */
	private String rdsMeldungsAktion;

	/**
	 * Gibt den  Wert des Parameters RDSMeldungsAktion zur&uuml;ck.
	 * 
	 * @return Wert des Parameters RDSMeldungsAktion
	 */
	public String getRDSMeldungsAktion() {
		return rdsMeldungsAktion;
	}

}
