/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;

/**
 * Applikationsklasse der SWE TMC-Meldungsverwaltung.<br>
 * Für die übergebene LMS wird eine Applikations-Instanz der SWE erzeugt
 * und initialisiert.<br>
 * SWE-spezifische Parameter:<br>
 * -lms: Die Landesmeldestelle, auf welche diese Instanz der SWE sich bezieht.
 * Das Konfigurationsobjekt der LMS enthält insbesondere die zu verwendeten
 * Parameter.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class TMCVerwaltungApp implements StandardApplication {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private Debug debug;

	/**
	 * -lms von Kommandozeile.
	 */
	private String lmsPid;

	/**
	 * -loeschen von Kommandozeile.
	 */
	private int loeschVerfahren;

	/**
	 * -ohneZeitstempelKorrektur von Kommandozeile.
	 */
	private int ohneZeitstempelKorrektur;

	
	private static final String LMS_TYP = "typ.tmcLandesMeldeStelle";

	/**
	 * Datenverteiler-Verbindung.
	 */
	private ClientDavInterface connection;

	/**
	 * Default- Konstruktor.<br>
	 * Wird typischerweise von <code>main</code> aufgerufen
	 */
	public TMCVerwaltungApp() {
		try {
			Thread.sleep(100);
		} catch (final InterruptedException e) {
		}
	}

	/**
	 * Extrahiert das Konfigurationsobjekt zur übergebenenen PID und führt ggf.
	 * eine Typprüfung durch.
	 * 
	 * @param pid
	 *            die PID des Konfigurationsobjektes, das extrahiert werden soll
	 * @param typ
	 *            der Typ, den das Konfigutationsobjekt haben muss - wenn hier
	 *            <code>null</code> übergeben wird, unterbleibt eine Typprüfung.
	 * @return das gefundene Konfigurationsobjekt
	 * 
	 * @throws IllegalArgumentException 
	 *             falls Konfigurationsobjekt nicht gefunden oder vom falschen
	 *             Typ
	 */
	private ConfigurationObject getKonfigObjekt(final String pid, final String typ) throws IllegalArgumentException {
		ConfigurationObject obj = null;

		try {
			obj = (ConfigurationObject) connection.getDataModel().getObject(pid);
		} catch (final ClassCastException e) {
			throw new IllegalArgumentException("Die PID \"" + pid + "\" bezeichnet kein KonfigurationsObjekt");
		}

		if (obj == null) {
			throw new IllegalArgumentException("Zur PID \"" + pid + "\" wurde kein Konfigurationsobjekt gefunden ");
		}

		if (typ != null && !obj.isOfType(typ)) {
			throw new IllegalArgumentException(
					"Das Konfigurationsobjekt \"" + obj.getPid() + "\" ist nicht vom Typ " + typ);
		}

		return obj;
	}

	/**
	 * überschriebene Methode von <code>StandardApplication</code>, die die
	 * speziellen Startparameter auswertet.<br>
	 * Die Argumtwerte, die ausgewertet werden, sind durch das Attribut
	 * {@link #lmsPid} spezifiziert
	 * 
	 * @param argumentList
	 *            siehe
	 *            sys.funclib.application.StandardApplication#parseArguments(sys.funclib.ArgumentList)
	 */
	public void parseArguments(final ArgumentList argumentList) {
		debug = Debug.getLogger();

		lmsPid = argumentList.fetchArgument("-lms=").asNonEmptyString();

		argumentList.fetchUnusedArguments();
	}

	/**
	 * Überschriebene Methode von StandardApplication, die die Initialisierung
	 * durchführt.<br>
	 * Es wird eine Instanz von <code>VerwaltungEngstellen</code> erzeugt und
	 * initialisiert
	 * 
	 * @param con
	 *            siehe
	 *            sys.funclib.application.StandardApplication#initialize(de.bsvrz.dav.daf.main.ClientDavInterface)
	 */
	public void initialize(final ClientDavInterface con) {
		this.connection = con;
		try {
			MessageSender.getInstance().setApplicationLabel(connection.getLocalApplicationObject().getName());

			final ConfigurationObject lmsobjekt = getKonfigObjekt(lmsPid, LMS_TYP);
			
			if (lmsobjekt == null) {
				throw new RuntimeException("Das angegebene Objekt '" + lmsPid + "' kann nicht vom Datenverteiler gelesen werden!");
			}
			
			if (!lmsobjekt.isOfType(LMS_TYP)) {
				throw new RuntimeException("Das angegebene Objekt '" + lmsPid + "' ist nicht vom Typ '" + LMS_TYP + "'!");
			}

			final TMCVerwaltung verwaltung = new TMCVerwaltung(connection, lmsobjekt, // Hauptapplikation
					loeschVerfahren | ohneZeitstempelKorrektur);

			// Raeumt System auf
			Runtime.getRuntime().addShutdownHook(new Thread() {
				public void run() {
					debug.info("Freigabe Ressourcen ...");
					verwaltung.dispose();
				}
			});

			debug.info("SWE TMC-Meldungsverwaltung initialisiert");
		} catch (final Exception e) {
			debug.error("Abbruch der SWE TMC-Meldungsverwaltung während der Initialisierung", e);
			System.exit(1);
		}
	}

	/**
	 * Programmeinstieg TMC-Meldungsverwaltung.
	 * 
	 * @param arguments
	 *            Kommandozeilenargumente, folgende speziellen Argumente werden
	 *            ausgewertet:
	 *            <p>
	 *            -netz: Netz, dessen Infrastrukturobjekte verwendet werden.
	 *            -lms: Zuständige Landesmeldestelle (LMS).
	 */
	public static void main(final String[] arguments) {
		StandardApplicationRunner.run(new TMCVerwaltungApp(), arguments);
	}
}
