/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.TextValue;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.ste.tmcvew.parameter.ParameterEmpfaenger;
import de.bsvrz.ste.tmcvew.parameter.ParameterSender;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Modelliert eine TMC-Meldungen so wie sie von der SWE TMC-Meldungsverwaltung
 * benötigt wird.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan
 * Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class TMCVewMeldung implements ClientReceiverInterface {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * DAV-Stellvertreter- Objekt vom Typ TMCVerkehrsmeldung.
	 */
	private SystemObject davObjekt;

	/**
	 * Gibt das DAV-Stellvertreter-Objekt zurück.
	 * 
	 * @return das DAV-Stellvertreter-Objekt vom Typ TMCVerkehrsmeldung
	 */
	public SystemObject getDavObjekt() {
		return davObjekt;
	}

	/**
	 * Übergeordnete Verwaltungs-Instanz.
	 */
	private TMCVerwaltung verwaltung;

	/**
	 * Datenbeschreibung "versendet".
	 */
	private final DataDescription datenBeschreibungVersendet;

	/**
	 * Datenbeschreibung "generiert".
	 */
	private final DataDescription datenBeschreibungGeneriert;

	public DataDescription getDatenBeschreibungGeneriert() {
		return datenBeschreibungGeneriert;
	}

	/**
	 * Datenbeschreibung "bearbeitet".
	 */
	private final DataDescription datenBeschreibungBearbeitet;

	/**
	 * Ist diese Instanz zum DAV-Empfang angemeldet?
	 */
	private boolean angemeldet;

	/**
	 * Gibt das Flag zurück, ob die Meldung bereits beim DAV angemeldet wurde.
	 * 
	 * @return ist diese Instanz zum DAV-Empfang angemeldet?
	 */
	public boolean isAngemeldet() {
		return angemeldet;
	}

	/**
	 * Aktuell empfangene Daten unter dem Aspekt Generiert.
	 */
	private ResultData datenGeneriert;

	/**
	 * Aktuell empfangene Daten unter dem Aspekt Bearbeitet.
	 */
	private ResultData datenBearbeitet;

	/**
	 * Modelliert die Herkunft einer Meldung.
	 */
	public enum Quelle {
		/** In der VRZ manuell erzeugt. */
		Manuell,

		/** In der VRZ automatisch erzeugt. */
		Automatisch,

		/** Von der Landesmeldestelle erzeugt. */
		LMS;

		@Override
		public String toString() {
			switch (this) {
			case Manuell:
				return "manuell";
			case Automatisch:
				return "automatisch";
			case LMS:
				return "LMS";
			default:
				return super.toString();
			}

		}

		/**
		 * Bestimmt die Quelle zu einem Text.
		 * 
		 * @param text
		 *            der Text, zu dem die Quelle bestimmt werde soll
		 * @return die Quelle zum übergebenen Text
		 */
		public static Quelle get(final String text) {
			for (Quelle quelle : values()) {
				if (text.equals(quelle.toString())) {
					return quelle;
				}
			}
			throw new IllegalArgumentException(text + " ist kein Text zu enum Quelle");
		}
	}

	public enum TMCStatus {

		/** unbekannter Status. */
		Unbekannt("Unbekannt", -1),

		/** Neu. */
		Neu("Neu", 1),

		/** Ge&auml;ndert. */
		Geaendert("Geändert", 2),

		/** Gel&ouml;scht. */
		Geloescht("Gelöscht", 3);

		/**
		 * textueller Wert.
		 */
		private final String value;

		/**
		 * numerischer Wert.
		 */
		private int id;

		/**
		 * Privater Konstruktor.
		 *
		 * @param value
		 *            textueller Wert
		 * @param id
		 *            numerischer Wert
		 */
		TMCStatus(final String value, final int id) {
			this.value = value;
			this.id = id;
		}

		/**
		 * Gibt den numerischen Wert zur&uuml;ck.
		 *
		 * @return numerischer Wert des Status
		 */
		public int getId() {
			return id;
		}

		/**
		 * Gibt den textuellen Wert zur&uuml;ck.
		 *
		 * @return textueller Wert des Status
		 */
		public String getValue() {
			return value;
		}

		@Override
		public String toString() {
			return value;
		}

		/**
		 * Bestimmt den Zustand zu einem Text.
		 * 
		 * @param text
		 *            der Text, zu dem der Zustand bestimmt werde soll
		 * @return den Zustand zum übergebenen Text
		 */
		public static TMCStatus get(final String text) {
			for (TMCStatus status : values()) {
				if (text.equals(status.toString())) {
					return status;
				}
			}
			throw new IllegalArgumentException(text + " ist kein Text zu enum Status");
		}
	}

	/**
	 * Der Zustand einer Meldung.
	 */
	public enum Zustand {
		NichtQuittiert, Quittiert, Editiert, Verworfen;

		@Override
		public String toString() {
			switch (this) {
			case NichtQuittiert:
				return "nicht quittiert";
			case Quittiert:
				return "quittiert";
			case Editiert:
				return "editiert";
			case Verworfen:
				return "verworfen";
			default:
				return super.toString();
			}
		}

		/**
		 * Bestimmt den Zustand zu einem Text.
		 * 
		 * @param text
		 *            der Text, zu dem der Zustand bestimmt werde soll
		 * @return den Zustand zum übergebenen Text
		 */
		public static Zustand get(final String text) {
			for (Zustand zustand : values()) {
				if (text.equals(zustand.toString())) {
					return zustand;
				}
			}
			throw new IllegalArgumentException(text + " ist kein Text zu enum Zustand");
		}
	}

	/**
	 * Bestimmt den Wert des Attributs Quelle eines Datensatzes der
	 * atg.tmcVerkehrsMeldung.
	 * 
	 * @param data
	 *            der Datensatz
	 * @return der Wert des Attributs
	 */
	protected static Quelle getQuelle(final Data data) {
		final Quelle result = null;

		try {
			final Data vewInfo = data.getItem("TMCDaten").getItem("VerwaltungsInformationen");
			final TextValue tv = vewInfo.getTextValue("Erzeugungsart");
			return Quelle.get(tv.getText());
		} catch (final RuntimeException e) {
			DEBUG.warning("Quelle kann nicht bestimmt werden für Datum " + data.getName(), e);
		}

		return result;
	}

	/**
	 * Bestimmt den Wert des Attributs Zustand eines Datensatzes der
	 * atg.tmcVerkehrsMeldung.
	 * 
	 * @param data
	 *            der Datensatz
	 * @return der Wert des Attributs
	 */
	protected static Zustand getZustand(final Data data) {
		final Zustand result = null;

		try {
			final Data vewInfo = data.getItem("TMCDaten").getItem("VerwaltungsInformationen");
			final TextValue tv = vewInfo.getTextValue("BearbeitungsZustand");
			return Zustand.get(tv.getText());
		} catch (final RuntimeException e) {
			DEBUG.warning("Zustand kann nicht bestimmt werden für Datum " + data.getName(), e);
		}

		return result;
	}

	/**
	 * Bestimmt den Wert des Attributs Status eines Datensatzes der
	 * atg.tmcVerkehrsMeldung.
	 * 
	 * @param data
	 *            der Datensatz
	 * @return der Wert des Attributs
	 */
	protected static TMCStatus getStatus(final Data data) {
		final TMCStatus result = null;

		try {
			final Data vewInfo = data.getItem("TMCDaten").getItem("VerwaltungsInformationen");
			final TextValue tv = vewInfo.getTextValue("TmcStatus");
			return TMCStatus.get(tv.getText());
		} catch (final RuntimeException e) {
			DEBUG.warning("Status kann nicht bestimmt werden für Datum " + data.getName(), e);
		}

		return result;
	}

	/**
	 * Zugeordnetes Objekt vom Typ TMCVewSender - über dieses wird die
	 * Meldungspublikation für diese Meldung abgewickelt.
	 */
	private TMCVewSender sender;

	/**
	 * Konstruiert eine Instanz vom Typ TMCVewMeldung.
	 * 
	 * @param objekt
	 *            Systemobjekt vom Typ TMCVerkehrsmeldung
	 * @param verwaltung
	 *            übergeordnetes Verwaltungs-Objekt
	 */
	public TMCVewMeldung(final SystemObject objekt, final TMCVerwaltung verwaltung) {
		davObjekt = objekt;
		this.verwaltung = verwaltung;

		final AttributeGroup atg = getConnection().getDataModel().getAttributeGroup("atg.tmcVerkehrsMeldung");
		Aspect asp = getConnection().getDataModel().getAspect("asp.tmcGeneriert");
		datenBeschreibungGeneriert = new DataDescription(atg, asp);

		asp = getConnection().getDataModel().getAspect("asp.tmcBearbeitet");
		datenBeschreibungBearbeitet = new DataDescription(atg, asp);

		asp = getConnection().getDataModel().getAspect("asp.tmcVersendet");
		datenBeschreibungVersendet = new DataDescription(atg, asp);

		sender = new TMCVewSender(this);

		datenBearbeitet = new ParameterEmpfaenger(getConnection(), getDavObjekt(), "atg.tmcVerkehrsMeldung",
				datenBeschreibungBearbeitet).getData();

		getConnection().subscribeReceiver(this, davObjekt, datenBeschreibungBearbeitet, ReceiveOptions.normal(),
				ReceiverRole.drain());

		DEBUG.fine(datenBeschreibungBearbeitet + " zum Empfang angemeldet für " + davObjekt);

		getConnection().subscribeReceiver(this, davObjekt, datenBeschreibungGeneriert, ReceiveOptions.normal(),
				ReceiverRole.receiver());
		DEBUG.fine(datenBeschreibungGeneriert + " zum Empfang angemeldet für " + davObjekt);

		getConnection().subscribeReceiver(this, davObjekt, datenBeschreibungVersendet, ReceiveOptions.normal(),
				ReceiverRole.receiver());
		DEBUG.fine(datenBeschreibungVersendet + " zum Empfang angemeldet für " + davObjekt);

		angemeldet = true;
	}

	/**
	 * Gibt die mit dieser TMC-Meldung verbundenen Ressourcen frei.
	 */
	public void dispose() {
		if (isAngemeldet()) {
			getConnection().unsubscribeReceiver(this, davObjekt, datenBeschreibungBearbeitet);
			DEBUG.fine("Empfang abgemeldet von " + datenBeschreibungBearbeitet + " für " + davObjekt);

			getConnection().unsubscribeReceiver(this, davObjekt, datenBeschreibungGeneriert);
			DEBUG.fine("Empfang abgemeldet von " + datenBeschreibungGeneriert + " für " + davObjekt);

			getConnection().unsubscribeReceiver(this, davObjekt, datenBeschreibungVersendet);
			DEBUG.fine("Empfang abgemeldet von " + datenBeschreibungVersendet + " für " + davObjekt);

			angemeldet = false;
		}
		if (sender != null) {
			sender.dispose();
			sender = null;
		}
	}

	/**
	 * Gibt die die aktuelle DAV-Verbindung zurück.
	 * 
	 * @return die aktuelle DAV-Verbindung
	 */
	public ClientDavInterface getConnection() {
		return verwaltung.getConnection();
	}

	/**
	 * Setzt den Wert des Attributs Status in einem Datensatz der
	 * atg.tmcVerkehrsMeldung.
	 * 
	 * @param data
	 *            Datensatz der atg.tmcVerkehrsMeldung
	 * @param status
	 *            Wert des Attributs Status
	 */
	protected static void setStatus(final Data data, final TMCStatus status) {
		final Data vewInfo = data.getItem("TMCDaten").getItem("VerwaltungsInformationen");
		final TextValue tv = vewInfo.getTextValue("TmcStatus");
		tv.setText(status.toString());
	}

	/**
	 * Löscht das aktuelle dynamische Meldungsobjekt - auch die dynamische Menge
	 * wird bereinigt. Nach dem Löschen sollte diese Instanz nicht mehr
	 * verwendet und dem gc überlassen werden.
	 */
	protected void loesche() {
		DEBUG.config("Das Meldungsobjekt '" + getDavObjekt() + "' wird gelöscht");

		final MutableSet menge = verwaltung.getKonfigObjekt().getMutableSet(TMCVerwaltung.TMCMELDUNGEN);
		if (menge != null) {
			try {
				menge.remove(getDavObjekt());
			} catch (final ConfigurationChangeException e) {
				throw new RuntimeException(
						"Objekt " + getDavObjekt() + " konnte nicht aus Menge " + menge + "entfernt werden", e);
			}

			try {
				getDavObjekt().invalidate();
			} catch (final ConfigurationChangeException e) {
				throw new RuntimeException("Objekt " + getDavObjekt() + " konnte nicht gelöscht werden", e);
			}
		} else {
			throw new IllegalStateException("Dynamische Menge \"" + TMCVerwaltung.TMCMELDUNGEN
					+ "\" nicht gefunden für " + verwaltung.getKonfigObjekt());
		}
	}

	/** {@inheritDoc} */
	@Override
	public void update(final ResultData[] resultData) {
		for (ResultData data : resultData) {
			final Aspect asp = data.getDataDescription().getAspect();
			final AttributeGroup atg = data.getDataDescription().getAttributeGroup();

			if (atg.equals(datenBeschreibungGeneriert.getAttributeGroup())
					&& asp.equals(datenBeschreibungGeneriert.getAspect())) {
				if (angemeldet) {
					DEBUG.fine("Update für " + data.getObject() + " mit " + data.getDataDescription());
					datenGeneriert = data;
					bearbeiteDaten(data);
				} else {
					DEBUG.warning(
							"Update nicht erwartet, da Datenbeschreibung für dieses Objekt bereits abgemeldet oder nicht angemeldet: "
									+ "\tAktuelle Zeit    : " + getConnection().getTime() + "\tZeitstempel      : "
									+ data.getDataTime() + "\tObjekt           : " + data.getObject()
									+ "\tDatenbeschreibung: " + data.getDataDescription());
				}
			}

			if (atg.equals(datenBeschreibungVersendet.getAttributeGroup())
					&& asp.equals(datenBeschreibungVersendet.getAspect())) {
				if (angemeldet) {
					DEBUG.fine("Update für " + data.getObject() + " mit " + data.getDataDescription());

					// System.out.println("RDSVewMeldung: ---- Update für " +
					// data.getObject() + " mit " + data.getDataDescription());

					if (data.hasData()) {
						final Data datenSatz = data.getData().createModifiableCopy();
						final long zeitStempel = data.getDataTime();

						final Quelle quelle = getQuelle(datenSatz);
						final Zustand zustand = getZustand(datenSatz);
						final TMCStatus status = getStatus(datenSatz);

						DEBUG.fine("Bearbeite empfangene Daten für " + getDavObjekt().getPid() + " Aspekt: "
								+ asp.getName() + " Quelle: " + quelle + " Zustand: " + zustand + " Status: " + status
								+ " ZeitStempel: " + zeitStempel);

						if (status == TMCStatus.Geloescht) { // Lösche Objekt
							loesche();
						}
					}
				} else {
					DEBUG.warning(
							"Update nicht erwartet, da Datenbeschreibung für dieses Objekt bereits abgemeldet oder nicht angemeldet: "
									+ "\tAktuelle Zeit    : " + getConnection().getTime() + "\tZeitstempel      : "
									+ data.getDataTime() + "\tObjekt           : " + data.getObject()
									+ "\tDatenbeschreibung: " + data.getDataDescription());
				}
			}

			if (atg.equals(datenBeschreibungBearbeitet.getAttributeGroup())
					&& asp.equals(datenBeschreibungBearbeitet.getAspect())) {
				if (angemeldet) {
					DEBUG.fine("Update für " + data.getObject() + " mit " + data.getDataDescription());
					datenBearbeitet = data;
					bearbeiteDaten(data);
					if (data.hasData()) {
						/*
						 * Persistiere unter dem Aspekt TMCBearbeitet empfangene
						 * Daten als Parameter - da diese Daten als Senke
						 * empfangen wurden
						 */
						new ParameterSender(getConnection(), getDavObjekt(), data, "atg.tmcVerkehrsMeldung").start();
					}
				} else {
					DEBUG.warning(
							"Update nicht erwartet, da Datenbeschreibung für dieses Objekt bereits abgemeldet oder nicht angemeldet: "
									+ "\tAktuelle Zeit    : " + getConnection().getTime() + "\tZeitstempel      : "
									+ data.getDataTime() + "\tObjekt           : " + data.getObject()
									+ "\tDatenbeschreibung: " + data.getDataDescription());
				}
			}
		}
	}

	/**
	 * Prüft die übergebenen Daten und gibt sie ggf. zum Senden frei oder loggt
	 * entsprechende Fehlermeldung.
	 * 
	 * @param data
	 *            die aktuell weiterzuleitenden Daten
	 */
	private void bearbeiteDaten(final ResultData data) {
		if (!data.hasData()) {
			return;
		}

		final Aspect asp = data.getDataDescription().getAspect();
		final Data datenSatz = data.getData().createModifiableCopy();
		final long zeitStempel = data.getDataTime();

		final Quelle quelle = getQuelle(datenSatz);
		final Zustand zustand = getZustand(datenSatz);
		final TMCStatus status = getStatus(datenSatz);

		boolean loeschen = false;

		String text = "Bearbeite empfangene Daten für " + getDavObjekt().getPid() + "\nAspekt: " + asp.getName()
				+ "\nQuelle: " + quelle + "\nZustand: " + zustand + "\nStatus: " + status + "\nZeitStempel: "
				+ zeitStempel;
		final String suffixFehler = "\nEmpfangene Daten werden NICHT weitergeleitet";

		/*
		 * Prüfe Plausibilität Quelle/Zustand
		 */
		if (asp.equals(datenBeschreibungGeneriert.getAspect())) {
			if (quelle == Quelle.Automatisch) {
				/*
				 * An dieser Stelle sind Daten mit dem Zustand "nicht quittiert"
				 * zulässig. Sie werden verzögert weitergeleitet.
				 */
				if (zustand != Zustand.NichtQuittiert) {
					DEBUG.error(text + "\nFEHLER: Zustand " + zustand + " - für Quelle " + quelle + "/" + asp.getPid()
							+ " ist nur " + Zustand.NichtQuittiert + " zulässig" + suffixFehler);
					return;
				}
			} else {
				DEBUG.error(text + "\nFEHLER: Quelle " + quelle + " - für Aspekt " + asp.getName() + " ist nur "
						+ Quelle.Automatisch + " zulässig" + suffixFehler);
				return;
			}
		} else if (asp.equals(datenBeschreibungBearbeitet.getAspect())) {
			if (quelle == Quelle.Automatisch) {
				/*
				 * An dieser Stelle sind Daten mit dem Zuständen "quittiert",
				 * "editiert" und "verworfen" zulässig. Sie werden sofort
				 * weitergeleitet.
				 */
				if (zustand != Zustand.Quittiert && zustand != Zustand.Editiert && zustand != Zustand.Verworfen) {
					DEBUG.error(text + "\nFEHLER: Zustand " + zustand + " - für Quelle " + quelle + "/" + asp.getPid()
							+ " sind nur " + Zustand.Quittiert + ", " + Zustand.Editiert + " und " + Zustand.Verworfen
							+ " zulässig" + suffixFehler);
					return;
				}
			} else if (quelle == Quelle.Manuell) {
				/*
				 * An dieser Stelle sind Daten mit dem Zuständen "editiert" und
				 * "verworfen" zulässig. Sie werden sofort weitergeleitet.
				 */
				if (zustand != Zustand.Editiert && zustand != Zustand.Verworfen) {
					DEBUG.error(text + "\nFEHLER: Zustand " + zustand + " - für Quelle " + quelle + "/" + asp.getPid()
							+ " sind nur " + Zustand.Editiert + " und " + Zustand.Verworfen + " zulässig"
							+ suffixFehler);
					return;
				}
			} else {
				DEBUG.error(text + "\nFEHLER: Quelle " + quelle + " - für Aspekt " + asp.getName() + " sind nur "
						+ Quelle.Automatisch + " und " + Quelle.Manuell + " zulässig" + suffixFehler);
				return;
			}
		}

		/*
		 * Prüfe Zeitstempel generierte Daten: Diese werden nicht
		 * weitergeleitet, wenn es bearbeitete Daten oder gesendete Daten mit
		 * höherem Zeitstempel gibt.
		 */
		if (asp.equals(datenBeschreibungGeneriert.getAspect())) {
			if (datenBearbeitet != null && datenBearbeitet.hasData() && datenBearbeitet.getDataTime() > zeitStempel) {
				DEBUG.warning(text + "\nEs liegen bereits bearbeite Daten mit höherem Zeitstempel "
						+ datenBearbeitet.getDataTime() + " vor" + suffixFehler);
				return;
			} else if (sender.getDatenSenden() != null && sender.getDatenSenden().hasData()
					&& sender.getDatenSenden().getDataTime() > zeitStempel) {
				DEBUG.warning(text + "\nEs liegen bereits gesendete Daten mit höherem Zeitstempel "
						+ sender.getDatenSenden().getDataTime() + " vor" + suffixFehler);
				return;
			}
		}

		/*
		 * Prüfe Zeitstempel bearbeitete Daten: Diese werden nicht
		 * weitergeleitet, wenn der Zustand quittiert ist und es generierte
		 * Daten mit neuerem Zeitstempel gibt oder wenn es gesendete Daten mit
		 * höherem Zeitstempel gibt.
		 */
		if (asp.equals(datenBeschreibungBearbeitet.getAspect())) {

			if (zustand == Zustand.Quittiert && datenGeneriert != null && datenGeneriert.hasData()
					&& datenGeneriert.getDataTime() > zeitStempel) {
				DEBUG.warning(text + "\nEs liegen bereits generierte Daten mit höherem Zeitstempel "
						+ datenGeneriert.getDataTime() + " vor" + suffixFehler);
				return;
			} else if (sender.getDatenSenden() != null && sender.getDatenSenden().hasData()
					&& sender.getDatenSenden().getDataTime() > zeitStempel) {
				DEBUG.warning(text + "\nEs liegen bereits gesendete Daten mit höherem Zeitstempel "
						+ sender.getDatenSenden().getDataTime() + " vor" + suffixFehler);
				return;
			}
		}

		final Data datenSenden = sender.getDatenSenden() != null && sender.getDatenSenden().hasData()
				? sender.getDatenSenden().getData() : null;

		/*
		 * Wenn bereits Status Aufhebung gesendet wurde, wird nur noch Status
		 * Löschung weitergeleitet, wenn bereits Status Löschung gesendet wurde,
		 * werden keine Daten mehr weitergeleitet.
		 */
		if (datenSenden != null) {
			final TMCStatus sendeStatus = getStatus(datenSenden);
			if (sendeStatus == TMCStatus.Geloescht) {
				DEBUG.config(text + "\nEs liegt bereits eine Aktualisierung mit Status " + sendeStatus + " vor"
						+ suffixFehler);
				return;
			}
		}

		/*
		 * Zu sendende Daten unter dem Aspekt TMCGeneriert werden dann nicht
		 * weitergeleitet, wenn aktuell bearbeitete Daten mit dem Zustand
		 * editiert oder verworfen vorliegen
		 */
		if (asp.equals(datenBeschreibungGeneriert.getAspect())) {
			final Data datenBearbeitetData = datenBearbeitet != null && datenBearbeitet.hasData()
					? datenBearbeitet.getData() : null;
			final Zustand zustandBearbeitet = datenBearbeitetData != null ? getZustand(datenBearbeitetData) : null;
			if ((zustandBearbeitet == Zustand.Editiert || zustandBearbeitet == Zustand.Verworfen)) {
				DEBUG.config(
						text + "\nEs liegen bearbeitete Daten im Zustand " + zustandBearbeitet + " vor" + suffixFehler);
				return;
			}
		}

		/*
		 * Wenn zu sendende Daten unter dem Aspekt TMCBearbeitet den Zustand
		 * 'verworfen' haben, wird mindestens der Status 'Gelöscht' erzwungen
		 */
		if (asp.equals(datenBeschreibungBearbeitet.getAspect())) {
			if (zustand == Zustand.Verworfen) {
				if (status != TMCStatus.Geloescht) {
					text += "\nDer Status wird auf \"" + TMCStatus.Geloescht.toString()
							+ "\" gesetzt, da der Zustand \"" + Zustand.Verworfen + "\" ist";
					try {
						setStatus(datenSatz, TMCStatus.Geloescht);
					} catch (final RuntimeException e) {
						DEBUG.error(text + "\nFehler beim Setzen von Status auf " + TMCStatus.Geloescht + suffixFehler,
								e);
						return;
					}
				}
			}
		}

		/*
		 * Ggf. Löschen des dynamischen Meldungsobjekts. Nach dem
		 * Löschen/Invalidieren des dynamischen Meldungsobjekts werden für diese
		 * Instanz keine Aktualiserungen - auch nicht der Status "Löschung"
		 * selbst - weitergeleitet
		 */
		if (loeschen) {
			DEBUG.config(text + suffixFehler + "\nAktuelle Meldung wird gelöscht");
			try {
				loesche();
			} catch (final RuntimeException e) {
				DEBUG.error("Fehler beim Löschen von " + getDavObjekt(), e);
			}
		} else if (asp.equals(datenBeschreibungGeneriert.getAspect())) {
			/*
			 * Zu sendende Daten unter dem Aspekt TMCGeneriert werden verzögert
			 * gesendet Die Verzögerungs-/Rückhaltezeit wird (bis auf min. 0)
			 * nach unten korrigiert, wenn der Zeitstempel in der Vergangenheit
			 * liegt.
			 */
			final long verzoegerung = getRDSMeldungsRueckhaltung() * 1000;

			text += "\nEmpfangene generierte Daten werden zur Weiterleitung gemäß RDSMeldungsAktion in "
					+ Math.round(verzoegerung / 1000) + "s freigegeben";
			DEBUG.config(text);
			sender.starteSenden(datenSatz, zeitStempel, verzoegerung, true, asp);
		} else if (asp.equals(datenBeschreibungBearbeitet.getAspect())) {
			/*
			 * Zu sendende Daten unter dem Aspekt TMCBearbeitet werden sofort
			 * gesendet
			 */
			text += "\nEmpfangene bearbeitete Daten werden zur sofortigen Weiterleitung freigegeben";
			DEBUG.config(text);
			sender.starteSenden(datenSatz, zeitStempel, 0, true, asp);
		}
	}

	/**
	 * Gibt den Wert des Parameters RDSMeldungsRückhaltung zur&uuml;ck.
	 * 
	 * @return Wert des Parameters RDSMeldungsRückhaltung
	 */
	public long getRDSMeldungsRueckhaltung() {
		return verwaltung.getRDSMeldungsRueckhaltung();
	}

	/**
	 * Gibt den Wert des Parameters RDSMeldungsAktion zur&uuml;ck.
	 * 
	 * @return Wert des Parameters RDSMeldungsAktion
	 */
	public String getRDSMeldungsAktion() {
		return verwaltung.getRDSMeldungsAktion();
	}
}
