/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Die Klasse zum Versenden von Daten unter dem Aspekt "Senden" der
 * Attributgruppe "TMCVerkehrsmeldung". Sie implementiert das
 * ClientSenderInterface welches die Methoden zum Versenden der Daten
 * bereitstellt.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan
 * Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class TMCVewSender implements ClientSenderInterface {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Ist auf true gesetzt, solange auf Sendesteuerung gewartet wird.
	 */
	private boolean warten;

	/**
	 * Rolle, mit der als Sender angemeldet wird.
	 */
	public static final SenderRole ROLLE = SenderRole.source();

	/**
	 * Zeitin ms, die max. auf Sendesteuerung gewartet wird.
	 */
	public static final long SENDE_TIMEOUT = 5000;

	/**
	 * Ist diese Instanz zum DAV-Senden angemeldet?
	 */
	private boolean angemeldet;

	/**
	 * Ist diese Instanz zum DAV-Senden angemeldet?
	 * 
	 * @return <code>true</code>, wenn angemeldet, sonst <code>false</code>
	 */
	public boolean isAngemeldet() {
		return angemeldet;
	}

	/**
	 * Datenbeschreibung "gesendet".
	 */
	private final DataDescription datenBeschreibung;

	/**
	 * Übergeordnete Instanz vom Typ TMCVewMeldung.
	 */
	private final TMCVewMeldung meldung;
	

	/**
	 * Gibt die Datenverteiler-Verbindung zur&uuml;ck.
	 * 
	 * @return die Datenverteiler-Verbindung
	 */
	public ClientDavInterface getConnection() {
		return meldung.getConnection();
	}

	/**
	 * Gibt das Dav-Systemobject zur&uuml;ck, für das das Senden von Daten erfolgt.
	 * 
	 * @return {@link SystemObject}
	 */
	public SystemObject getDavObjekt() {
		return meldung.getDavObjekt();
	}

	/**
	 * Konstruktor der Klasse.
	 * 
	 * @param meldung
	 *            Objekt vom Typ TMCVewMeldung, für das das Senden von Daten
	 *            erfolgt
	 */
	public TMCVewSender(final TMCVewMeldung meldung) {
		this.meldung = meldung;

		final AttributeGroup atg = getConnection().getDataModel().getAttributeGroup("atg.tmcVerkehrsMeldung");
		final Aspect asp = getConnection().getDataModel().getAspect("asp.tmcSenden");
		datenBeschreibung = new DataDescription(atg, asp);

		try {
			getConnection().subscribeSender(this, getDavObjekt(), datenBeschreibung, ROLLE);
			DEBUG.fine(datenBeschreibung + " zum Senden angemeldet für " + getDavObjekt());
		} catch (final OneSubscriptionPerSendData e) {
			/*
			 * Es liegt bereits eine Sendeanmeldung von anderer Stelle vor =>
			 * hier nicht anelden
			 */
			DEBUG.warning(meldung.getDavObjekt().getPid() + " ist bereits an anderer Stelle zum Senden angemeldet", e);
			return;
		}
		angemeldet = true;
	}

	/**
	 * Gibt die mit dieser Instanz verbundenen Ressourcen wieder frei.
	 */
	public void dispose() {
		stopSenden();
		if (angemeldet) {
			getConnection().unsubscribeSender(this, getDavObjekt(), datenBeschreibung);
			DEBUG.fine("Senden abgemeldet von " + datenBeschreibung + " für " + getDavObjekt());
			angemeldet = false;
		}
	}

	/**
	 * Unmittelbares Senden von Daten.
	 * 
	 * @param resultData
	 *            die zu sendenden Daten
	 * @return true, wenn das Senden erfolgreich war, false sonst
	 */
	protected boolean send(final ResultData resultData) {
		try {
			getConnection().sendData(resultData);
		} catch (final SendSubscriptionNotConfirmed e) {
			if (SENDE_TIMEOUT > 0) {
				DEBUG.fine("Warte max. " + SENDE_TIMEOUT + "ms auf Sendesteuerung für " + getDavObjekt().getPid());
				synchronized (this) {
					warten = true;
					try {
						wait(SENDE_TIMEOUT);
					} catch (final InterruptedException ex) {
						// tue nichts
					}
					warten = false;
				}
				try {
					getConnection().sendData(resultData);
				} catch (final SendSubscriptionNotConfirmed ex) {
					DEBUG.warning("Keine Sendesteuerung für " + getDavObjekt().getPid(), ex);
					return false;
				}
				return true;
			}
		} catch (final DataNotSubscribedException e) {
			DEBUG.error("Keine Sendeanmeldung bzw. Sendeanmeldung ungültig für " + getDavObjekt().getPid(), e);
			return false;
		}
		return true;
	}

	@Override
	public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
		DEBUG.fine("dataRequest() für " + getDavObjekt().getPid() + " - Sender state: " + state);
		if (warten && state == ClientSenderInterface.START_SENDING) {
			synchronized (this) {
				notify();
			}
		}
	}

	@Override
	public boolean isRequestSupported(final SystemObject so, final DataDescription dd) {
		return true;
	}

	/**
	 * Aktuell unter dem Aspekt senden gesendete Daten.
	 */
	private ResultData datenSenden;

	/**
	 * Gibt die unter dem Aspekt senden aktuell gesendeten Daten zur&uuml;ck.
	 * 
	 * @return die unter dem Aspekt senden aktuell gesendeten Daten
	 */
	public ResultData getDatenSenden() {
		return datenSenden;
	}

	/**
	 * Setzt die unter dem Aspekt senden aktuell gesendeten Daten.
	 * 
	 * @param datenSenden
	 *            die aktuell gesendeten Daten
	 */
	void setDatenSenden(final ResultData datenSenden) {
		this.datenSenden = datenSenden;
	}

	/**
	 * Aktuell aktiver SendeThread.
	 */
	private SendeThread sendeThread;

	/**
	 * Gibt den aktuell aktiven SendeThread zur&uuml;ck.
	 * 
	 * @return den aktuell aktiven SendeThread
	 */
	SendeThread getSendeThread() {
		return sendeThread;
	}

	/**
	 * Setzt den aktuell aktiven SendeThread.
	 * 
	 * @param sendeThread
	 *            der zu setztende SendeThread
	 */
	void setSendeThread(final SendeThread sendeThread) {
		this.sendeThread = sendeThread;
	}

	/**
	 * Starte neuen SendeTread - ein evtl. laufender Sendethread wird durch den
	 * übergebenen ersetzt und ggf. abgebrochen
	 * 
	 * @param data
	 *            die zu sendenden Daten - der Sendeauftrag verwaltet eine Kopie
	 *            dieser Daten
	 * @param zeitStempel
	 *            Zeitstempel
	 * @param verzoegerung
	 *            die Sendeverzögerung in s
	 * @param bedingterAbbruch
	 *            wenn <code>true</code> wird ein laufender SendeThread
	 *            abgebrochen, wenn er mit Verzögerung gestartet wurde
	 * @param empfangsAspekt Aspekt
	 */
	public void starteSenden(final Data data, final long zeitStempel, final long verzoegerung,
			final boolean bedingterAbbruch, final Aspect empfangsAspekt) {
		if (bedingterAbbruch) {
			if (sendeThread != null
					&& sendeThread.empfangsAspekt.equals(meldung.getDatenBeschreibungGeneriert().getAspect())) {
				stopSenden();
				DEBUG.fine("Laufenden Sende-Auftrag abgebrochen für " + getDavObjekt().getPid());
			}
		}
		synchronized (this) {
			sendeThread = new SendeThread(data, zeitStempel, verzoegerung, empfangsAspekt);
			sendeThread.start();
		}
	}

	/**
	 * Abbrechen eines evtl. gestarteten und laufenden SendeThreads
	 */
	public void stopSenden() {
		synchronized (this) {
			if (sendeThread != null) {
				sendeThread.abbrechen();
			}
		}
	}

	/**
	 * Senden eines Datensatzes mit optionaler Verzögerung in einem eigenen
	 * Thread.
	 */
	class SendeThread extends Thread {
		/**
		 * Sende-Verzögerungszeit in ms.
		 */
		private final long verzoegerung;

		/**
		 * Datensatz, der gesendet werden soll.
		 */
		private final Data data;

		/**
		 * Zeitstempel der Daten.
		 */
		private final long zeitStempel;

		/**
		 * Aspekt, unter dem die zu sendenden Daten ursprünglich empfangen
		 * wurden.
		 */
		private final Aspect empfangsAspekt;

		/**
		 * Flag, das angibt, ob der Thread abgebrochen wurde.
		 */
		private boolean abgebrochen;

		/**
		 * Wurde der SendeThread abgebrochen?
		 * 
		 * @return wurde der SendeThread abgebrochen?
		 */
		boolean isAbgebrochen() {
			return abgebrochen;
		}

		/**
		 * Konstruiert einen neuen SendeThread zum Senden eines Datensatzes mit
		 * optionaler Sendeverzögerung.
		 * 
		 * @param data
		 *            der Datensatz, der gesendet werden soll
		 * @param zeitStempel Zeitstempel            
		 * @param verzoegerung
		 *            Sendeverzögerung in ms
		 * @param empfangsAspekt Aspekt
		 * 
		 */
		SendeThread(final Data data, final long zeitStempel, final long verzoegerung, final Aspect empfangsAspekt) {
			super();
			this.data = data;
			this.zeitStempel = zeitStempel;
			this.verzoegerung = verzoegerung;
			this.empfangsAspekt = empfangsAspekt;
		}

		/**
		 * Abbrechen des laufenden Threads.
		 */
		void abbrechen() {
			synchronized (this) {
				interrupt();
				abgebrochen = true;
			}
		}

		@Override
		public void run() {
			if (!isAbgebrochen() && verzoegerung > 0) {
				getConnection().sleep(verzoegerung);
			}
			synchronized (TMCVewSender.this) {
				try {
					if (!isAbgebrochen()) {
						final String text = "Daten mit Zeitstempel " + zeitStempel + " empfangen unter dem Aspekt "
								+ empfangsAspekt.getName();

						if (empfangsAspekt.equals(meldung.getDatenBeschreibungGeneriert().getAspect())
								&& TMCVewMeldung.getQuelle(data) == TMCVewMeldung.Quelle.Automatisch
								&& TMCVewMeldung.getZustand(data) == TMCVewMeldung.Zustand.NichtQuittiert
								&& meldung.getRDSMeldungsAktion().equals("0")) {
							DEBUG.fine(
									text + " werden NICHT unter dem Aspekt " + datenBeschreibung.getAspect().getName()
											+ " gesendet, da RDSMeldungsAktion den Wert \""
											+ meldung.getRDSMeldungsAktion() + "\" hat");
							return;
						}

						final ResultData resultData = new ResultData(getDavObjekt(), datenBeschreibung, zeitStempel,
								data);

						if (send(resultData)) {
							DEBUG.fine(text + " wurden erfolgreich unter dem Aspekt "
									+ datenBeschreibung.getAspect().getName() + " gesendet");

							setDatenSenden(resultData);
						}
					}
				} finally {
					if (getSendeThread() == this) {
						setSendeThread(null);
					}
				}
			}
		}
	}
}
