/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew.parameter;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Empfangen und Versenden eines Parameterdatensatzes. Empfängt einen Datensatz
 * einer Quell-ATG und kopiert ihn in einen Datensatz einer Ziel-ATG. Der
 * Datensatz der Ziel-ATG wird mit
 * {@link de.bsvrz.dav.daf.main.ClientDavInterface#createData(AttributeGroup)}
 * erzeugt und muss die Struktur des Quelldatensatzes exakt enthalten. muss,
 * aber eine unterschiedliche ATG haben Die Empfangs-Anmeldung -und Abmeldung
 * erfolgt im Konstruktor.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class ParameterEmpfaenger implements ClientReceiverInterface {

	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Zeit in ms, die max. auf Sendesteuerung gewartet wird.
	 */
	static final long TIMEOUT = 5000;

	/**
	 * Die Datenverteilerverbindung.
	 */
	private final ClientDavInterface connection;

	/**
	 * Das DAV-Objekt, für das das Senden erfolgt.
	 */
	private final SystemObject davObjekt;

	/**
	 * Datenbeschreibung der Daten, die aus der Parametrierung gelesen werden.
	 */
	private final DataDescription datenBeschreibungQuell;

	/**
	 * Datenbeschreibung unter der die Daten zurückgegeben werden.
	 */
	private final DataDescription datenBeschreibungZiel;

	/**
	 * Ist auf true gesetzt, solange auf update gewartet wird.
	 */
	private boolean warten;

	/**
	 * Empfangene Daten.
	 */
	private ResultData davData;

	
	/**
	 * Konstruktor.
	 * 
	 * @param con {@link ClientDavInterface}
	 * @param obj {@link SystemObject}
	 * @param atgQuellPid PID der Quell-Attributgruppe
	 * @param datenBeschreibungZiel {@link DataDescription} für Zieldaten
	 */
	public ParameterEmpfaenger(final ClientDavInterface con, final SystemObject obj, final String atgQuellPid,
			final DataDescription datenBeschreibungZiel) {
		connection = con;
		davObjekt = obj;

		final AttributeGroup atg = connection.getDataModel().getAttributeGroup(atgQuellPid);
		final Aspect asp = connection.getDataModel().getAspect("asp.parameterSoll");
		datenBeschreibungQuell = new DataDescription(atg, asp);

		this.datenBeschreibungZiel = datenBeschreibungZiel;

		connection.subscribeReceiver(this, davObjekt, datenBeschreibungQuell, ReceiveOptions.normal(),
				ReceiverRole.receiver());
		DEBUG.finest(datenBeschreibungQuell + " zum Empfang angemeldet für " + davObjekt);

		DEBUG.finer("Warte max. " + TIMEOUT + "ms auf Update Parameter-Daten von " + atg.getPid() + " für "
				+ davObjekt.getPid());
		synchronized (this) {
			if (getData() == null) {
				warten = true;
				try {
					wait(TIMEOUT);
				} catch (final InterruptedException e) {
					// tue nichts
				}
				warten = false;
			}
		}

		if (getData() == null) {
			DEBUG.warning("Keine Parameter-Daten empfangen von " + atg.getPid() + " für " + davObjekt.getPid());
		}

		connection.unsubscribeReceiver(this, davObjekt, datenBeschreibungQuell);
		DEBUG.finest("Empfang abgemeldet von " + datenBeschreibungQuell + " für " + davObjekt);
	}


	/**
	 * Gibt die empfangenen Daten zur&uuml;ck.
	 * 
	 * @return {@link ResultData}
	 */
	public ResultData getData() {
		return davData;
	}

	@Override
	public void update(final ResultData[] resultData) {
		for (ResultData data : resultData) {
			final Aspect asp = data.getDataDescription().getAspect();
			final AttributeGroup atg = data.getDataDescription().getAttributeGroup();

			if (atg.equals(datenBeschreibungQuell.getAttributeGroup())
					&& asp.equals(datenBeschreibungQuell.getAspect())) {
				if (data.hasData()) {
					final Data datenSatz = connection.createData(datenBeschreibungZiel.getAttributeGroup());
					DatenTools.tiefeKopie(data.getData(), datenSatz);
					davData = new ResultData(davObjekt, datenBeschreibungZiel, data.getDataTime(), datenSatz);
					DEBUG.fine(
							"Bestehende Parameter-Daten empfangen von " + atg.getPid() + " für " + davObjekt.getPid());
				} else {
					davData = new ResultData(davObjekt, datenBeschreibungZiel, data.getDataTime(), null);
				}
			}
		}
		if (warten) {
			synchronized (this) {
				notify();
			}
		}
	}
}
