/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew.parameter;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Kopieren und Versenden eines Parameterdatensatzes. Kopiert einen Datensatz in
 * einen anderen Datensatz, der die Struktur des Quelldatensatzes exakt
 * enthalten muss, aber eine unterschiedliche ATG haben kann, und bietet
 * Funktionalität zum Versenden der Daten.<br>
 * Das Senden der Daten unter asp.parameterVorgabe erfolgt nach dem Starten des
 * Threads, der eine Instanz dieser Klasse verkörpert.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Stefan Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class ParameterSender extends Thread implements ClientSenderInterface {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Zeitin ms, die max. auf Sendesteuerung gewartet wird.
	 */
	static final long TIMEOUT = 5000;

	/**
	 * Die Datenverteilerverbindung.
	 */
	private final ClientDavInterface connection;

	/**
	 * Das DAV-Objekt, für das das Senden erfolgt.
	 */
	private final SystemObject davObjekt;

	/**
	 * Datenbeschreibung der Daten, die verarbeitet werden.
	 */
	private final DataDescription datenBeschreibung;

	/**
	 * Die Daten, die gesendet werden sollen.
	 */
	private final ResultData daten;

	/**
	 * Ist auf true gesetzt, solange auf Sendesteuerung gewartet wird.
	 */
	private boolean warten;

	/**
	 * Konstruiert eine neue Instanz vom Typ <code>ParameterSender</code>. Die
	 * übergebenen Daten werden auf die angegebene Parameter-ATG kopiert und
	 * nach starten des Threads unter dem Aspekt asp.parameterVorgabe gesendet.
	 * 
	 * @param con
	 *            ClientDavInterface 
	 * @param obj
	 *            Systemobjekt 
	 * @param data
	 *            die Daten, die gesendet werden sollen
	 * @param atgPid
	 *            die PID der Parameter-ATG, auf die die Daten kopiert werden
	 *            sollen
	 */
	public ParameterSender(final ClientDavInterface con, final SystemObject obj, final ResultData data, final String atgPid) {
		super();

		connection = con;
		davObjekt = obj;

		final AttributeGroup atg = connection.getDataModel().getAttributeGroup(atgPid);
		final Aspect asp = connection.getDataModel().getAspect("asp.parameterVorgabe");
		datenBeschreibung = new DataDescription(atg, asp);

		final Data datenSatz = connection.createData(atg);
		DatenTools.tiefeKopie(data.getData(), datenSatz);

		daten = new ResultData(davObjekt, datenBeschreibung, data.getDataTime(), datenSatz);
	}

	@Override
	public void run() {
		boolean angemeldet = false;
		
		try {
			try {
				connection.subscribeSender(this, davObjekt, datenBeschreibung, SenderRole.sender());
				angemeldet = true;
			} catch (final OneSubscriptionPerSendData e) {
				// angemeldet bleibt false
			}

			try {
				connection.sendData(daten);
				DEBUG.finer("Parameter erfolgreich aktualisiert mit " + datenBeschreibung.getAttributeGroup() + " für "
						+ davObjekt);
			} catch (final SendSubscriptionNotConfirmed e) {
				if (TIMEOUT > 0) {
					DEBUG.finer("Warte max. " + TIMEOUT + "ms auf Sendesteuerung für " + davObjekt.getPid());
					synchronized (this) {
						warten = true;
						try {
							wait(TIMEOUT);
						} catch (final InterruptedException ex) {
							// tue nichts
						}
						warten = false;
					}
					try {
						connection.sendData(daten);
					} catch (final SendSubscriptionNotConfirmed ex) {
						DEBUG.warning("Keine Sendesteuerung für " + davObjekt.getPid(), ex);
					}
				} 
			} catch (final DataNotSubscribedException e) {
				DEBUG.error("Keine Sendeanmeldung bzw. Sendeanmeldung ungültig für " + davObjekt.getPid(), e);
			}
		} finally {
			if (angemeldet) {
				connection.unsubscribeSender(this, davObjekt, datenBeschreibung);
			}
		}
	}

	@Override
	public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
		DEBUG.finest("dataRequest() für " + davObjekt.getPid() + " - Sender state: " + state);
		if (warten && state == ClientSenderInterface.START_SENDING) {
			synchronized (this) {
				notify();
			}
		}
	}

	@Override
	public boolean isRequestSupported(final SystemObject so, final DataDescription dd) {
		return true;
	}
}
