/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Die Klasse zum Versenden von Daten unter dem Aspekt "Senden" der
 * Attributgruppe "TMCVerkehrsMeldung"". Sie implementiert das
 * ClientSenderInterface welches die Methoden zum Versenden der Daten
 * bereitstellt.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Meldungen von: Dambach Werke GmbH, Stefan
 * Sans
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $
 */
public class Meldung implements ClientSenderInterface {
	/**
	 * Debug-Logger für Logging-Ausgaben.
	 */
	private static final Debug DEBUG = Debug.getLogger();

	/**
	 * Ist auf true gesetzt, solange auf Sendesteuerung gewartet wird.
	 */
	private boolean warten;

	/**
	 * Rolle, mit der als Quelle angemeldet wird.
	 */
	public static final SenderRole QUELLE = SenderRole.source(); // Quelle

	/**
	 * Rolle, mit der als Sender angemeldet wird.
	 */
	public static final SenderRole SENDER = SenderRole.sender(); // Sender

	/**
	 * Zeitin ms, die max. auf Sendesteuerung gewartet wird.
	 */
	public static final long SENDE_TIMEOUT = 5000;

	/**
	 * Ist diese Instanz zum DAV-Senden angemeldet?
	 */
	private boolean angemeldet;

	/**
	 * @return ist diese Instanz zum DAV-Senden angemeldet?
	 */
	public boolean isAngemeldet() {
		return angemeldet;
	}

	/**
	 * Datenbeschreibung "generiert".
	 */
	protected final DataDescription _datenBeschreibung;

	private ClientDavInterface connection;

	/**
	 * @return die Datenverteiler-Verbindung.
	 */
	public ClientDavInterface getConnection() {
		return connection;
	}

	private SystemObject davObjekt;

	/**
	 * @return das Dav-Systemobject, für das das Senden von Daten erfolgt.
	 */
	public SystemObject getDavObjekt() {
		return davObjekt;
	}

	/**
	 * Konstruktor der Klasse.
	 * 
	 * @param connection
	 *            {@link ClientDavInterface}
	 * @param davObjekt
	 *            Objekt vom Typ Meldung, für das das Senden von Daten erfolgt
	 * @param aspekt
	 *            Aspect
	 */
	public Meldung(final ClientDavInterface connection, final SystemObject davObjekt, final String aspekt) {
		this.connection = connection;
		this.davObjekt = davObjekt;

		String text = null;

		final AttributeGroup atg = getConnection().getDataModel().getAttributeGroup(RdsMeldung.MELDUNGEN_ATG);

		if ("generiert".equalsIgnoreCase(aspekt)) {
			text = "asp.tmcGeneriert";
		}

		if ("bearbeitet".equalsIgnoreCase(aspekt)) {
			text = "asp.tmcBearbeitet";
		}

		if ("versendet".equalsIgnoreCase(aspekt)) {
			text = "asp.tmcVersendet";
		}

		final Aspect asp = getConnection().getDataModel().getAspect(text);

		_datenBeschreibung = new DataDescription(atg, asp);

		try {
			if ("generiert".equalsIgnoreCase(aspekt) || "versendet".equalsIgnoreCase(aspekt)) {
				getConnection().subscribeSender(this, getDavObjekt(), _datenBeschreibung, QUELLE);
			} else {
				getConnection().subscribeSender(this, getDavObjekt(), _datenBeschreibung, SENDER);
			}
			// System.out.println("Meldung-> zum Senden angemeldet für " +
			// getDavObjekt());
			angemeldet = true;
		} catch (final OneSubscriptionPerSendData e) {
			/*
			 * Es liegt bereits eine Sendeanmeldung von anderer Stelle vor =>
			 * hier nicht anmnelden
			 */
			if (false) {
				System.out.println(
						" " + getDavObjekt().getPid() + " ist bereits an anderer Stelle zum Senden angemeldet");
			}
		}
	}

	/**
	 * Gibt die mit dieser Instanz verbundenen Ressourcen wieder frei
	 */
	public void dispose() {
		if (angemeldet) {
			getConnection().unsubscribeSender(this, getDavObjekt(), _datenBeschreibung);
			DEBUG.config("Senden abgemeldet von " + _datenBeschreibung + " für " + getDavObjekt());
			angemeldet = false;
		}
	}

	/**
	 * Unmittelbares Senden von Daten.
	 * 
	 * @param resultData
	 *            die zu sendenden Daten
	 * @return true, wenn das Senden erfolgreich war, false sonst
	 */
	protected boolean send(final ResultData resultData) {
		try {
			getConnection().sendData(resultData);
		} catch (final SendSubscriptionNotConfirmed e) {
			if (SENDE_TIMEOUT > 0) {
				// _debug.config("Warte max. " + SENDE_TIMEOUT + "ms auf
				// Sendesteuerung für " + getDavObjekt().getPid());
				System.out.println(
						"Warte max. " + SENDE_TIMEOUT + "ms auf Sendesteuerung für " + getDavObjekt().getPid());
				synchronized (this) {
					warten = true;
					try {
						wait(SENDE_TIMEOUT);
					} catch (final InterruptedException ex) {
						// tue nichts
					}
					warten = false;
				}
				try {
					getConnection().sendData(resultData);
				} catch (final SendSubscriptionNotConfirmed ex) {
					System.out.println("FEHLER: " + ex + "\nKeine Sendesteuerung für " + getDavObjekt().getPid()
							+ " mit " + _datenBeschreibung);
					// _debug.warning("Keine Sendesteuerung für " +
					// getDavObjekt().getPid(), ex);
					return false;
				}
				return true;
			} else {
				System.out.println("FEHLER: " + e + "\nKeine Sendesteuerung für " + getDavObjekt().getPid() + " mit "
						+ _datenBeschreibung);
				// _debug.warning("Keine Sendesteuerung für " +
				// getDavObjekt().getPid(), e);
				return false;
			}
		} catch (final DataNotSubscribedException e) {
			System.out.println("FEHLER: " + e + "\nKeine Sendeanmeldung bzw. Sendeanmeldung ungültig für "
					+ getDavObjekt().getPid() + " mit " + _datenBeschreibung);
			// _debug.error("Keine Sendeanmeldung bzw. Sendeanmeldung ungültig
			// für " + getDavObjekt().getPid(), e);
			return false;
		}
		return true;
	}

	/*
	 * (Kein Javadoc)
	 * 
	 * @see
	 * de.bsvrz.dav.daf.main.ClientSenderInterface#dataRequest(de.bsvrz.dav.daf.
	 * main.config.SystemObject, de.bsvrz.dav.daf.main.DataDescription, byte)
	 */
	public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
		DEBUG.config("dataRequest() für " + getDavObjekt().getPid() + " - Sender state: " + state);
		if (warten && state == ClientSenderInterface.START_SENDING) {
			synchronized (this) {
				notify();
			}
		}
	}

	/*
	 * (Kein Javadoc)
	 * 
	 * @see
	 * de.bsvrz.dav.daf.main.ClientSenderInterface#isRequestSupported(de.bsvrz.
	 * dav.daf.main.config.SystemObject, de.bsvrz.dav.daf.main.DataDescription)
	 */
	public boolean isRequestSupported(final SystemObject so, final DataDescription dd) {
		return true;
	}

	/**
	 * Aktuell unter dem Aspekt senden gesendete Daten
	 */
	private ResultData _datenSenden = null;

	/**
	 * @return die unter dem Aspekt senden aktuell gesendeten Daten
	 */
	public ResultData getDatenSenden() {
		return _datenSenden;
	}

	/**
	 * Setzt die unter dem Aspekt senden aktuell gesendeten Daten.
	 * 
	 * @param datenSenden
	 *            die aktuell gesendeten Daten
	 */
	void setDatenSenden(final ResultData datenSenden) {
		_datenSenden = datenSenden;
	}

	/**
	 * Bildet eine sendefertige ResultData-Instanz für das aktuelle
	 * Meldungsobjekt mit dem Zeitstempel der aktuellen Zeit und einem
	 * DAV-Datensatz für das übergebene Ereignis.
	 * 
	 * @param quelle
	 *            Quelle
	 * @param zustand
	 *            Zustand
	 * @param aspekt
	 *            Aspekt
	 * @param status
	 *            Status
	 * 
	 * @return ResultData Instanz
	 */
	public ResultData getResultData(final String quelle, final String zustand, final String aspekt,
			final String status) {
		String text = null;
		final Data data = getData(quelle, zustand, status);
		final AttributeGroup atg = getConnection().getDataModel().getAttributeGroup(RdsMeldung.MELDUNGEN_ATG);

		if (aspekt.equalsIgnoreCase("generiert")) {
			text = "asp.tmcGeneriert";
		}

		if (aspekt.equalsIgnoreCase("bearbeitet")) {
			text = "asp.tmcBearbeitet";
		}

		if (aspekt.equalsIgnoreCase("versendet")) {
			text = "asp.tmcVersendet";
		}

		final Aspect asp = getConnection().getDataModel().getAspect(text);
		return new ResultData(getDavObjekt(), new DataDescription(atg, asp), getConnection().getTime(), data);
	}

	/**
	 * Bildet einen DAV-Datensatz zur vorliegenden Meldung und initialisiert die
	 * relevanten Attribute bezgl. Location(s) und Ereignis.
	 * 
	 * @param quelle Quelle
	 * @param zustand Zustand
	 * @param status Status
	 * 
	 * @return den gebildeten DAV-Datensatz
	 */
	private Data getData(final String quelle, final String zustand, final String status) {
		final AttributeGroup atg = getConnection().getDataModel().getAttributeGroup(RdsMeldung.MELDUNGEN_ATG);
		final Data data = getConnection().createData(atg);

		Data item = data.getItem("TMCID");
		item.getTextValue("Datenproduzent").setText("");
		item.getTextValue("Datenkennung").setText("");

		data.getTextValue("Beschreibung").setText("");

		item = data.getItem("TMCDaten");

		final Data vi = item.getItem("VerwaltungsInformationen");
		if ("0".equalsIgnoreCase(status)) {
			vi.getItem("TmcStatus").asTextValue().setText("Geändert");
		}

		if ("1".equalsIgnoreCase(status)) {
			vi.getItem("TmcStatus").asTextValue().setText("Gelöscht");
		}

		if ("9".equalsIgnoreCase(status)) {
			vi.getItem("TmcStatus").asTextValue().setText("Gelöscht");
		}

		if ("nicht quittiert".equalsIgnoreCase(zustand)) {
			vi.getItem("BearbeitungsZustand").asTextValue().setText("nicht quittiert");
		}

		if ("quittiert".equalsIgnoreCase(zustand)) {
			vi.getItem("BearbeitungsZustand").asTextValue().setText("quittiert");
		}

		if ("editiert".equalsIgnoreCase(zustand)) {
			vi.getItem("BearbeitungsZustand").asTextValue().setText("editiert");
		}

		if ("verworfen".equalsIgnoreCase(zustand)) {
			vi.getItem("BearbeitungsZustand").asTextValue().setText("verworfen");
		}

		if ("Automatik".equalsIgnoreCase(quelle)) {
			vi.getItem("Erzeugungsart").asTextValue().setText("automatisch");
		}

		if ("Manuell".equalsIgnoreCase(quelle)) {
			vi.getItem("Erzeugungsart").asTextValue().setText("manuell");
		}
		// data.getItem("InfoGUID").asTextValue().setText("");
		//
		// final Data dataId = data.getItem("ID");
		// dataId.getItem("IDOrganisation").asTextValue().setText("");
		// dataId.getItem("IDNummer").asTextValue().setText("");
		//
		// final Data dataVersion = data.getItem("Version");
		// dataVersion.getItem("VersionGUID").asTextValue().setText("");
		//
		// final Data dataVerwaltung =
		// dataVersion.getItem("VerwaltungsInformationen");
		// dataVerwaltung.getItem("AuthorisierendeOrganisationsEinheit").asTextValue().setText("");
		// dataVerwaltung.getItem("AuthorisierenderNutzer").asTextValue().setText("");
		// dataVerwaltung.getItem("AuthorisierungsZeit").asTimeValue().setMillis(1);
		// dataVerwaltung.getItem("AuthorisierungsKommentar").asTextValue().setText("");
		// //
		// dataVerwaltung.getItem("AuthorisierungsErgebnis").asUnscaledValue().set(0);
		// dataVerwaltung.getItem("VersionsNummer").asUnscaledValue().set(1);
		//
		// dataVerwaltung.getItem("AktualisierungsZeit").asTimeValue().setMillis(getConnection().getTime());
		// dataVerwaltung.getItem("AktivierungsZeit").asTimeValue().setMillis(1);
		// dataVerwaltung.getItem("AblaufZeit").asTimeValue().setMillis(1);
		// dataVerwaltung.getItem("ErinnerungsZeit").asTimeValue().setMillis(1);
		// dataVerwaltung.getItem("ErinnerungsZeitOffset").asTimeValue().setMillis(0);
		// dataVerwaltung.getItem("ErzeugungsZeit").asTimeValue().setMillis(getConnection().getTime());
		// final Data dataAenderungsInfo =
		// dataVerwaltung.getItem("ÄnderungsInformationen");
		// dataAenderungsInfo.getItem("Veranlasser").asTextValue().setText("");
		// dataAenderungsInfo.getItem("ModifikationsZeitpunkt").asTimeValue().setMillis(1);
		// dataAenderungsInfo.getItem("ModifikationsKommentar").asTextValue().setText("");
		//
		// if (status.equalsIgnoreCase("0")) {
		// dataVerwaltung.getItem("Status").asTextValue().setText("Modifikation");
		// }
		//
		// if (status.equalsIgnoreCase("1")) {
		// dataVerwaltung.getItem("Status").asTextValue().setText("Aufhebung");
		// }
		//
		// if (status.equalsIgnoreCase("9")) {
		// dataVerwaltung.getItem("Status").asTextValue().setText("Löschung");
		// }
		//
		// dataVerwaltung.getItem("Priorität").asUnscaledValue().set(0); // 0 ==
		// // "Normal"
		// dataVerwaltung.getItem("LandesKennung").asTextValue().setText("rdsLandesKennung.DE");
		//
		// if (zustand.equalsIgnoreCase("nicht quittiert")) {
		// dataVerwaltung.getItem("Zustand").asTextValue().setText("nicht
		// quittiert");
		// }
		//
		// if (zustand.equalsIgnoreCase("quittiert")) {
		// dataVerwaltung.getItem("Zustand").asTextValue().setText("quittiert");
		// }
		//
		// if (zustand.equalsIgnoreCase("editiert")) {
		// dataVerwaltung.getItem("Zustand").asTextValue().setText("editiert");
		// }
		//
		// if (zustand.equalsIgnoreCase("verworfen")) {
		// dataVerwaltung.getItem("Zustand").asTextValue().setText("verworfen");
		// }
		//
		// if (quelle.equalsIgnoreCase("Automatik")) {
		// dataVerwaltung.getItem("Quelle").asTextValue().setText("automatisch");
		// }
		//
		// if (quelle.equalsIgnoreCase("Manuell")) {
		// dataVerwaltung.getItem("Quelle").asTextValue().setText("manuell");
		// }
		//
		// final Data dataVerkehr =
		// dataVersion.getItem("VerkehrsInformationen");
		// dataVerkehr.getItem("RDSFormat").asUnscaledValue().set(0);
		//
		// final Data dataNachrichten = dataVerkehr.getItem("Nachrichten");
		// dataNachrichten.getItem("NachrichtenSprache").asUnscaledValue().set(1031);
		//
		// final Data dataFreierText = dataVerkehr.getItem("FreierText");
		// dataFreierText.getItem("Text").asTextValue().setText("");
		// dataFreierText.getItem("TextSprache").asUnscaledValue().set(1031);
		//
		// final Data dataLocationTabelleInfo =
		// dataVerkehr.getItem("LocationTabelleInfo");
		// dataLocationTabelleInfo.getItem("LocationTabelle").asUnscaledValue().set(1001);
		// dataLocationTabelleInfo.getItem("LocationTabelleVersion").asTextValue().setText("");
		// dataLocationTabelleInfo.getItem("LocationTabelleTyp").asUnscaledValue().set(0);
		// dataLocationTabelleInfo.getItem("LocationTabelleName").asTextValue().setText("");
		//
		// final Data dataLocationDaten = dataVerkehr.getItem("LocationDaten");
		// final Data dataRDSLocation =
		// dataLocationDaten.getItem("RDSLocation"); // !!!
		// // Location
		// // Codes
		// dataLocationDaten.getItem("TMCRichtung").asTextValue().setText("Positiv");
		// dataLocationDaten.getItem("LocationPrimärEntfernung").asUnscaledValue().set(0);
		// dataLocationDaten.getItem("LocationSekundärEntfernung").asUnscaledValue().set(0);
		// dataLocationDaten.getItem("LocationVorLocation").asUnscaledValue().set(0);
		// dataLocationDaten.getItem("LocationNachLocation").asUnscaledValue().set(0);
		//
		// final Data dataEreignis = dataVerkehr.getItem("Ereignis");
		// final Data dataEreignisDaten = dataEreignis.getItem("EreignisDaten");
		// // !!!
		// // Ereignis
		// // Codes
		// dataEreignis.getItem("EreignisTabelleNummer").asUnscaledValue().set(0);
		// dataEreignis.getItem("EreignisTabelleVersion").asTextValue().setText("");
		// final NumberArray arrayEreignisTyp =
		// dataEreignis.getUnscaledArray("EreignisTyp");
		// arrayEreignisTyp.setLength(1);
		// arrayEreignisTyp.getValue(0).set(0);
		//
		// final Data dataZusatzAttribute =
		// dataVerkehr.getItem("ZusatzAttribute");
		//
		// dataVerkehr.getItem("AlertCCode").asTextValue().setText("");

		return data;
	}
}
