/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.ste.tmcvew;

import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Klasse mit Methoden zum Empfangen und Bearbeiten der Rdsmeldungen. Dabei wird
 * die Attributgruppe "atg.rdsMeldung" unter dem Aspekt "asp.rdsSenden"
 * ausgewertet. Die Klasse wird pro Objekt instanziert.
 * 
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Verwaltung von: Dambach Werke GmbH, Hilpp
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $ 
 */
public class RdsMeldungsEmpfang implements ClientReceiverInterface {
	/**
	 * Debug Möglichkeit.
	 */
	private boolean _debugAnmelde = false;
	private Debug debug;
	private boolean _debug = false;

	/**
	 * Verbindung zum DaV
	 */
	private ClientDavConnection m_connection;

	/**
	 * Definition der Attributgruppe.
	 */
	private String m_atg = RdsMeldung.MELDUNGEN_ATG;

	/**
	 * benötigter Aspekt.
	 */
	private String m_asp = "asp.tmcSenden";

	/**
	 * Liste zur Verwaltung der Listener.
	 */

	private List<INeueDatenListener> m_angemeldeteListener = new LinkedList<INeueDatenListener>();

	/**
	 * Pid des Objekts
	 */
	private String m_pid;

	private String m_status;

	private String m_zustand;

	private String m_quelle;

	/**
	 * Konstruktor der Klasse. Konstruktor meldet sich beim DaV zum Empfang der
	 * Daten an. Attributgruppe "atg.rdsMeldung" und Aspekt "asp.rdsSenden"
	 * 
	 * @param connection
	 *            Verbindung zum DaV
	 * @param pid
	 *            Pid eines Objekts vom Typ "typ.rdsMeldung"
	 */
	public RdsMeldungsEmpfang(ClientDavConnection connection, String pid) {
		debug = Debug.getLogger();

		m_connection = connection;

		m_pid = pid;

		anmeldeLeseDaV(pid, m_atg, m_asp, ReceiverRole.receiver());
	}

	/**
	 * Update Methode des ClientReceiverInterface. Die Daten werden aus der
	 * Attributgruppe "atg.situationsEigenschaften" ausgelesen und als
	 * Klassenvariable gespeichert. Danach wird der Listener benachrichtigt.
	 * Tritt eine {@link Exception} auf wird der entsprechende Offset auf 0
	 * gesetzt.
	 * 
	 * @param arg0
	 *            beinhaltet die empfangenen Daten
	 */
	public void update(final ResultData[] arg0) {
		// Alle empfangene Daten auslesen

		for (int i = 0; i < arg0.length; i++) {
			final ResultData dat = arg0[i];

			// ----------------------------------------------------------------------
			// Auslesen der "atg.tmcVerkehrsMeldung"
			// ----------------------------------------------------------------------

			if (dat.getDataDescription().getAttributeGroup().getPid().equals(m_atg)) {

				// Absender feststellen

				final String absenderPid = dat.getObject().getPid();

				if (dat.hasData()) {
					final Data vewInfo = dat.getData().getItem("TMCDaten").getItem("VerwaltungsInformationen");

					// Data vewInfo =
					// dat.getData().getItem("Version").getItem("VerwaltungsInformationen");

					m_status = vewInfo.getTextValue("TmcStatus").getText();

					m_zustand = vewInfo.getItem("BearbeitungsZustand").asTextValue().getValueText();

					m_quelle = vewInfo.getItem("Erzeugungsart").asTextValue().getValueText();
					
					final String log = "< TmcMeldung von Verwaltung (" + dat.getDataDescription().getAspect() + "):  " + new Date().toString() 
							+ ", PID: " + absenderPid + ", Status: " + m_status + ", Zustand: " + m_zustand + ",  Quelle: " + m_quelle;

					System.out.println(log);

					debug.info(log);

					benachrichtigeListener();

				}
			}

		} // for (int i = 0; i < arg0.length; i++)
	}

	public String getStatus() {
		return m_status;
	}

	public String getQuelle() {
		return m_quelle;
	}

	public String getZustand() {
		return m_zustand;
	}

	/**
	 * Methode die das Objekt mit der PID objPid beim Datenverteiler anmeldet
	 * zum Lesen der Attibutgruppe atgPid unter dem Aspekt aspPid.
	 * 
	 * @param objPid
	 *            Pid des Objekts
	 * @param atgPid
	 *            Attributgruppe die angemeldet werden soll
	 * @param aspPid
	 *            Apekt der angemeldet werden soll
	 * @param role
	 *            Rolle des Empfängers (siehe
	 *            stauma.dav.clientside.ReceiveOptions)
	 */
	private void anmeldeLeseDaV(String objPid, String atgPid, String aspPid, ReceiverRole role) {
		String buffer = "Anmelden am DaV (Lesen): " + objPid + " " + atgPid + "  " + aspPid;

		if (_debugAnmelde)
			System.out.println(buffer);

		ClientDavInterface verb = m_connection;

		AttributeGroup atg = verb.getDataModel().getAttributeGroup(atgPid);

		Aspect asp = verb.getDataModel().getAspect(aspPid);

		DataDescription dd = new DataDescription(atg, asp);

		SystemObject obj = m_connection.getDataModel().getObject(objPid);

		m_connection.subscribeReceiver(this, obj, dd, ReceiveOptions.normal(), role);
	}

	/**
	 * Methode die das Objekt mit der Pid objPid beim Datenverteiler abmeldet
	 * zum Lesen der Attibutgruppe atgPid unter dem Aspekt aspPid.
	 * 
	 * @param objPid
	 *            Pid des Objekts
	 * @param atgPid
	 *            Attributgruppe die abgemeldet werden soll
	 * @param aspPid
	 *            Apekt der abgemeldet werden soll
	 */
	private void abmeldeLeseDaV(String objPid, String atgPid, String aspPid) {
		String buffer = "Abmelden am DaV (Lesen): " + objPid + " " + atgPid + "  " + aspPid;

		if (_debugAnmelde)
			System.out.println(buffer);

		AttributeGroup atg = m_connection.getDataModel().getAttributeGroup(atgPid);
		Aspect asp = m_connection.getDataModel().getAspect(aspPid);

		DataDescription dd = new DataDescription(atg, asp);

		SystemObject obj = m_connection.getDataModel().getObject(objPid);

		m_connection.unsubscribeReceiver(this, obj, dd);
	}

	/**
	 * Methode die dieses Stauobjekt am DaV abmeldet
	 */
	public void dispose() {
		abmeldeLeseDaV(m_pid, m_atg, m_asp);
	}

	// ----------------------------------------------------------------------------------------------
	// Listener Funktionalitäten
	// ----------------------------------------------------------------------------------------------

	/**
	 * Methode zum Hinzufügen eines Listeners
	 * 
	 * @param listener
	 *            Listener der hinzugefügt werden soll
	 */
	public void addListener(INeueDatenListener listener) {
		m_angemeldeteListener.add(listener);
	}

	/**
	 * Methode zum Entfernen eines Listeners
	 * 
	 * @param listener
	 *            Listener der entfernt werden soll
	 */
	public void removeListener(INeueDatenListener listener) {
		m_angemeldeteListener.remove(listener);
	}

	/**
	 * Methode mit der die Klasse alle bei ihr angemeldeten Listener über die
	 * Änderung der Selektion informiert.
	 */
	private void benachrichtigeListener() {
		Iterator<INeueDatenListener> it = m_angemeldeteListener.iterator();
		while (it.hasNext()) {
			INeueDatenListener l = it.next();
			l.neueDaten();
		}
	}
}
