/*
 * Segment 7 (Ste), SWE TMC-Meldungsverwaltung
 * Copyright (C) 2016 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This programm is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */


package de.bsvrz.ste.tmcvew;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.MutableSetChangeListener;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Klasse zum Verwalten der dynamischen Objekte vom Typ "typ.tmcVerkehrsMeldung". Die
 * Klasse meldet sich auf Änderung der dynamischen Menge TMCMeldungen an
 * und verwaltet diese in einer Hashmap.
 * 
 * Erstellt auf Basis der SWE RDS/TMC-Meldungen von: Dambach Werke GmbH, Hilpp
 * 
 * @author BitCtrl Systems GmbH, Gieseler
 * @version $Id: $ 
 */
public class TmcMeldungsVerwaltung implements ClientReceiverInterface, MutableSetChangeListener {
	/**
	 * Übergebene Datenverteiler Verbindung.
	 */
	private ClientDavConnection connection;
	
	/**
	 * Übergebenes Datenmodell.
	 */
	private static DataModel dataModel;
	/**
	 * Liste der aktuellen Engstellen.
	 */
	private List<DynamicObject> listeMeldungen;
	/**
	 * Verkehrsmodellnetz.
	 */
	private ConfigurationObject verkehrsModellNetz;

	/**
	 * Hashmap mit den einzelnen Instanzen der von der Meldungsverwaltung gesendeten
	 * Objekte: key = Pid des Meldungsobjekts; value = Instanz der Meldung.
	 */
	private HashMap<String, RdsMeldungsAuswertung> meldungsAuswertung = new HashMap<String, RdsMeldungsAuswertung>();

	/**
	 * Konstruktor der Klasse. Anmeldung auf Änderung der dynamischen Menge
	 * TMCMeldungen.<br>
	 * Holen der aktuellen Objekte der dynamischen Menge<br>
	 * Anmeldung der Attributgruppenverwendungen<br>
	 * 
	 * @param connection
	 *            Datenverteilerverbindung
	 * @param netz
	 *            Verkehrsmodellnetz
	 */
	public TmcMeldungsVerwaltung(final ClientDavConnection connection, final String netz) {
		this.connection = connection;
		dataModel = connection.getDataModel();
		verkehrsModellNetz = (ConfigurationObject) dataModel.getObject(netz);

		listeMeldungen = new ArrayList<DynamicObject>();

		verkehrsModellNetz.getMutableSet(RdsMeldung.MELDUNGEN_MENGE).addChangeListener(this);

		rdsObjekteBestimmen();
	}

	/**
	 * Bestimmt die im Verkehrsmodellnetz aktuell vorhandenen TMC-Meldungen.
	 */
	protected void rdsObjekteBestimmen() {
		if (verkehrsModellNetz == null) {
			return;
		}

		final List<?> liste = verkehrsModellNetz.getMutableSet(RdsMeldung.MELDUNGEN_MENGE).getElements();

		for (int i = 0; i < liste.size(); i++) {
			final DynamicObject obj = (DynamicObject) liste.get(i);
			rdsListeVerwalten(obj);
		}
	}

	/**
	 * Methode liefert eine HashMap mit den aktiven Meldungsobjekten.
	 * 
	 * @return Objekt StauAuswertung
	 */

	public HashMap<String, RdsMeldungsAuswertung> getMeldungen() {
		if (meldungsAuswertung.size() > 0) {
			return meldungsAuswertung;
		} else {
			return null;
		}
	}

	/**
	 * Verwaltet die Liste der dynamischen Objekte. In einer Liste werden
	 * alle aktiven Meldungsobjekte gehalten. Neue Objekte empfangen werden
	 * in die Liste eingefügt. Werden Objekte ungültig, werden diese aus
	 * der Liste gelöscht.
	 * 
	 * @param obj
	 *            Dynamisches Objekt Meldung
	 */
	private void rdsListeVerwalten(final DynamicObject obj) {
		if (obj == null) {
			return;
		}

		if (obj.isValid()) {
			if (!listeMeldungen.contains(obj)) {
				System.out.println(this.getClass().getSimpleName() + ": " + obj.getPid() + " zu Liste hinzugefügt.");
				listeMeldungen.add(obj);
				meldungsAuswertung.put(obj.getPid(), new RdsMeldungsAuswertung(connection, obj.getPid()));
			}
		} else {
			listeMeldungen.remove(obj);
			System.out.println(this.getClass().getSimpleName() + ": " + obj.getPid() + " aus Liste gelöscht.");

			if (meldungsAuswertung.containsKey(obj.getPid())) {
				meldungsAuswertung.get(obj.getPid()).dispose();
				meldungsAuswertung.remove(obj.getPid());
			}
		}
	}

	/**
	 * Update Methode des ClientReceiverInterface zum Empfang der "normalen"
	 * Daten.
	 * 
	 * @param results
	 *            beinhaltet die empfangenen Daten
	 */
	public void update(final ResultData[] results) {
//		for (int i = 0; i < results.length; i++) {
//			final ResultData data = results[i];
//		}
	}

	/**
	 * Update Methode des ClientReceiverInterface zum Empfang für die
	 * dynamischen Objekte.
	 * 
	 * @param set
	 *            Menge der dynamischen Objekte
	 * @param addedObjects
	 *            Dynamische Objekte die neu hinzugekommen sind
	 * @param removedObjects
	 *            Dynamische Objekte die entfernt wurden
	 */

	public void update(final MutableSet set, final SystemObject[] addedObjects, final SystemObject[] removedObjects) {
		for (int i = 0; i < removedObjects.length; i++) {
			final DynamicObject obj = (DynamicObject) removedObjects[i];

			rdsListeVerwalten(obj);
		}

		for (int i = 0; i < addedObjects.length; i++) {
			final DynamicObject obj = (DynamicObject) addedObjects[i];

			rdsListeVerwalten(obj);
		}
	}
}
